# gitlab-cloud-connector (Ruby)

Ruby gem containing shared code for Cloud Connector token issuers (GitLab, CustomersDot).

## Usage

We expect Bundler is used to manage dependencies. To add the dependency, add it to `Gemfile`:

```Gemfile
gem "gitlab-cloud-connector", "~> 1.0.0", require: 'gitlab/cloud_connector'
```

### Data Storage

Cloud Connector uses YAML files stored in the project's `/config` directory. The data is organized by model type:

```yaml
# config/backend_services/ai_gateway.yml
name: ai_gateway
description: "AI Gateway"
jwt_aud: gitlab-ai-gateway

# config/add_ons/duo_pro.yml
name: duo_pro
description: "Duo Pro"

# config/unit_primitives/duo_chat.yml
name: duo_chat
description: "Duo Chat"
backend_services:
  - ai_gateway
add_ons:
  - duo_pro
  - duo_enterprise
```

### Working with DataModel

`DataModel` contains the core data structures in the Ruby library.
Each model inherits from `DataModel::Base`. The data is loaded from YAML files into memory at runtime, making it efficient
for read operations and relationship traversal.

```ruby
require 'gitlab/cloud_connector'

# Find unit primitive by name
# Note:  value should be a symbol (e.g., `:duo_chat` instead of `'duo_chat'`).
duo_chat = Gitlab::CloudConnector::DataModel::UnitPrimitive.find_by_name(:duo_chat)
duo_chat.description
duo_chat.cut_off_date

# access associations
duo_chat.add_ons
duo_chat.license_types
duo_chat.backend_services

# read duo_chat audiences
duo_chat.backend_services.map(&:jwt_aud)
```

## Contributing

### Test your changes locally

To test changes to the Ruby gem in your environment, you need to source gem contents from you local dir:

1. Set `path` in `Gemfile` to point to the library project source:

   ```ruby
   gem "gitlab-cloud-connector", "~> 1.5", require: 'gitlab/cloud_connector', feature_category: :plan_provisioning, path: '/path/to/gitlab-cloud-connector/src/ruby/'
   ```

1. Set the `CLOUD_CONNECTOR_CONFIG_DIR` environment variable to your local `gitlab-cloud-connector` config dir, for example `CLOUD_CONNECTOR_CONFIG_DIR=/Users/name/workspace/gitlab-cloud-connector/config`

1. Run `bundle install` to apply the changes

Note: you may need to restart GitLab instance for changes to take effect (depends on your hot-reload configuration)

## Release Process

See [Release Process](../../README.md#release-process)

Once the new gem version is visible on [RubyGems.org](https://rubygems.org/gems/gitlab-cloud-connector),
it is recommended to update [GitLab's `Gemfile`](https://gitlab.com/gitlab-org/gitlab/-/blob/master/Gemfile)
and [Customers Dot's `Gemfile`](https://gitlab.com/gitlab-org/customers-gitlab-com/-/blob/main/Gemfile) to bump the `gitlab-cloud-connector`
Ruby gem to the new version also. See [Usage](#usage) for how to do this.
