 /* Copyright 2009-2025 NVIDIA CORPORATION & AFFILIATES.  All rights reserved. 
  * 
  * NOTICE TO LICENSEE: 
  * 
  * The source code and/or documentation ("Licensed Deliverables") are 
  * subject to NVIDIA intellectual property rights under U.S. and 
  * international Copyright laws. 
  * 
  * The Licensed Deliverables contained herein are PROPRIETARY and 
  * CONFIDENTIAL to NVIDIA and are being provided under the terms and 
  * conditions of a form of NVIDIA software license agreement by and 
  * between NVIDIA and Licensee ("License Agreement") or electronically 
  * accepted by Licensee.  Notwithstanding any terms or conditions to 
  * the contrary in the License Agreement, reproduction or disclosure 
  * of the Licensed Deliverables to any third party without the express 
  * written consent of NVIDIA is prohibited. 
  * 
  * NOTWITHSTANDING ANY TERMS OR CONDITIONS TO THE CONTRARY IN THE 
  * LICENSE AGREEMENT, NVIDIA MAKES NO REPRESENTATION ABOUT THE 
  * SUITABILITY OF THESE LICENSED DELIVERABLES FOR ANY PURPOSE.  THEY ARE 
  * PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY OF ANY KIND. 
  * NVIDIA DISCLAIMS ALL WARRANTIES WITH REGARD TO THESE LICENSED 
  * DELIVERABLES, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY, 
  * NONINFRINGEMENT, AND FITNESS FOR A PARTICULAR PURPOSE. 
  * NOTWITHSTANDING ANY TERMS OR CONDITIONS TO THE CONTRARY IN THE 
  * LICENSE AGREEMENT, IN NO EVENT SHALL NVIDIA BE LIABLE FOR ANY 
  * SPECIAL, INDIRECT, INCIDENTAL, OR CONSEQUENTIAL DAMAGES, OR ANY 
  * DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, 
  * WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS 
  * ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE 
  * OF THESE LICENSED DELIVERABLES. 
  * 
  * U.S. Government End Users.  These Licensed Deliverables are a 
  * "commercial item" as that term is defined at 48 C.F.R_Ctx. 2.101 (OCT 
  * 1995), consisting of "commercial computer software" and "commercial 
  * computer software documentation" as such terms are used in 48 
  * C.F.R_Ctx. 12.212 (SEPT 1995) and are provided to the U.S. Government 
  * only as a commercial end item.  Consistent with 48 C.F.R_Ctx.12.212 and 
  * 48 C.F.R_Ctx. 227.7202-1 through 227.7202-4 (JUNE 1995), all 
  * U.S. Government End Users acquire the Licensed Deliverables with 
  * only those rights set forth herein. 
  * 
  * Any use of the Licensed Deliverables in individual and commercial 
  * software must include, in the user documentation and internal 
  * comments to the code, the above Disclaimer and U.S. Government End 
  * Users Notice. 
  */ 
#ifndef NV_NPPI_STATISTICS_FUNCTIONS_H
#define NV_NPPI_STATISTICS_FUNCTIONS_H
 
/**
 * \file nppi_statistics_functions.h
 * NPP Image Processing Functionality.
 */
 
#include "nppdefs.h"


#ifdef __cplusplus
extern "C" {
#endif

/** 
 *  \page image_statistics_functions Statistical Operations
 *  @defgroup image_statistics_functions Statistical Operations
 *  @ingroup nppi
 *
 * Primitives for computing the statistical properties of an image.
 * Some statistical primitives also require scratch buffer during the computation. For details, please 
 * refer to \ref general_scratch_buffer.
 * 
 * @{
 *
 * These functions can be found in the nppist library. Linking to only the sub-libraries that you use can significantly
 * save link time, application load time, and CUDA runtime startup time when using dynamic libraries.
 *
 * \section CommonGetBufferHostSizeParameters
 * Common parameters for nppiGetBufferHostSize functions include:
 * \param oSizeROI \ref roi_specification.
 * \param hpBufferSize Required buffer size. Important: hpBufferSize is a 
 *        <em>host pointer.</em> \ref general_scratch_buffer.
 * \param nppStreamCtx \ref application_managed_stream_context. 
 * \return NPP_NULL_POINTER_ERROR if hpBufferSize is 0 (NULL),  \ref roi_error_codes.  
 */

/** 
 * \section image_sum Sum
 * @defgroup image_sum Sum
 * Primitives for computing the sum of all the pixel values in an image.
 * @{
 */

/** @name Sum
 * @{
 * Given an image \f$pSrc\f$ with width \f$W\f$ and height \f$H\f$, the sum
 * will be computed as
 * \f[Sum = \sum_{j=0}^{H-1}\sum_{i=0}^{W-1}pSrc(j,i)\f]
 * All the results are stored in a 64-bit double precision format, 
 * except for two primitives \ref nppiSum_8u64s_C1R_Ctx and \ref nppiSum_8u64s_C4R_Ctx.
 * The sum functions require additional scratch buffer for computations. 
 *
 * \section CommonSumParameters Common parameters for nppiSum functions include:
 *
 * \param pSrc \ref source_image_pointer.
 * \param nSrcStep \ref source_image_line_step.
 * \param oSizeROI \ref roi_specification.
 * \param pDeviceBuffer Pointer to the required device memory allocation, \ref general_scratch_buffer.
 *                      Use nppiSumGetBufferHostSize_XX_XXX to determine the minium number of bytes required.
 * \param pSum Pointer to the computed sum.
 * \param nppStreamCtx \ref application_managed_stream_context. 
 * \return \ref image_data_error_codes, \ref roi_error_codes
 *
 * @{
 *
 */
 
/**
 * One-channel 8-bit unsigned image sum. 
 *
 * For common parameter descriptions, see \ref CommonSumParameters. 
 *
 */
NppStatus 
nppiSum_8u_C1R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp64f * pSum, NppStreamContext nppStreamCtx);
/**
 * One-channel 8-bit unsigned image sum. The result is 64-bit long long integer.
 *
 * For common parameter descriptions, see \ref CommonSumParameters. 
 *
 */
NppStatus 
nppiSum_8u64s_C1R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp64s * pSum, NppStreamContext nppStreamCtx);
/**
 * One-channel 16-bit unsigned image sum.
 *
 * For common parameter descriptions, see \ref CommonSumParameters. 
 *
 */
NppStatus 
nppiSum_16u_C1R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp64f * pSum, NppStreamContext nppStreamCtx);
/**
 * One-channel 16-bit signed image sum.
 *
 * For common parameter descriptions, see \ref CommonSumParameters. 
 *
 */
NppStatus 
nppiSum_16s_C1R_Ctx(const Npp16s * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp64f * pSum, NppStreamContext nppStreamCtx);
/**
 * One-channel 32-bit floating point image sum.
 *
 * For common parameter descriptions, see \ref CommonSumParameters. 
 *
 */
NppStatus 
nppiSum_32f_C1R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp64f * pSum, NppStreamContext nppStreamCtx);
/**
 * Three-channel 8-bit unsigned image sum.
 *
 * For common parameter descriptions, see \ref CommonSumParameters. 
 *
 */
NppStatus 
nppiSum_8u_C3R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp64f aSum[3], NppStreamContext nppStreamCtx);
/**
 * Three-channel 16-bit unsigned image sum.
 *
 * For common parameter descriptions, see \ref CommonSumParameters. 
 *
 */
NppStatus 
nppiSum_16u_C3R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp64f aSum[3], NppStreamContext nppStreamCtx);
/**
 * Three-channel 16-bit signed image sum.
 *
 * For common parameter descriptions, see \ref CommonSumParameters. 
 *
 */
NppStatus 
nppiSum_16s_C3R_Ctx(const Npp16s * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp64f aSum[3], NppStreamContext nppStreamCtx);
/**
 * Three-channel 32-bit floating point image sum.
 *
 * For common parameter descriptions, see \ref CommonSumParameters.  
 *
 */
NppStatus 
nppiSum_32f_C3R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp64f aSum[3], NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image sum ignoring alpha channel.
 *
 * For common parameter descriptions, see \ref CommonSumParameters. 
 *
 */
NppStatus 
nppiSum_8u_AC4R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp64f aSum[3], NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit unsigned image sum ignoring alpha channel.
 *
 * For common parameter descriptions, see \ref CommonSumParameters. 
 *
 */
NppStatus 
nppiSum_16u_AC4R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp64f aSum[3], NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit signed image sum ignoring alpha channel.
 *
 * For common parameter descriptions, see \ref CommonSumParameters. 
 *
 */
NppStatus 
nppiSum_16s_AC4R_Ctx(const Npp16s * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp64f aSum[3], NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit floating point image sum ignoring alpha channel.
 *
 * For common parameter descriptions, see \ref CommonSumParameters. 
 *
 */
NppStatus 
nppiSum_32f_AC4R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp64f aSum[3], NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image sum.
 *
 * For common parameter descriptions, see \ref CommonSumParameters. 
 *
 */
NppStatus 
nppiSum_8u_C4R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp64f aSum[4], NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image sum. The result is 64-bit long long integer.
 *
 * For common parameter descriptions, see \ref CommonSumParameters. 
 *
 */
NppStatus 
nppiSum_8u64s_C4R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp64s aSum[4], NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit unsigned image sum.
 *
 * For common parameter descriptions, see \ref CommonSumParameters. 
 *
 */
NppStatus 
nppiSum_16u_C4R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp64f aSum[4], NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit signed image sum.
 *
 * For common parameter descriptions, see \ref CommonSumParameters. 
 *
 */
NppStatus 
nppiSum_16s_C4R_Ctx(const Npp16s * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp64f aSum[4], NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit floating point image sum.
 *
 * For common parameter descriptions, see \ref CommonSumParameters. 
 *
 */
NppStatus 
nppiSum_32f_C4R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp64f aSum[4], NppStreamContext nppStreamCtx);

/** @} Sum */

/** @name SumGetBufferHostSize 
 * Companion primitives for computing the device buffer size (in bytes) required by the sum primitives.
 *  
 * \section CommonSumGetBufferHostSizeParameters
 * Common parameters for nppiSumGetBufferHostSize functions include:
 * \param oSizeROI \ref roi_specification.
 * \param hpBufferSize Required buffer size. Important: hpBufferSize is a 
 *        <em>host pointer.</em> \ref general_scratch_buffer.
 * \param nppStreamCtx \ref application_managed_stream_context. 
 * \return NPP_NULL_POINTER_ERROR if hpBufferSize is 0 (NULL),  \ref roi_error_codes.  
 * @{
 */

/**
 * Buffer size for \ref nppiSum_8u_C1R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonSumGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiSumGetBufferHostSize_8u_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/**
 * Buffer size for \ref nppiSum_8u64s_C1R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonSumGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiSumGetBufferHostSize_8u64s_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiSum_16u_C1R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonSumGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiSumGetBufferHostSize_16u_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);

/** 
 * Buffer size for \ref nppiSum_16s_C1R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonSumGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiSumGetBufferHostSize_16s_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiSum_32f_C1R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonSumGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiSumGetBufferHostSize_32f_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiSum_8u_C3R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonSumGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiSumGetBufferHostSize_8u_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiSum_16u_C3R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonSumGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiSumGetBufferHostSize_16u_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiSum_16s_C3R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonSumGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiSumGetBufferHostSize_16s_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiSum_32f_C3R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonSumGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiSumGetBufferHostSize_32f_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiSum_8u_AC4R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonSumGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiSumGetBufferHostSize_8u_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiSum_16u_AC4R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonSumGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiSumGetBufferHostSize_16u_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiSum_16s_AC4R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonSumGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiSumGetBufferHostSize_16s_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiSum_32f_AC4R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonSumGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiSumGetBufferHostSize_32f_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiSum_8u64s_C4R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonSumGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiSumGetBufferHostSize_8u64s_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiSum_8u_C4R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonSumGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiSumGetBufferHostSize_8u_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiSum_16u_C4R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonSumGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiSumGetBufferHostSize_16u_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiSum_16s_C4R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonSumGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiSumGetBufferHostSize_16s_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiSum_32f_C4R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonSumGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiSumGetBufferHostSize_32f_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
 
/** @} SumGetBufferHostSize */
/** @} image_sum */

/** 
 * \section image_min Min
 * @defgroup image_min Min
 * Primitives for computing the minimal pixel value of an image.
 * @{
 */

/** @name Min
 * The scratch buffer is required by the min functions.
 *
 * \section CommonMinParameters Common parameters for nppiMin functions include:
 *
 * \param pSrc \ref source_image_pointer.
 * \param nSrcStep \ref source_image_line_step.
 * \param oSizeROI \ref roi_specification.
 * \param pDeviceBuffer Pointer to the required device memory allocation, \ref general_scratch_buffer.
 *                      Use nppiMinGetBufferHostSize_XX_XXX to determine the minium number of bytes required.
 * \param pMin Pointer to the computed min.
 * \param nppStreamCtx \ref application_managed_stream_context. 
 * \return \ref image_data_error_codes, \ref roi_error_codes
 *
 * @{
 */

/**
 * One-channel 8-bit unsigned image min. 
 *
 * For common parameter descriptions, see \ref CommonMinParameters.
 *
 */
NppStatus 
nppiMin_8u_C1R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp8u * pMin, NppStreamContext nppStreamCtx);
/**
 * One-channel 16-bit unsigned image min. 
 *
 * For common parameter descriptions, see \ref CommonMinParameters.
 *
 */
NppStatus 
nppiMin_16u_C1R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp16u * pMin, NppStreamContext nppStreamCtx);
/**
 * One-channel 16-bit signed image min. 
 *
 * For common parameter descriptions, see \ref CommonMinParameters.
 *
 */
NppStatus 
nppiMin_16s_C1R_Ctx(const Npp16s * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp16s * pMin, NppStreamContext nppStreamCtx);
/**
 * One-channel 32-bit floating point image min. 
 *
 * For common parameter descriptions, see \ref CommonMinParameters.
 *
 */
NppStatus 
nppiMin_32f_C1R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp32f * pMin, NppStreamContext nppStreamCtx);
/**
 * Three-channel 8-bit unsigned image min. 
 *
 * For common parameter descriptions, see \ref CommonMinParameters.
 *
 */
NppStatus 
nppiMin_8u_C3R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp8u aMin[3], NppStreamContext nppStreamCtx);
/**
 * Three-channel 16-bit unsigned image min. 
 *
 * For common parameter descriptions, see \ref CommonMinParameters.
 *
 */
NppStatus 
nppiMin_16u_C3R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp16u aMin[3], NppStreamContext nppStreamCtx);
/**
 * Three-channel 16-bit signed image min. 
 *
 * For common parameter descriptions, see \ref CommonMinParameters.
 *
 */
NppStatus 
nppiMin_16s_C3R_Ctx(const Npp16s * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp16s aMin[3], NppStreamContext nppStreamCtx);
/**
 * Three-channel 32-bit floating point image min. 
 *
 * For common parameter descriptions, see \ref CommonMinParameters.
 *
 */
NppStatus 
nppiMin_32f_C3R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp32f aMin[3], NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image min. 
 *
 * For common parameter descriptions, see \ref CommonMinParameters.
 *
 */
NppStatus 
nppiMin_8u_C4R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp8u aMin[4], NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit unsigned image min. 
 *
 * For common parameter descriptions, see \ref CommonMinParameters.
 *
 */
NppStatus 
nppiMin_16u_C4R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp16u aMin[4], NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit signed image min. 
 *
 * For common parameter descriptions, see \ref CommonMinParameters.
 *
 */
NppStatus 
nppiMin_16s_C4R_Ctx(const Npp16s * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp16s aMin[4], NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit floating point image min. 
 *
 * For common parameter descriptions, see \ref CommonMinParameters.
 *
 */
NppStatus 
nppiMin_32f_C4R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp32f aMin[4], NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image min ignoring alpha channel.
 *
 * For common parameter descriptions, see \ref CommonMinParameters.
 *
 */
NppStatus 
nppiMin_8u_AC4R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp8u aMin[3], NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit unsigned image min ignoring alpha channel.
 *
 * For common parameter descriptions, see \ref CommonMinParameters.
 *
 */
NppStatus 
nppiMin_16u_AC4R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp16u aMin[3], NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit signed image min ignoring alpha channel.
 *
 * For common parameter descriptions, see \ref CommonMinParameters.
 *
 */
NppStatus 
nppiMin_16s_AC4R_Ctx(const Npp16s * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp16s aMin[3], NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit floating point image min ignoring alpha channel.
 *
 * For common parameter descriptions, see \ref CommonMinParameters.
 *
 */
NppStatus 
nppiMin_32f_AC4R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp32f aMin[3], NppStreamContext nppStreamCtx);

/** @} Min */

/** @name MinGetBufferHostSize
 * Companion primitives for computing the device buffer size (in bytes) required by the min primitives.
 * \section CommonMinGetBufferHostSizeParameters
 * Common parameters for nppiMinGetBufferHostSize functions include:
 * \param oSizeROI \ref roi_specification.
 * \param hpBufferSize Required buffer size. Important: hpBufferSize is a 
 *        <em>host pointer.</em> \ref general_scratch_buffer.
 * \param nppStreamCtx \ref application_managed_stream_context. 
 * \return NPP_NULL_POINTER_ERROR if hpBufferSize is 0 (NULL),  \ref roi_error_codes.  
 * @{
 */

/**
 * Buffer size for \ref nppiMin_8u_C1R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMinGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMinGetBufferHostSize_8u_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMin_16u_C1R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMinGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMinGetBufferHostSize_16u_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMin_16s_C1R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMinGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMinGetBufferHostSize_16s_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMin_32f_C1R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMinGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMinGetBufferHostSize_32f_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMin_8u_C3R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMinGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMinGetBufferHostSize_8u_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMin_16u_C3R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMinGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMinGetBufferHostSize_16u_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMin_16s_C3R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMinGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMinGetBufferHostSize_16s_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMin_32f_C3R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMinGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMinGetBufferHostSize_32f_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMin_8u_C4R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMinGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMinGetBufferHostSize_8u_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);

/** 
 * Buffer size for \ref nppiMin_16u_C4R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMinGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMinGetBufferHostSize_16u_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMin_16s_C4R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMinGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMinGetBufferHostSize_16s_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMin_32f_C4R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMinGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMinGetBufferHostSize_32f_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMin_8u_AC4R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMinGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMinGetBufferHostSize_8u_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMin_16u_AC4R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMinGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMinGetBufferHostSize_16u_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMin_16s_AC4R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMinGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMinGetBufferHostSize_16s_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMin_32f_AC4R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMinGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMinGetBufferHostSize_32f_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);

/** @} MinGetBufferHostSize */
/** @} image_min */

/** 
 * \section image_min_index MinIndx
 * @defgroup image_min_index MinIndx
 * Primitives for computing the minimal value and its indices (X and Y coordinates) of an image.
 * @{
 */
/** @name MinIndx
 * If there are several minima in the selected ROI, the function returns one on the top leftmost position.
 * The scratch buffer is required by the functions.
 * 
 * \section CommonMinIndxParameters Common parameters for nppiMinIndx functions include:
 *
 * \param pSrc \ref source_image_pointer.
 * \param nSrcStep \ref source_image_line_step.
 * \param oSizeROI \ref roi_specification.
 * \param pDeviceBuffer Pointer to the required device memory allocation, \ref general_scratch_buffer
 *        Use nppiMinIndxGetBufferHostSize_XX_XXX to determine the minium number of bytes required.
 * \param pMin Pointer to the computed min result.
 * \param pIndexX Pointer to the X coordinate of the image min value.
 * \param pIndexY Ppointer to the Y coordinate of the image min value.
 * \param nppStreamCtx \ref application_managed_stream_context. 
 * \return \ref image_data_error_codes, \ref roi_error_codes
 *
 * @{
 */
/**
 * One-channel 8-bit unsigned image MinIndx.
 *
 * For common parameter descriptions, see \ref CommonMinIndxParameters>.
 *
 */
NppStatus 
nppiMinIndx_8u_C1R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp8u * pMin, int * pIndexX, int * pIndexY, NppStreamContext nppStreamCtx);
/**
 * One-channel 16-bit unsigned image MinIndx.
 *
 * For common parameter descriptions, see \ref CommonMinIndxParameters.
 *
 */
NppStatus 
nppiMinIndx_16u_C1R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp16u * pMin, int * pIndexX, int * pIndexY, NppStreamContext nppStreamCtx);
/**
 * One-channel 16-bit signed image MinIndx.
 *
 * For common parameter descriptions, see \ref CommonMinIndxParameters.
 *
 */
NppStatus 
nppiMinIndx_16s_C1R_Ctx(const Npp16s * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp16s * pMin, int * pIndexX, int * pIndexY, NppStreamContext nppStreamCtx);
/**
 * One-channel 32-bit floating point image MinIndx.
 *
 * For common parameter descriptions, see \ref CommonMinIndxParameters.
 *
 */
NppStatus 
nppiMinIndx_32f_C1R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp32f * pMin, int * pIndexX, int * pIndexY, NppStreamContext nppStreamCtx);
/**
 * Three-channel 8-bit unsigned image MinIndx.
 *
 * For common parameter descriptions, see \ref CommonMinIndxParameters.
 *
 */
NppStatus 
nppiMinIndx_8u_C3R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp8u aMin[3], int aIndexX[3], int aIndexY[3], NppStreamContext nppStreamCtx);
/**
 * Three-channel 16-bit unsigned image MinIndx.
 *
 * For common parameter descriptions, see \ref CommonMinIndxParameters.
 *
 */
NppStatus 
nppiMinIndx_16u_C3R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp16u aMin[3], int aIndexX[3], int aIndexY[3], NppStreamContext nppStreamCtx);
/**
 * Three-channel 16-bit signed image MinIndx.
 *
 * For common parameter descriptions, see \ref CommonMinIndxParameters.
 *
 */
NppStatus 
nppiMinIndx_16s_C3R_Ctx(const Npp16s * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp16s aMin[3], int aIndexX[3], int aIndexY[3], NppStreamContext nppStreamCtx);
/**
 * Three-channel 32-bit floating point image MinIndx.
 *
 * For common parameter descriptions, see \ref CommonMinIndxParameters.
 *
 */
NppStatus 
nppiMinIndx_32f_C3R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp32f aMin[3], int aIndexX[3], int aIndexY[3], NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image MinIndx.
 *
 * For common parameter descriptions, see \ref CommonMinIndxParameters.
 *
 */
NppStatus 
nppiMinIndx_8u_C4R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp8u aMin[4], int aIndexX[4], int aIndexY[4], NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit unsigned image MinIndx.
 *
 * For common parameter descriptions, see \ref CommonMinIndxParameters.
 *
 */
NppStatus 
nppiMinIndx_16u_C4R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp16u aMin[4], int aIndexX[4], int aIndexY[4], NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit signed image MinIndx.
 *
 * For common parameter descriptions, see \ref CommonMinIndxParameters.
 *
 */
NppStatus 
nppiMinIndx_16s_C4R_Ctx(const Npp16s * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp16s aMin[4], int aIndexX[4], int aIndexY[4], NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit floating point image MinIndx.
 *
 * For common parameter descriptions, see \ref CommonMinIndxParameters.
 *
 */
NppStatus 
nppiMinIndx_32f_C4R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp32f aMin[4], int aIndexX[4], int aIndexY[4], NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image MinIndx ignoring alpha channel.
 *
 * For common parameter descriptions, see \ref CommonMinIndxParameters.
 *
 */
NppStatus 
nppiMinIndx_8u_AC4R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp8u aMin[3], int aIndexX[3], int aIndexY[3], NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit unsigned image MinIndx ignoring alpha channel.
 *
 * For common parameter descriptions, see \ref CommonMinIndxParameters.
 *
 */
NppStatus 
nppiMinIndx_16u_AC4R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp16u aMin[3], int aIndexX[3], int aIndexY[3], NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit signed image MinIndx ignoring alpha channel.
 *
 * For common parameter descriptions, see \ref CommonMinIndxParameters.
 *
 */
NppStatus 
nppiMinIndx_16s_AC4R_Ctx(const Npp16s * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp16s aMin[3], int aIndexX[3], int aIndexY[3], NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit floating point image MinIndx ignoring alpha channel.
 *
 * For common parameter descriptions, see \ref CommonMinIndxParameters.
 *
 */
NppStatus 
nppiMinIndx_32f_AC4R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp32f aMin[3], int aIndexX[3], int aIndexY[3], NppStreamContext nppStreamCtx);

/** @} Min */

/** @name MinIndxGetBufferHostSize
 * Companion primitives for computing the device buffer size (in bytes) required by the MinIndx primitives.
 * \section CommonMinIndxGetBufferHostSizeParameters
 * Common parameters for nppiMinIndxGetBufferHostSize functions include:
 * \param oSizeROI \ref roi_specification.
 * \param hpBufferSize Required buffer size. Important: hpBufferSize is a 
 *        <em>host pointer.</em> \ref general_scratch_buffer.
 * \param nppStreamCtx \ref application_managed_stream_context. 
 * \return NPP_NULL_POINTER_ERROR if hpBufferSize is 0 (NULL),  \ref roi_error_codes.  
 * @{
 */
/**
 * Computes the dvice scratch buffer size (in bytes) for nppiMinIndx_8u_C1R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMinIndxGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMinIndxGetBufferHostSize_8u_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the dvice scratch buffer size (in bytes) for nppiMinIndx_16u_C1R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMinIndxGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMinIndxGetBufferHostSize_16u_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the dvice scratch buffer size (in bytes) for nppiMinIndx_16s_C1R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMinIndxGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMinIndxGetBufferHostSize_16s_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the dvice scratch buffer size (in bytes) for nppiMinIndx_32f_C1R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMinIndxGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMinIndxGetBufferHostSize_32f_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the dvice scratch buffer size (in bytes) for nppiMinIndx_8u_C3R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMinIndxGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMinIndxGetBufferHostSize_8u_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the dvice scratch buffer size (in bytes) for nppiMinIndx_16u_C3R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMinIndxGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMinIndxGetBufferHostSize_16u_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the dvice scratch buffer size (in bytes) for nppiMinIndx_16s_C3R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMinIndxGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMinIndxGetBufferHostSize_16s_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the dvice scratch buffer size (in bytes) for nppiMinIndx_32f_C3R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMinIndxGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMinIndxGetBufferHostSize_32f_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the dvice scratch buffer size (in bytes) for nppiMinIndx_8u_C4R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMinIndxGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMinIndxGetBufferHostSize_8u_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the dvice scratch buffer size (in bytes) for nppiMinIndx_16u_C4R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMinIndxGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMinIndxGetBufferHostSize_16u_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the dvice scratch buffer size (in bytes) for nppiMinIndx_16s_C4R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMinIndxGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMinIndxGetBufferHostSize_16s_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the dvice scratch buffer size (in bytes) for nppiMinIndx_32f_C4R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMinIndxGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMinIndxGetBufferHostSize_32f_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the dvice scratch buffer size (in bytes) for nppiMinIndx_8u_AC4R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMinIndxGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMinIndxGetBufferHostSize_8u_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the dvice scratch buffer size (in bytes) for nppiMinIndx_8u_AC4R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMinIndxGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMinIndxGetBufferHostSize_16u_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the dvice scratch buffer size (in bytes) for nppiMinIndx_16u_AC4R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMinIndxGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMinIndxGetBufferHostSize_16s_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the dvice scratch buffer size (in bytes) for nppiMinIndx_32f_AC4R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMinIndxGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMinIndxGetBufferHostSize_32f_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);

/** @} MinIndxGetBufferHostSize */
/** @} image_min_index */

/** 
 * \section image_max Max
 * @defgroup image_max Max
 * Primitives for computing the maximal pixel value of an image.
 * 
 * \subsection CommonMaxParameters Common parameters for nppiMax functions include:
 *
 * \param pSrc \ref source_image_pointer.
 * \param nSrcStep \ref source_image_line_step.
 * \param oSizeROI \ref roi_specification.
 * \param pDeviceBuffer Pointer to the required device memory allocation, \ref general_scratch_buffer.
 *                      Use nppiMaxGetBufferHostSize_XX_XXX to determine the minium number of bytes required.
 * \param pMax Pointer to the computed max.
 * \param nppStreamCtx \ref application_managed_stream_context. 
 * \return \ref image_data_error_codes, \ref roi_error_codes
 * 
 * @{
 */

/** @name Max
 *  The scratch buffer is required by the functions.
 * @{
 */

/**
 * One-channel 8-bit unsigned image Max.
 *
 * For common parameter descriptions, see \ref CommonMaxParameters.
 *
 */
NppStatus 
nppiMax_8u_C1R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp8u * pMax, NppStreamContext nppStreamCtx);
/**
 * One-channel 16-bit unsigned image Max.
 *
 * For common parameter descriptions, see \ref CommonMaxParameters.
 *
 */
NppStatus 
nppiMax_16u_C1R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp16u * pMax, NppStreamContext nppStreamCtx);
/**
 * One-channel 16-bit signed image Max.
 *
 * For common parameter descriptions, see \ref CommonMaxParameters.
 *
 */
NppStatus 
nppiMax_16s_C1R_Ctx(const Npp16s * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp16s * pMax, NppStreamContext nppStreamCtx);
/**
 * One-channel 32-bit floating point image Max.
 *
 * For common parameter descriptions, see \ref CommonMaxParameters.
 *
 */
NppStatus 
nppiMax_32f_C1R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp32f * pMax, NppStreamContext nppStreamCtx);
/**
 * Three-channel 8-bit unsigned image Max.
 *
 * For common parameter descriptions, see \ref CommonMaxParameters.
 *
 */
NppStatus 
nppiMax_8u_C3R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp8u aMax[3], NppStreamContext nppStreamCtx);
/**
 * Three-channel 16-bit unsigned image Max.
 *
 * For common parameter descriptions, see \ref CommonMaxParameters.
 *
 */
NppStatus 
nppiMax_16u_C3R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp16u aMax[3], NppStreamContext nppStreamCtx);
/**
 * Three-channel 16-bit signed image Max.
 *
 * For common parameter descriptions, see \ref CommonMaxParameters.
 *
 */
NppStatus 
nppiMax_16s_C3R_Ctx(const Npp16s * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp16s aMax[3], NppStreamContext nppStreamCtx);
/**
 * Three-channel 32-bit floating point image Max. 
 *
 * For common parameter descriptions, see \ref CommonMaxParameters.
 *
 */
NppStatus 
nppiMax_32f_C3R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp32f aMax[3], NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image Max. 
 *
 * For common parameter descriptions, see \ref CommonMaxParameters.
 *
 */
NppStatus 
nppiMax_8u_C4R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp8u aMax[4], NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit unsigned image Max. 
 *
 * For common parameter descriptions, see \ref CommonMaxParameters.
 *
 */
NppStatus 
nppiMax_16u_C4R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp16u aMax[4], NppStreamContext nppStreamCtx);
/**
 *
 * For common parameter descriptions, see \ref CommonMaxParameters.
 *
 */
NppStatus 
nppiMax_16s_C4R_Ctx(const Npp16s * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp16s aMax[4], NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit floating point image Max. 
 *
 * For common parameter descriptions, see \ref CommonMaxParameters.
 *
 */
NppStatus 
nppiMax_32f_C4R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp32f aMax[4], NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image Max ignoring alpha channel.
 *
 * For common parameter descriptions, see \ref CommonMaxParameters.
 *
 */
NppStatus 
nppiMax_8u_AC4R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp8u aMax[3], NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit unsigned image Max ignoring alpha channel.
 *
 * For common parameter descriptions, see \ref CommonMaxParameters.
 *
 */
NppStatus 
nppiMax_16u_AC4R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp16u aMax[3], NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit signed image Max ignoring alpha channel.
 *
 * For common parameter descriptions, see \ref CommonMaxParameters.
 *
 */
NppStatus 
nppiMax_16s_AC4R_Ctx(const Npp16s * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp16s aMax[3], NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit floating point image Max ignoring alpha channel.
 *
 * For common parameter descriptions, see \ref CommonMaxParameters.
 *
 */
NppStatus 
nppiMax_32f_AC4R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp32f aMax[3], NppStreamContext nppStreamCtx);

/** @} */

/** @name MaxGetBufferHostSize
 * Companion primitives for computing the device buffer size (in bytes) required by the Max primitives.
 * \section CommonMaxGetBufferHostSizeParameters
 * Common parameters for nppiMaxGetBufferHostSize functions include:
 * \param oSizeROI \ref roi_specification.
 * \param hpBufferSize Required buffer size. Important: hpBufferSize is a 
 *        <em>host pointer.</em> \ref general_scratch_buffer.
 * \param nppStreamCtx \ref application_managed_stream_context. 
 * \return NPP_NULL_POINTER_ERROR if hpBufferSize is 0 (NULL),  \ref roi_error_codes.  
 * @{
 */
/**
 * Buffer size for \ref nppiMax_8u_C1R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMaxGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMaxGetBufferHostSize_8u_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMax_16u_C1R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMaxGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMaxGetBufferHostSize_16u_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMax_16s_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonMaxGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaxGetBufferHostSize_16s_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMax_32f_C1R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMaxGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMaxGetBufferHostSize_32f_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMax_8u_C3R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMaxGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMaxGetBufferHostSize_8u_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMax_16u_C3R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMaxGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMaxGetBufferHostSize_16u_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMax_16s_C3R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMaxGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMaxGetBufferHostSize_16s_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMax_32f_C3R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMaxGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMaxGetBufferHostSize_32f_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMax_8u_C4R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMaxGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMaxGetBufferHostSize_8u_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMax_16u_C4R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMaxGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMaxGetBufferHostSize_16u_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMax_16s_C4R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMaxGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMaxGetBufferHostSize_16s_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMax_32f_C4R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMaxGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMaxGetBufferHostSize_32f_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMax_8u_AC4R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMaxGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMaxGetBufferHostSize_8u_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMax_16u_AC4R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMaxGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMaxGetBufferHostSize_16u_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMax_16s_AC4R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMaxGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMaxGetBufferHostSize_16s_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMax_32f_AC4R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMaxGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMaxGetBufferHostSize_32f_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);

/** @} MaxGetBufferHostSize */
/** @} image_max */

/** 
 * \section image_max_index MaxIndx
 * @defgroup image_max_index MaxIndx
 * Primitives for computing the maximal value and its indices (X and Y coordinates) of an image.
 * @{
 */
/** @name MaxIndx
 * If there are several maxima in the selected region of interest, the function returns one on the top leftmost position. 
 * The scratch buffer is required by the functions.
 * 
 * \section CommonMaxIndxParameters Common parameters for nppiMaxIndx functions include:
 *
 * \param pSrc \ref source_image_pointer.
 * \param nSrcStep \ref source_image_line_step.
 * \param oSizeROI \ref roi_specification.
 * \param pDeviceBuffer Pointer to the required device memory allocation, \ref general_scratch_buffer
 *        Use nppiMaxIndxGetBufferHostSize_XX_XXX to determine the minium number of bytes required.
 * \param pMax Pointer to the computed max result.
 * \param pIndexX Pointer to the X coordinate of the image max value.
 * \param pIndexY Ppointer to the Y coordinate of the image max value.
 * \param nppStreamCtx \ref application_managed_stream_context. 
 * \return \ref image_data_error_codes, \ref roi_error_codes
 *
 * @{
 *
 */

/**
 * One-channel 8-bit unsigned image MaxIndx.
 *
 * For common parameter descriptions, see \ref CommonMaxIndxParameters.
 *
 */
NppStatus 
nppiMaxIndx_8u_C1R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp8u * pMax, int * pIndexX, int * pIndexY, NppStreamContext nppStreamCtx);
/**
 * One-channel 16-bit unsigned image MaxIndx.
 *
 * For common parameter descriptions, see \ref CommonMaxIndxParameters.
 *
 */
NppStatus 
nppiMaxIndx_16u_C1R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp16u * pMax, int * pIndexX, int * pIndexY, NppStreamContext nppStreamCtx);
/**
 * One-channel 16-bit signed image MaxIndx.
 *
 * For common parameter descriptions, see \ref CommonMaxIndxParameters.
 *
 */
NppStatus 
nppiMaxIndx_16s_C1R_Ctx(const Npp16s * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp16s * pMax, int * pIndexX, int * pIndexY, NppStreamContext nppStreamCtx);
/**
 * One-channel 32-bit floating point image MaxIndx.
 *
 * For common parameter descriptions, see \ref CommonMaxIndxParameters.
 *
 */
NppStatus 
nppiMaxIndx_32f_C1R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp32f * pMax, int * pIndexX, int * pIndexY, NppStreamContext nppStreamCtx);
/**
 * Three-channel 8-bit unsigned image MaxIndx.
 *
 * For common parameter descriptions, see \ref CommonMaxIndxParameters.
 *
 */
NppStatus 
nppiMaxIndx_8u_C3R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp8u aMax[3], int aIndexX[3], int aIndexY[3], NppStreamContext nppStreamCtx);
/**
 * Three-channel 16-bit unsigned image MaxIndx.
 *
 * For common parameter descriptions, see \ref CommonMaxIndxParameters.
 *
 */
NppStatus 
nppiMaxIndx_16u_C3R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp16u aMax[3], int aIndexX[3], int aIndexY[3], NppStreamContext nppStreamCtx);
/**
 * Three-channel 16-bit signed image MaxIndx.
 *
 * For common parameter descriptions, see \ref CommonMaxIndxParameters.
 *
 */
NppStatus 
nppiMaxIndx_16s_C3R_Ctx(const Npp16s * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp16s aMax[3], int aIndexX[3], int aIndexY[3], NppStreamContext nppStreamCtx);
/**
 * Three-channel 32-bit floating point image MaxIndx.
 *
 * For common parameter descriptions, see \ref CommonMaxIndxParameters.
 *
 */
NppStatus 
nppiMaxIndx_32f_C3R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp32f aMax[3], int aIndexX[3], int aIndexY[3], NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image MaxIndx.
 *
 * For common parameter descriptions, see \ref CommonMaxIndxParameters.
 *
 */
NppStatus 
nppiMaxIndx_8u_C4R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp8u aMax[4], int aIndexX[4], int aIndexY[4], NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit unsigned image MaxIndx.
 *
 * For common parameter descriptions, see \ref CommonMaxIndxParameters.
 *
 */
NppStatus 
nppiMaxIndx_16u_C4R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp16u aMax[4], int aIndexX[4], int aIndexY[4], NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit signed image MaxIndx.
 *
 * For common parameter descriptions, see \ref CommonMaxIndxParameters.
 *
 */
NppStatus 
nppiMaxIndx_16s_C4R_Ctx(const Npp16s * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp16s aMax[4], int aIndexX[4], int aIndexY[4], NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit floating point image MaxIndx.
 *
 * For common parameter descriptions, see \ref CommonMaxIndxParameters.
 *
 */
NppStatus 
nppiMaxIndx_32f_C4R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp32f aMax[4], int aIndexX[4], int aIndexY[4], NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image MaxIndx ignoring alpha channel.
 *
 * For common parameter descriptions, see \ref CommonMaxIndxParameters.
 *
 */
NppStatus 
nppiMaxIndx_8u_AC4R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp8u aMax[3], int aIndexX[3], int aIndexY[3], NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit unsigned image MaxIndx ignoring alpha channel.
 *
 * For common parameter descriptions, see \ref CommonMaxIndxParameters.
 *
 */
NppStatus 
nppiMaxIndx_16u_AC4R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp16u aMax[3], int aIndexX[3], int aIndexY[3], NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit signed image MaxIndx ignoring alpha channel.
 *
 * For common parameter descriptions, see \ref CommonMaxIndxParameters.
 *
 */
NppStatus 
nppiMaxIndx_16s_AC4R_Ctx(const Npp16s * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp16s aMax[3], int aIndexX[3], int aIndexY[3], NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit floating point image MaxIndx ignoring alpha channel.
 *
 * For common parameter descriptions, see \ref CommonMaxIndxParameters.
 *
 */
NppStatus 
nppiMaxIndx_32f_AC4R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp32f aMax[3], int aIndexX[3], int aIndexY[3], NppStreamContext nppStreamCtx);

/** @} */

/** @name MaxIndxGetBufferHostSize
 * Companion primitives for computing the device buffer size (in bytes) required by the MaxIndx primitives.
 * \section CommonMaxIndxGetBufferHostSizeParameters
 * Common parameters for nppiMaxIndxGetBufferHostSize functions include:
 * \param oSizeROI \ref roi_specification.
 * \param hpBufferSize Required buffer size. Important: hpBufferSize is a 
 *        <em>host pointer.</em> \ref general_scratch_buffer.
 * \param nppStreamCtx \ref application_managed_stream_context. 
 * \return NPP_NULL_POINTER_ERROR if hpBufferSize is 0 (NULL),  \ref roi_error_codes.  
 * @{
 */

/**
 * Computes the dvice scratch buffer size (in bytes) for nppiMaxIndx_8u_C1R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMaxIndxGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMaxIndxGetBufferHostSize_8u_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the dvice scratch buffer size (in bytes) for nppiMaxIndx_16u_C1R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMaxIndxGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMaxIndxGetBufferHostSize_16u_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the dvice scratch buffer size (in bytes) for nppiMaxIndx_16s_C1R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMaxIndxGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMaxIndxGetBufferHostSize_16s_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the dvice scratch buffer size (in bytes) for nppiMaxIndx_32f_C1R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMaxIndxGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMaxIndxGetBufferHostSize_32f_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the dvice scratch buffer size (in bytes) for nppiMaxIndx_8u_C3R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMaxIndxGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMaxIndxGetBufferHostSize_8u_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the dvice scratch buffer size (in bytes) for nppiMaxIndx_16u_C3R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMaxIndxGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMaxIndxGetBufferHostSize_16u_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the dvice scratch buffer size (in bytes) for nppiMaxIndx_16s_C3R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMaxIndxGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMaxIndxGetBufferHostSize_16s_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the dvice scratch buffer size (in bytes) for nppiMaxIndx_32f_C3R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMaxIndxGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMaxIndxGetBufferHostSize_32f_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the dvice scratch buffer size (in bytes) for nppiMaxIndx_8u_C4R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMaxIndxGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMaxIndxGetBufferHostSize_8u_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the dvice scratch buffer size (in bytes) for nppiMaxIndx_16u_C4R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMaxIndxGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMaxIndxGetBufferHostSize_16u_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the dvice scratch buffer size (in bytes) for nppiMaxIndx_16s_C4R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMaxIndxGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMaxIndxGetBufferHostSize_16s_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the dvice scratch buffer size (in bytes) for nppiMaxIndx_32f_C4R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMaxIndxGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMaxIndxGetBufferHostSize_32f_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the dvice scratch buffer size (in bytes) for nppiMaxIndx_8u_AC4R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMaxIndxGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMaxIndxGetBufferHostSize_8u_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the dvice scratch buffer size (in bytes) for nppiMaxIndx_8u_AC4R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMaxIndxGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMaxIndxGetBufferHostSize_16u_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the dvice scratch buffer size (in bytes) for nppiMaxIndx_16u_AC4R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMaxIndxGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMaxIndxGetBufferHostSize_16s_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the dvice scratch buffer size (in bytes) for nppiMaxIndx_32f_AC4R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMaxIndxGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMaxIndxGetBufferHostSize_32f_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);

/** @} */
/** @} image_max_index */

/** 
 * \section image_min_max MinMax
 * @defgroup image_min_max MinMax
 * Primitives for computing both the minimal and the maximal values of an image.
 * @{
 */

/** @name MinMax
 * The functions require the device scratch buffer.
 * 
 * \section CommonMinMaxParameters Common parameters for nppiMinMax functions include:
 * 
 * \param pSrc \ref source_image_pointer.
 * \param nSrcStep \ref source_image_line_step.
 * \param oSizeROI \ref roi_specification.
 * \param pMin Pointer to the computed minimal result.
 * \param pMax Pointer to the computed maximal result.
 * \param pDeviceBuffer Buffer to a scratch memory. 
 *        Use nppiMinMax_XX_XXX to determine the minium number of bytes required.
 * \param nppStreamCtx \ref application_managed_stream_context. 
 * \return \ref image_data_error_codes, \ref roi_error_codes
 * 
 * @{
 */
/**
 * One-channel 8-bit unsigned image MinMax.
 * 
 */
NppStatus 
nppiMinMax_8u_C1R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pMin, Npp8u * pMax, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 16-bit unsigned image MinMax.
 *
 * For common parameter descriptions, see \ref CommonMinMaxParameters.
 *
 */
NppStatus 
nppiMinMax_16u_C1R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp16u * pMin, Npp16u * pMax, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 16-bit signed image MinMax.
 *
 * For common parameter descriptions, see \ref CommonMinMaxParameters.
 *
 */
NppStatus 
nppiMinMax_16s_C1R_Ctx(const Npp16s * pSrc, int nSrcStep, NppiSize oSizeROI, Npp16s * pMin, Npp16s * pMax, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 32-bit floating point image MinMax.
 *
 * For common parameter descriptions, see \ref CommonMinMaxParameters.
 *
 */
NppStatus 
nppiMinMax_32f_C1R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSizeROI, Npp32f * pMin, Npp32f * pMax, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 8-bit unsigned image MinMax.
 *
 * For common parameter descriptions, see \ref CommonMinMaxParameters.
 *
 */
NppStatus 
nppiMinMax_8u_C3R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u aMin[3], Npp8u aMax[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 16-bit unsigned image MinMax.
 *
 * For common parameter descriptions, see \ref CommonMinMaxParameters.
 *
 */
NppStatus 
nppiMinMax_16u_C3R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp16u aMin[3], Npp16u aMax[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 16-bit signed image MinMax.
 *
 * For common parameter descriptions, see \ref CommonMinMaxParameters.
 *
 */
NppStatus 
nppiMinMax_16s_C3R_Ctx(const Npp16s * pSrc, int nSrcStep, NppiSize oSizeROI, Npp16s aMin[3], Npp16s aMax[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 32-bit floating point image MinMax.
 *
 * For common parameter descriptions, see \ref CommonMinMaxParameters.
 *
 */
NppStatus 
nppiMinMax_32f_C3R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSizeROI, Npp32f aMin[3], Npp32f aMax[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image MinMax ignoring alpha channel.
 *
 * For common parameter descriptions, see \ref CommonMinMaxParameters.
 *
 */
NppStatus 
nppiMinMax_8u_AC4R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u aMin[3], Npp8u aMax[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit unsigned image MinMax ignoring alpha channel.
 *
 * For common parameter descriptions, see \ref CommonMinMaxParameters.
 *
 */
NppStatus 
nppiMinMax_16u_AC4R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp16u aMin[3], Npp16u aMax[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 *
 * For common parameter descriptions, see \ref CommonMinMaxParameters.
 *
 */
NppStatus 
nppiMinMax_16s_AC4R_Ctx(const Npp16s * pSrc, int nSrcStep, NppiSize oSizeROI, Npp16s aMin[3], Npp16s aMax[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit floating point image MinMax ignoring alpha channel.
 *
 * For common parameter descriptions, see \ref CommonMinMaxParameters.
 *
 */
NppStatus 
nppiMinMax_32f_AC4R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSizeROI, Npp32f aMin[3], Npp32f aMax[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image MinMax.
 *
 * For common parameter descriptions, see \ref CommonMinMaxParameters.
 *
 */
NppStatus 
nppiMinMax_8u_C4R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u aMin[4], Npp8u aMax[4], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit unsigned image MinMax.
 *
 * For common parameter descriptions, see \ref CommonMinMaxParameters.
 *
 */
NppStatus 
nppiMinMax_16u_C4R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp16u aMin[4], Npp16u aMax[4], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit signed image MinMax.
 *
 * For common parameter descriptions, see \ref CommonMinMaxParameters.
 *
 */
NppStatus 
nppiMinMax_16s_C4R_Ctx(const Npp16s * pSrc, int nSrcStep, NppiSize oSizeROI, Npp16s aMin[4], Npp16s aMax[4], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit floating point image MinMax.
 *
 * For common parameter descriptions, see \ref CommonMinMaxParameters.
 *
 */
NppStatus 
nppiMinMax_32f_C4R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSizeROI, Npp32f aMin[4], Npp32f aMax[4], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);

/** @} */

/** @name MinMaxGetBufferHostSize
 * Companion primitives for computing the device buffer size (in bytes) required by the MinMax primitives.
 * \section CommonMinMaxGetBufferHostSizeParameters
 * Common parameters for nppiMinMaxGetBufferHostSize functions include:
 * \param oSizeROI \ref roi_specification.
 * \param hpBufferSize Required buffer size. Important: hpBufferSize is a 
 *        <em>host pointer.</em> \ref general_scratch_buffer.
 * \param nppStreamCtx \ref application_managed_stream_context. 
 * \return NPP_NULL_POINTER_ERROR if hpBufferSize is 0 (NULL),  \ref roi_error_codes.  
 * @{
 */
/** 
 * Buffer size for \ref nppiMinMax_8u_C1R_Ctx.
 * 
 * For common parameter descriptions, see \ref CommonMinMaxGetBufferHostSizeParameters.
 *
 */
NppStatus
nppiMinMaxGetBufferHostSize_8u_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMinMax_16u_C1R_Ctx.
 * 
 * For common parameter descriptions, see \ref CommonMinMaxGetBufferHostSizeParameters.
 *
 */
NppStatus
nppiMinMaxGetBufferHostSize_16u_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMinMax_16s_C1R_Ctx.
 * 
 * For common parameter descriptions, see \ref CommonMinMaxGetBufferHostSizeParameters.
 *
 */
NppStatus
nppiMinMaxGetBufferHostSize_16s_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMinMax_32f_C1R_Ctx.
 * 
 * For common parameter descriptions, see \ref CommonMinMaxGetBufferHostSizeParameters.
 *
 */
NppStatus
nppiMinMaxGetBufferHostSize_32f_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMinMax_8u_C3R_Ctx.
 * 
 * For common parameter descriptions, see \ref CommonMinMaxGetBufferHostSizeParameters.
 *
 */
NppStatus
nppiMinMaxGetBufferHostSize_8u_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMinMax_16u_C3R_Ctx.
 * 
 * For common parameter descriptions, see \ref CommonMinMaxGetBufferHostSizeParameters.
 *
 */
NppStatus
nppiMinMaxGetBufferHostSize_16u_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMinMax_16s_C3R_Ctx.
 * 
 * For common parameter descriptions, see \ref CommonMinMaxGetBufferHostSizeParameters.
 *
 */
NppStatus
nppiMinMaxGetBufferHostSize_16s_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMinMax_32f_C3R_Ctx.
 * 
 * For common parameter descriptions, see \ref CommonMinMaxGetBufferHostSizeParameters.
 *
 */
NppStatus
nppiMinMaxGetBufferHostSize_32f_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMinMax_8u_AC4R_Ctx.
 * 
 * For common parameter descriptions, see \ref CommonMinMaxGetBufferHostSizeParameters.
 *
 */
NppStatus
nppiMinMaxGetBufferHostSize_8u_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMinMax_16u_AC4R_Ctx.
 * 
 * For common parameter descriptions, see \ref CommonMinMaxGetBufferHostSizeParameters.
 *
 */
NppStatus
nppiMinMaxGetBufferHostSize_16u_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMinMax_16s_AC4R_Ctx.
 * 
 * For common parameter descriptions, see \ref CommonMinMaxGetBufferHostSizeParameters.
 *
 */
NppStatus
nppiMinMaxGetBufferHostSize_16s_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMinMax_32f_AC4R_Ctx.
 * 
 * For common parameter descriptions, see \ref CommonMinMaxGetBufferHostSizeParameters.
 *
 */
NppStatus
nppiMinMaxGetBufferHostSize_32f_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);

/** 
 * Buffer size for \ref nppiMinMax_8u_C4R_Ctx.
 * 
 * For common parameter descriptions, see \ref CommonMinMaxGetBufferHostSizeParameters.
 *
 */
NppStatus
nppiMinMaxGetBufferHostSize_8u_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMinMax_16u_C4R_Ctx.
 * 
 * For common parameter descriptions, see \ref CommonMinMaxGetBufferHostSizeParameters.
 *
 */
NppStatus
nppiMinMaxGetBufferHostSize_16u_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMinMax_16s_C4R_Ctx.
 * 
 * For common parameter descriptions, see \ref CommonMinMaxGetBufferHostSizeParameters.
 *
 */
NppStatus
nppiMinMaxGetBufferHostSize_16s_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * 
 * For common parameter descriptions, see \ref CommonMinMaxGetBufferHostSizeParameters.
 *
 */
NppStatus
nppiMinMaxGetBufferHostSize_32f_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);

/** @} */
/** @} image_min_max */

/** 
 * \section image_min_max_index MinMaxIndx
 * @defgroup image_min_max_index MinMaxIndx
 * Primitives for computing the minimal and the maximal values with their indices (X and Y coordinates) of an image.
 * @{
 */

/** @name MinMaxIndx
 * If there are several minima and maxima in the selected region of interest, the function returns ones on the top leftmost position.
 * The scratch buffer is required by the functions.
 * 
 * \section CommonMinMaxIndxParameters Common parameters for nppiMinIndx functions include:
 * 
 * \param pSrc \ref source_image_pointer.
 * \param nSrcStep \ref source_image_line_step.
 * \param oSizeROI \ref roi_specification.
 * \param pMask \ref mask_image_pointer.
 * \param nMaskStep \ref mask_image_line_step.  
 * \param nCOI \ref channel_of_interest_number.
 * \param pMinValue Pointer to the minimum value.
 * \param pMaxValue Pointer to the maximum value.
 * \param pMinIndex Pointer to the indicies (X and Y coordinates) of the minimum value.
 * \param pMaxIndex Pointer to the indicies (X and Y coordinates) of the maximum value.
 * \param pDeviceBuffer Buffer to a scratch memory. Use nppiMinMaxIndxGetBufferHostSize_XX_XXX to determine
 *          the minium number of bytes required.
 * \param nppStreamCtx \ref application_managed_stream_context. 
 * \return \ref image_data_error_codes, \ref roi_error_codes, or NPP_COI_ERROR if an invalid channel of interest is specified.
 * If any of pMinValue, pMaxValue, pMinIndex, or pMaxIndex is not needed, zero pointer must be passed correspondingly.
 * If the mask is filled with zeros, then all the returned values are zeros, i.e., pMinIndex = {0, 0}, pMaxIndex = {0, 0},
 * pMinValue = 0, pMaxValue = 0.
 * @{
 */

/**
 * Computes the minimal and the maximal pixel values with their X and Y coordinates of 1-channel 8-bit unsigned char image. 
 *
 * For common parameter descriptions, see \ref CommonMinMaxIndxParameters.
 *
 */
NppStatus 
nppiMinMaxIndx_8u_C1R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pMinValue, Npp8u * pMaxValue, NppiPoint * pMinIndex, NppiPoint * pMaxIndex, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Computes the minimal and the maximal pixel values with their X and Y coordinates of 1-channel 8-bit signed char image. 
 *
 * For common parameter descriptions, see \ref CommonMinMaxIndxParameters.
 *
 */
NppStatus 
nppiMinMaxIndx_8s_C1R_Ctx(const Npp8s * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8s * pMinValue, Npp8s * pMaxValue, NppiPoint * pMinIndex, NppiPoint * pMaxIndex, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Computes the minimal and the maximal pixel values with their X and Y coordinates of 1-channel 16-bit unsigned short image. 
 *
 * For common parameter descriptions, see \ref CommonMinMaxIndxParameters.
 *
 */
NppStatus 
nppiMinMaxIndx_16u_C1R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp16u * pMinValue, Npp16u * pMaxValue, NppiPoint * pMinIndex, NppiPoint * pMaxIndex, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Computes the minimal and the maximal pixel values with their X and Y coordinates of 1-channel 32-bit floating point image. 
 *
 * For common parameter descriptions, see \ref CommonMinMaxIndxParameters.
 *
 */
NppStatus 
nppiMinMaxIndx_32f_C1R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSizeROI, Npp32f * pMinValue, Npp32f * pMaxValue, NppiPoint * pMinIndex, NppiPoint * pMaxIndex, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Masked one-channel 8-bit unsigned image MinMaxIndx.
 *
 * For common parameter descriptions, see \ref CommonMinMaxIndxParameters.
 *  
 */
NppStatus 
nppiMinMaxIndx_8u_C1MR_Ctx(const Npp8u * pSrc, int nSrcStep, const Npp8u * pMask, int nMaskStep, NppiSize oSizeROI, Npp8u * pMinValue, Npp8u * pMaxValue, NppiPoint * pMinIndex, NppiPoint * pMaxIndex, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Masked one-channel 8-bit signed image MinMaxIndx.
 * For common parameter descriptions, see \ref CommonMinMaxIndxParameters.
 */
NppStatus 
nppiMinMaxIndx_8s_C1MR_Ctx(const Npp8s * pSrc, int nSrcStep, const Npp8u * pMask, int nMaskStep, NppiSize oSizeROI, Npp8s * pMinValue, Npp8s * pMaxValue, NppiPoint * pMinIndex, NppiPoint * pMaxIndex, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Masked one-channel 16-bit unsigned image MinMaxIndx.
 * For common parameter descriptions, see \ref CommonMinMaxIndxParameters.
 */
NppStatus 
nppiMinMaxIndx_16u_C1MR_Ctx(const Npp16u * pSrc, int nSrcStep, const Npp8u * pMask, int nMaskStep, NppiSize oSizeROI, Npp16u * pMinValue, Npp16u * pMaxValue, NppiPoint * pMinIndex, NppiPoint * pMaxIndex, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Masked one-channel 32-bit floating point image MinMaxIndx.
 * For common parameter descriptions, see \ref CommonMinMaxIndxParameters.
 */
NppStatus 
nppiMinMaxIndx_32f_C1MR_Ctx(const Npp32f * pSrc, int nSrcStep, const Npp8u * pMask, int nMaskStep, NppiSize oSizeROI, Npp32f * pMinValue, Npp32f * pMaxValue, NppiPoint * pMinIndex, NppiPoint * pMaxIndex, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 8-bit unsigned image MinMaxIndx affecting only single channel.
 * For common parameter descriptions, see \ref CommonMinMaxIndxParameters.
 */
NppStatus 
nppiMinMaxIndx_8u_C3CR_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSizeROI, int nCOI, Npp8u * pMinValue, Npp8u * pMaxValue, NppiPoint * pMinIndex, NppiPoint * pMaxIndex, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 8-bit signed image MinMaxIndx affecting only single channel.
 * For common parameter descriptions, see \ref CommonMinMaxIndxParameters.
 */
NppStatus 
nppiMinMaxIndx_8s_C3CR_Ctx(const Npp8s * pSrc, int nSrcStep, NppiSize oSizeROI, int nCOI, Npp8s * pMinValue, Npp8s * pMaxValue, NppiPoint * pMinIndex, NppiPoint * pMaxIndex, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 16-bit unsigned image MinMaxIndx affecting only single channel.
 * For common parameter descriptions, see \ref CommonMinMaxIndxParameters.
 */
NppStatus 
nppiMinMaxIndx_16u_C3CR_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSizeROI, int nCOI, Npp16u * pMinValue, Npp16u * pMaxValue, NppiPoint * pMinIndex, NppiPoint * pMaxIndex, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 32-bit floating point image MinMaxIndx affecting only single channel.
 * For common parameter descriptions, see \ref CommonMinMaxIndxParameters.
 */
NppStatus 
nppiMinMaxIndx_32f_C3CR_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSizeROI, int nCOI, Npp32f * pMinValue, Npp32f * pMaxValue, NppiPoint * pMinIndex, NppiPoint * pMaxIndex, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Masked three-channel 8-bit unsigned image MinMaxIndx affecting only single channel.
 * For common parameter descriptions, see \ref CommonMinMaxIndxParameters.
 */ 
NppStatus 
nppiMinMaxIndx_8u_C3CMR_Ctx(const Npp8u * pSrc, int nSrcStep, const Npp8u * pMask, int nMaskStep, NppiSize oSizeROI, int nCOI, Npp8u * pMinValue, Npp8u * pMaxValue, NppiPoint * pMinIndex, NppiPoint * pMaxIndex, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Masked three-channel 8-bit signed image MinMaxIndx affecting only single channel.
 * For common parameter descriptions, see \ref CommonMinMaxIndxParameters.
 */
NppStatus 
nppiMinMaxIndx_8s_C3CMR_Ctx(const Npp8s * pSrc, int nSrcStep, const Npp8u * pMask, int nMaskStep, NppiSize oSizeROI, int nCOI, Npp8s * pMinValue, Npp8s * pMaxValue, NppiPoint * pMinIndex, NppiPoint * pMaxIndex, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Masked three-channel 16-bit unsigned image MinMaxIndx affecting only single channel.
 * For common parameter descriptions, see \ref CommonMinMaxIndxParameters.
 */ 
NppStatus 
nppiMinMaxIndx_16u_C3CMR_Ctx(const Npp16u * pSrc, int nSrcStep, const Npp8u * pMask, int nMaskStep, NppiSize oSizeROI, int nCOI, Npp16u * pMinValue, Npp16u * pMaxValue, NppiPoint * pMinIndex, NppiPoint * pMaxIndex, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Masked three-channel 32-bit floating point image MinMaxIndx affecting only single channel.
 * For common parameter descriptions, see \ref CommonMinMaxIndxParameters.
 */ 
NppStatus 
nppiMinMaxIndx_32f_C3CMR_Ctx(const Npp32f * pSrc, int nSrcStep, const Npp8u * pMask, int nMaskStep, NppiSize oSizeROI, int nCOI, Npp32f * pMinValue, Npp32f * pMaxValue, NppiPoint * pMinIndex, NppiPoint * pMaxIndex, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);

/** @} */

/** @name MinMaxIndxGetBufferHostSize
 * Companion primitives for computing the device buffer size (in bytes) required by the MinMaxIndx primitives.
 * \section CommonMinMaxIndxGetBufferHostSizeParameters
 * Common parameters for nppiMinMaxIndxGetBufferHostSize functions include:
 * \param oSizeROI \ref roi_specification.
 * \param hpBufferSize Required buffer size. Important: hpBufferSize is a 
 *        <em>host pointer.</em> \ref general_scratch_buffer.
 * \param nppStreamCtx \ref application_managed_stream_context. 
 * \return NPP_NULL_POINTER_ERROR if hpBufferSize is 0 (NULL),  \ref roi_error_codes.  
 * @{
 */
/** 
 * Buffer size for \ref nppiMinMaxIndx_8u_C1R_Ctx.
 * 
 * For common parameter descriptions, see \ref CommonMinMaxIndxGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMinMaxIndxGetBufferHostSize_8u_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMinMaxIndx_8s_C1R_Ctx.
 * 
 * For common parameter descriptions, see \ref CommonMinMaxIndxGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMinMaxIndxGetBufferHostSize_8s_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMinMaxIndx_16u_C1R_Ctx.
 * 
 * For common parameter descriptions, see \ref CommonMinMaxIndxGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMinMaxIndxGetBufferHostSize_16u_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMinMaxIndx_32f_C1R_Ctx.
 * 
 * For common parameter descriptions, see \ref CommonMinMaxIndxGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMinMaxIndxGetBufferHostSize_32f_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMinMaxIndx_8u_C1MR_Ctx.
 * 
 * For common parameter descriptions, see \ref CommonMinMaxIndxGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMinMaxIndxGetBufferHostSize_8u_C1MR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMinMaxIndx_8s_C1MR_Ctx.
 * 
 * For common parameter descriptions, see \ref CommonMinMaxIndxGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMinMaxIndxGetBufferHostSize_8s_C1MR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMinMaxIndx_16u_C1MR_Ctx.
 * 
 * For common parameter descriptions, see \ref CommonMinMaxIndxGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMinMaxIndxGetBufferHostSize_16u_C1MR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMinMaxIndx_32f_C1MR_Ctx.
 * 
 * For common parameter descriptions, see \ref CommonMinMaxIndxGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMinMaxIndxGetBufferHostSize_32f_C1MR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMinMaxIndx_8u_C3CR_Ctx.
 * 
 * For common parameter descriptions, see \ref CommonMinMaxIndxGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMinMaxIndxGetBufferHostSize_8u_C3CR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMinMaxIndx_8s_C3CR_Ctx.
 * 
 * For common parameter descriptions, see \ref CommonMinMaxIndxGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMinMaxIndxGetBufferHostSize_8s_C3CR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMinMaxIndx_16u_C3CR_Ctx.
 * 
 * For common parameter descriptions, see \ref CommonMinMaxIndxGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMinMaxIndxGetBufferHostSize_16u_C3CR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMinMaxIndx_32f_C3CR_Ctx.
 * 
 * For common parameter descriptions, see \ref CommonMinMaxIndxGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMinMaxIndxGetBufferHostSize_32f_C3CR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMinMaxIndx_8u_C3CMR_Ctx.
 * 
 * For common parameter descriptions, see \ref CommonMinMaxIndxGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMinMaxIndxGetBufferHostSize_8u_C3CMR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMinMaxIndx_8s_C3CMR_Ctx.
 * 
 * For common parameter descriptions, see \ref CommonMinMaxIndxGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMinMaxIndxGetBufferHostSize_8s_C3CMR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMinMaxIndx_16u_C3CMR_Ctx.
 * 
 * For common parameter descriptions, see \ref CommonMinMaxIndxGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMinMaxIndxGetBufferHostSize_16u_C3CMR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMinMaxIndx_32f_C3CMR_Ctx.
 * 
 * For common parameter descriptions, see \ref CommonMinMaxIndxGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMinMaxIndxGetBufferHostSize_32f_C3CMR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);

/** @} */
/** @} image_min_max_index */

/** 
 * \section image_mean Mean
 * @defgroup image_mean Mean
 * Primitives for computing the arithmetic mean of all the pixel values in an image.
 * @{
 */

/** @name Mean
 * Given an image \f$pSrc\f$ with width \f$W\f$ and height \f$H\f$, the arithmetic mean
 * will be computed as
 * \f[Mean = \frac{1}{W\cdot H}\sum_{j=0}^{H-1}\sum_{i=0}^{W-1}pSrc(j,i)\f]
 * The mean functions require additional scratch buffer for computations.
 * 
 * \section CommonMeanParameters Common parameters for nppiMean functions include:
 * 
 * \param pSrc \ref source_image_pointer.
 * \param nSrcStep \ref source_image_line_step.
 * \param oSizeROI \ref roi_specification. 
 * \param pMask \ref mask_image_pointer.
 * \param nMaskStep \ref mask_image_line_step.
 * \param nCOI \ref channel_of_interest_number.
 * \param pDeviceBuffer Pointer to the required device memory allocation, \ref general_scratch_buffer
 *        Use nppiMeanGetBufferHostSize_XX_XXX to determine the minium number of bytes required.
 * \param pMean Pointer to the computed mean result.
 * \param nppStreamCtx \ref application_managed_stream_context. 
 * \return \ref image_data_error_codes, \ref roi_error_codes, or NPP_COI_ERROR if an invalid channel of interest is specified.s
 * 
 * @{
 */
/**
 * One-channel 8-bit unsigned image Mean.
 *
 * For common parameter descriptions, see \ref CommonMeanParameters.
 *
 */
NppStatus 
nppiMean_8u_C1R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp64f * pMean, NppStreamContext nppStreamCtx);
/**
 * One-channel 16-bit unsigned image Mean.
 *
 * For common parameter descriptions, see \ref CommonMeanParameters.
 *
 */
NppStatus 
nppiMean_16u_C1R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp64f * pMean, NppStreamContext nppStreamCtx);
/**
 * One-channel 16-bit signed image Mean.
 *
 * For common parameter descriptions, see \ref CommonMeanParameters.
 *
 */
NppStatus 
nppiMean_16s_C1R_Ctx(const Npp16s * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp64f * pMean, NppStreamContext nppStreamCtx);
/**
 * One-channel 32-bit floating point image Mean.
 *
 * For common parameter descriptions, see \ref CommonMeanParameters.
 *
 */
NppStatus 
nppiMean_32f_C1R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp64f * pMean, NppStreamContext nppStreamCtx);
/**
 * Three-channel 8-bit unsigned image Mean.
 *
 * For common parameter descriptions, see \ref CommonMeanParameters.
 *
 */
NppStatus 
nppiMean_8u_C3R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp64f aMean[3], NppStreamContext nppStreamCtx);
/**
 * Three-channel 16-bit unsigned image Mean.
 *
 * For common parameter descriptions, see \ref CommonMeanParameters.
 *
 */
NppStatus 
nppiMean_16u_C3R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp64f aMean[3], NppStreamContext nppStreamCtx);
/**
 * Three-channel 16-bit signed image Mean.
 *
 * For common parameter descriptions, see \ref CommonMeanParameters.
 *
 */
NppStatus 
nppiMean_16s_C3R_Ctx(const Npp16s * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp64f aMean[3], NppStreamContext nppStreamCtx);
/**
 * Three-channel 32-bit floating point image Mean.
 *
 * For common parameter descriptions, see \ref CommonMeanParameters.
 *
 */
NppStatus 
nppiMean_32f_C3R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp64f aMean[3], NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image Mean.
 *
 * For common parameter descriptions, see \ref CommonMeanParameters.
 *
 */
NppStatus 
nppiMean_8u_C4R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp64f aMean[4], NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit unsigned image Mean.
 *
 * For common parameter descriptions, see \ref CommonMeanParameters.
 *
 */
NppStatus 
nppiMean_16u_C4R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp64f aMean[4], NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit signed image Mean.
 *
 * For common parameter descriptions, see \ref CommonMeanParameters.
 *
 */
NppStatus 
nppiMean_16s_C4R_Ctx(const Npp16s * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp64f aMean[4], NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit floating point image Mean.
 *
 * For common parameter descriptions, see \ref CommonMeanParameters.
 *
 */
NppStatus 
nppiMean_32f_C4R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp64f aMean[4], NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image Mean ignoring alpha channel.
 *
 * For common parameter descriptions, see \ref CommonMeanParameters.
 *
 */
NppStatus 
nppiMean_8u_AC4R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp64f aMean[3], NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit unsigned image Mean ignoring alpha channel.
 *
 * For common parameter descriptions, see \ref CommonMeanParameters.
 *
 */
NppStatus 
nppiMean_16u_AC4R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp64f aMean[3], NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit signed image Mean ignoring alpha channel.
 *
 * For common parameter descriptions, see \ref CommonMeanParameters.
 *
 */
NppStatus 
nppiMean_16s_AC4R_Ctx(const Npp16s * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp64f aMean[3], NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit floating point image Mean ignoring alpha channel.
 *
 * For common parameter descriptions, see \ref CommonMeanParameters.
 *
 */
NppStatus 
nppiMean_32f_AC4R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp64f aMean[3], NppStreamContext nppStreamCtx);
/**
 * Masked one-channel 8-bit unsigned image Mean.
 *
 * For common parameter descriptions, see \ref CommonMeanParameters.

 */
NppStatus 
nppiMean_8u_C1MR_Ctx(const Npp8u * pSrc, int nSrcStep, const Npp8u * pMask, int nMaskStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp64f * pMean, NppStreamContext nppStreamCtx);
/**
 * Masked one-channel 8-bit signed image Mean.
 *
 * For common parameter descriptions, see \ref CommonMeanParameters.
 */
NppStatus 
nppiMean_8s_C1MR_Ctx(const Npp8s * pSrc, int nSrcStep, const Npp8u * pMask, int nMaskStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp64f * pMean, NppStreamContext nppStreamCtx);
/**
 * Masked one-channel 16-bit unsigned image Mean.
 *
 * For common parameter descriptions, see \ref CommonMeanParameters.
 */
NppStatus 
nppiMean_16u_C1MR_Ctx(const Npp16u * pSrc, int nSrcStep, const Npp8u * pMask, int nMaskStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp64f * pMean, NppStreamContext nppStreamCtx);
/**
 * Masked one-channel 32-bit floating point image Mean.
 *
 * For common parameter descriptions, see \ref CommonMeanParameters.
 */
NppStatus 
nppiMean_32f_C1MR_Ctx(const Npp32f * pSrc, int nSrcStep, const Npp8u * pMask, int nMaskStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp64f * pMean, NppStreamContext nppStreamCtx);
/**
 * Masked three-channel 8-bit unsigned image Mean affecting only single channel.
 *
 * For common parameter descriptions, see \ref CommonMeanParameters.
 */
NppStatus 
nppiMean_8u_C3CMR_Ctx(const Npp8u * pSrc, int nSrcStep, const Npp8u * pMask, int nMaskStep, NppiSize oSizeROI, int nCOI, Npp8u * pDeviceBuffer, Npp64f * pMean, NppStreamContext nppStreamCtx);
/**
 * Masked three-channel 8-bit signed image Mean affecting only single channel.
 *
 * For common parameter descriptions, see \ref CommonMeanParameters.
 */
NppStatus 
nppiMean_8s_C3CMR_Ctx(const Npp8s * pSrc, int nSrcStep, const Npp8u * pMask, int nMaskStep, NppiSize oSizeROI, int nCOI, Npp8u * pDeviceBuffer, Npp64f * pMean, NppStreamContext nppStreamCtx);
/**
 * Masked three-channel 16-bit unsigned image Mean affecting only single channel.
 *
 * For common parameter descriptions, see \ref CommonMeanParameters.
 */
NppStatus 
nppiMean_16u_C3CMR_Ctx(const Npp16u * pSrc, int nSrcStep, const Npp8u * pMask, int nMaskStep, NppiSize oSizeROI, int nCOI, Npp8u * pDeviceBuffer, Npp64f * pMean, NppStreamContext nppStreamCtx);
/**
 * Masked three-channel 32-bit floating point image Mean affecting only single channel.
 *
 * For common parameter descriptions, see \ref CommonMeanParameters.
 */
NppStatus 
nppiMean_32f_C3CMR_Ctx(const Npp32f * pSrc, int nSrcStep, const Npp8u * pMask, int nMaskStep, NppiSize oSizeROI, int nCOI, Npp8u * pDeviceBuffer, Npp64f * pMean, NppStreamContext nppStreamCtx);

/** @} */

/** @name MeanGetBufferHostSize
 * Companion primitives for computing the device buffer size (in bytes) required by the Mean primitives.
 * \section CommonMeanGetBufferHostSizeParameters
 * Common parameters for nppiMeanGetBufferHostSize functions include:
 * \param oSizeROI \ref roi_specification.
 * \param hpBufferSize Required buffer size. Important: hpBufferSize is a 
 *        <em>host pointer.</em> \ref general_scratch_buffer.
 * \param nppStreamCtx \ref application_managed_stream_context. 
 * \return NPP_NULL_POINTER_ERROR if hpBufferSize is 0 (NULL),  \ref roi_error_codes.  
 * @{
 */
/**
 * Buffer size for \ref nppiMean_8u_C1R_Ctx.
 * 
 */
NppStatus 
nppiMeanGetBufferHostSize_8u_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/**
 * Buffer size for \ref nppiMean_16u_C1R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMeanGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMeanGetBufferHostSize_16u_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/**
 * Buffer size for \ref nppiMean_16s_C1R_Ctx.
 * 
 * For common parameter descriptions, see \ref CommonMeanGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMeanGetBufferHostSize_16s_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMean_32f_C1R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMeanGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMeanGetBufferHostSize_32f_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMean_8u_C3R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMeanGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMeanGetBufferHostSize_8u_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMean_16u_C3R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMeanGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMeanGetBufferHostSize_16u_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMean_16s_C3R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMeanGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMeanGetBufferHostSize_16s_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMean_32f_C3R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMeanGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMeanGetBufferHostSize_32f_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMean_8u_AC4R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMeanGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMeanGetBufferHostSize_8u_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMean_16u_AC4R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMeanGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMeanGetBufferHostSize_16u_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMean_16s_AC4R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMeanGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMeanGetBufferHostSize_16s_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMean_32f_AC4R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMeanGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMeanGetBufferHostSize_32f_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMean_8u_C4R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMeanGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMeanGetBufferHostSize_8u_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMean_16u_C4R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMeanGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMeanGetBufferHostSize_16u_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMean_16s_C4R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMeanGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMeanGetBufferHostSize_16s_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMean_32f_C4R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMeanGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMeanGetBufferHostSize_32f_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMean_8u_C1MR_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMeanGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMeanGetBufferHostSize_8u_C1MR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMean_8s_C1MR_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMeanGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMeanGetBufferHostSize_8s_C1MR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMean_16u_C1MR_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMeanGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMeanGetBufferHostSize_16u_C1MR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMean_32f_C1MR_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMeanGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMeanGetBufferHostSize_32f_C1MR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMean_8u_C3CMR_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMeanGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMeanGetBufferHostSize_8u_C3CMR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMean_8s_C3CMR_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMeanGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMeanGetBufferHostSize_8s_C3CMR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMean_16u_C3CMR_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMeanGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMeanGetBufferHostSize_16u_C3CMR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMean_32f_C3CMR_Ctx.
 *
 * For common parameter descriptions, see \ref CommonMeanGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMeanGetBufferHostSize_32f_C3CMR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize /* host pointer */, NppStreamContext nppStreamCtx);

/** @} */
/** @} image_mean */

/** 
 * \section image_mean_stddev Mean_StdDev
 * @defgroup image_mean_stddev Mean_StdDev
 * Primitives for computing both the arithmetic mean and the standard deviation of an image.
 * @{
 */

/** @name Mean_StdDev
 * Given an image \f$pSrc\f$ with width \f$W\f$ and height \f$H\f$, the mean and the standard deviation
 * will be computed as
 * \f[Mean = \frac{1}{W\cdot H}\sum_{j=0}^{H-1}\sum_{i=0}^{W-1}pSrc(j,i)\f]
 * \f[StdDev = \sqrt{\frac{1}{W\cdot H}\sum_{j=0}^{H-1}\sum_{i=0}^{W-1}(pSrc(j,i)-Mean)^2}\f]
 * The Mean_StdDev primitives require additional scratch buffer for computations.
 *  
 * \section CommonMeanStdDevParameters Common parameters for nppiMean_StdDev functions include:
 *
 * \param pSrc \ref source_image_pointer.
 * \param nSrcStep \ref source_image_line_step.
 * \param oSizeROI \ref roi_specification.
 * \param pMask \ref mask_image_pointer.
 * \param nMaskStep \ref mask_image_line_step.
 * \param nCOI \ref channel_of_interest_number.
 * \param pDeviceBuffer Pointer to the required device memory allocation, \ref general_scratch_buffer
 *        Use \ref MeanStdDevGetBufferHostSize to determine the minium number of bytes required.
 * \param pMean Pointer to the computed mean.
 * \param pStdDev Pointer to the computed standard deviation.
 * \param nppStreamCtx \ref application_managed_stream_context. 
 * \return \ref image_data_error_codes, \ref roi_error_codes, or NPP_COI_ERROR if an invalid channel of interest is specified.
 * 
 * @{
 */

/**
 * One-channel 8-bit unsigned image Mean_StdDev.
 * 
 * For common parameter descriptions, see \ref CommonMeanStdDevParameters.
 *
 */
NppStatus nppiMean_StdDev_8u_C1R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp64f * pMean, Npp64f * pStdDev , NppStreamContext nppStreamCtx);
/**
 * One-channel 8-bit signed image Mean_StdDev.
 * 
 * For common parameter descriptions, see \ref CommonMeanStdDevParameters.
 *
 */
NppStatus nppiMean_StdDev_8s_C1R_Ctx(const Npp8s * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp64f * pMean, Npp64f * pStdDev , NppStreamContext nppStreamCtx);
/**
 * One-channel 16-bit unsigned image Mean_StdDev.
 * 
 * For common parameter descriptions, see \ref CommonMeanStdDevParameters.
 *
 */
NppStatus nppiMean_StdDev_16u_C1R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp64f * pMean, Npp64f * pStdDev , NppStreamContext nppStreamCtx);
/**
 * One-channel 32-bit floating point image Mean_StdDev.
 * 
 * For common parameter descriptions, see \ref CommonMeanStdDevParameters.
 *
 */
NppStatus nppiMean_StdDev_32f_C1R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp64f * pMean, Npp64f * pStdDev , NppStreamContext nppStreamCtx);
/**
 * Masked one-channel 8-bit unsigned image Mean_StdDev.
 * 
 * For common parameter descriptions, see \ref CommonMeanStdDevParameters.
 *
 */
NppStatus nppiMean_StdDev_8u_C1MR_Ctx(const Npp8u * pSrc, int nSrcStep, const Npp8u * pMask, int nMaskStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp64f * pMean, Npp64f * pStdDev , NppStreamContext nppStreamCtx);
/**
 * Masked one-channel 8-bit signed image Mean_StdDev.
 * 
 * For common parameter descriptions, see \ref CommonMeanStdDevParameters.
 */
NppStatus nppiMean_StdDev_8s_C1MR_Ctx(const Npp8s * pSrc, int nSrcStep, const Npp8u * pMask, int nMaskStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp64f * pMean, Npp64f * pStdDev , NppStreamContext nppStreamCtx);
/**
 * Masked one-channel 16-bit unsigned image Mean_StdDev.
 * 
 * For common parameter descriptions, see \ref CommonMeanStdDevParameters.
 *
 */
NppStatus nppiMean_StdDev_16u_C1MR_Ctx(const Npp16u * pSrc, int nSrcStep, const Npp8u * pMask, int nMaskStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp64f * pMean, Npp64f * pStdDev , NppStreamContext nppStreamCtx);
/**
 * Masked one-channel 32-bit floating point image Mean_StdDev.
 * 
 * For common parameter descriptions, see \ref CommonMeanStdDevParameters.
 *
 */
NppStatus nppiMean_StdDev_32f_C1MR_Ctx(const Npp32f * pSrc, int nSrcStep, const Npp8u * pMask, int nMaskStep, NppiSize oSizeROI, Npp8u * pDeviceBuffer, Npp64f * pMean, Npp64f * pStdDev , NppStreamContext nppStreamCtx);

/** @} */

 /** @name Channel Mean_StdDev
 * See \ref channel_of_interest.
 * @{
 */
 
/**
 * Three-channel 8-bit unsigned image Mean_StdDev affecting only single channel.
 * 
 * For common parameter descriptions, see \ref CommonMeanStdDevParameters.
 */
NppStatus nppiMean_StdDev_8u_C3CR_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSizeROI, int nCOI, Npp8u * pDeviceBuffer, Npp64f * pMean, Npp64f * pStdDev , NppStreamContext nppStreamCtx);
/**
 * Three-channel 8-bit signed image Mean_StdDev affecting only single channel.
 * 
 * For common parameter descriptions, see \ref CommonMeanStdDevParameters.
 * 
 */
NppStatus nppiMean_StdDev_8s_C3CR_Ctx(const Npp8s * pSrc, int nSrcStep, NppiSize oSizeROI, int nCOI, Npp8u * pDeviceBuffer, Npp64f * pMean, Npp64f * pStdDev , NppStreamContext nppStreamCtx);
/**
 * Three-channel 16-bit unsigned image Mean_StdDev affecting only single channel.
 * 
 * For common parameter descriptions, see \ref CommonMeanStdDevParameters.
 * 
 */
NppStatus nppiMean_StdDev_16u_C3CR_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSizeROI, int nCOI, Npp8u * pDeviceBuffer, Npp64f * pMean, Npp64f * pStdDev , NppStreamContext nppStreamCtx);
/**
 * Three-channel 32-bit floating point image Mean_StdDev affecting only single channel.
 * 
 * For common parameter descriptions, see \ref CommonMeanStdDevParameters.
 *
 * 
 */
NppStatus nppiMean_StdDev_32f_C3CR_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSizeROI, int nCOI, Npp8u * pDeviceBuffer, Npp64f * pMean, Npp64f * pStdDev , NppStreamContext nppStreamCtx);
/**
 * Masked three-channel 8-bit unsigned image Mean_StdDev.
 * 
 * For common parameter descriptions, see \ref CommonMeanStdDevParameters.
 * 
 */
NppStatus nppiMean_StdDev_8u_C3CMR_Ctx(const Npp8u * pSrc, int nSrcStep, const Npp8u * pMask, int nMaskStep, NppiSize oSizeROI, int nCOI, Npp8u * pDeviceBuffer, Npp64f * pMean, Npp64f * pStdDev , NppStreamContext nppStreamCtx);
/**
 * Masked three-channel 8-bit signed image Mean_StdDev.
 * 
 * For common parameter descriptions, see \ref CommonMeanStdDevParameters.
 *
 */
NppStatus nppiMean_StdDev_8s_C3CMR_Ctx(const Npp8s * pSrc, int nSrcStep, const Npp8u * pMask, int nMaskStep, NppiSize oSizeROI, int nCOI, Npp8u * pDeviceBuffer, Npp64f * pMean, Npp64f * pStdDev , NppStreamContext nppStreamCtx);
/**
 * Masked three-channel 16-bit unsigned image Mean_StdDev.
 * 
 * For common parameter descriptions, see \ref CommonMeanStdDevParameters.
 * 
 */
NppStatus nppiMean_StdDev_16u_C3CMR_Ctx(const Npp16u * pSrc, int nSrcStep, const Npp8u * pMask, int nMaskStep, NppiSize oSizeROI, int nCOI, Npp8u * pDeviceBuffer, Npp64f * pMean, Npp64f * pStdDev , NppStreamContext nppStreamCtx);
/**
 * Masked three-channel 32-bit floating point image Mean_StdDev.
 * 
 * For common parameter descriptions, see \ref CommonMeanStdDevParameters.
 * 
 */
NppStatus nppiMean_StdDev_32f_C3CMR_Ctx(const Npp32f * pSrc, int nSrcStep, const Npp8u * pMask, int nMaskStep, NppiSize oSizeROI, int nCOI, Npp8u * pDeviceBuffer, Npp64f * pMean, Npp64f * pStdDev , NppStreamContext nppStreamCtx);

/** @} */

/** \section MeanStdDevGetBufferHostSize
 * 
 * Companion primitives for computing the device buffer size (in bytes) required by the Mean_StdDev primitives.
 * 
 * @{
 */
/** 
 * Buffer size for \ref nppiMean_StdDev_8u_C1R_Ctx.
 *
 * \section CommonMeanStdDevGetBufferHostSizeParameters Common parameters for MeanStdDevGetBufferHostSize functions include:
 *
 */
NppStatus 
nppiMeanStdDevGetBufferHostSize_8u_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);

/** 
 * Buffer size for \ref nppiMean_StdDev_8s_C1R_Ctx.
 * 
 * For common parameter descriptions, see \ref CommonMeanStdDevGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMeanStdDevGetBufferHostSize_8s_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMean_StdDev_16u_C1R_Ctx.
 * 
 * For common parameter descriptions, see \ref CommonMeanStdDevGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMeanStdDevGetBufferHostSize_16u_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMean_StdDev_32f_C1R_Ctx.
 * 
 * For common parameter descriptions, see \ref CommonMeanStdDevGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMeanStdDevGetBufferHostSize_32f_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMean_StdDev_8u_C1MR_Ctx.
 * 
 * For common parameter descriptions, see \ref CommonMeanStdDevGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMeanStdDevGetBufferHostSize_8u_C1MR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMean_StdDev_8s_C1MR_Ctx.
 * 
 * For common parameter descriptions, see \ref CommonMeanStdDevGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMeanStdDevGetBufferHostSize_8s_C1MR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMean_StdDev_16u_C1MR_Ctx.
 * 
 * For common parameter descriptions, see \ref CommonMeanStdDevGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMeanStdDevGetBufferHostSize_16u_C1MR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMean_StdDev_32f_C1MR_Ctx.
 * 
 * For common parameter descriptions, see \ref CommonMeanStdDevGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMeanStdDevGetBufferHostSize_32f_C1MR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMean_StdDev_8u_C3CR_Ctx.
 * 
 * For common parameter descriptions, see \ref CommonMeanStdDevGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMeanStdDevGetBufferHostSize_8u_C3CR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMean_StdDev_8s_C3CR_Ctx.
 * 
 * For common parameter descriptions, see \ref CommonMeanStdDevGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMeanStdDevGetBufferHostSize_8s_C3CR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMean_StdDev_16u_C3CR_Ctx.
 * 
 * For common parameter descriptions, see \ref CommonMeanStdDevGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMeanStdDevGetBufferHostSize_16u_C3CR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMean_StdDev_32f_C3CR_Ctx.
 * 
 * For common parameter descriptions, see \ref CommonMeanStdDevGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMeanStdDevGetBufferHostSize_32f_C3CR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMean_StdDev_8u_C3CMR_Ctx.
 * 
 * For common parameter descriptions, see \ref CommonMeanStdDevGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMeanStdDevGetBufferHostSize_8u_C3CMR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMean_StdDev_8s_C3CMR_Ctx.
 * 
 * For common parameter descriptions, see \ref CommonMeanStdDevGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMeanStdDevGetBufferHostSize_8s_C3CMR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMean_StdDev_16u_C3CMR_Ctx.
 * 
 * For common parameter descriptions, see \ref CommonMeanStdDevGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMeanStdDevGetBufferHostSize_16u_C3CMR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMean_StdDev_32f_C3CMR_Ctx.
 * 
 * For common parameter descriptions, see \ref CommonMeanStdDevGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiMeanStdDevGetBufferHostSize_32f_C3CMR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);

/** @} */
/** @} image_mean_stddev */

/** 
 * \section image_norm Image Norms
 * @defgroup image_norm Image Norms
 *
 * Primitives for computing the norms of an image, the norms of difference, and the relative errors of two images.
 * Given an image \f$pSrc\f$ with width \f$W\f$ and height \f$H\f$, 
 * -# The infinity norm (Norm_Inf) is defined as the largest absolute pixel value of the image.
 * -# The L1 norm (Norm_L1) is defined as the sum of the absolute pixel value of the image, i.e., \f[Norm\_L1 = \sum_{j=0}^{H-1}\sum_{i=0}^{W-1}\left| pSrc(j,i)\right|\f].
 * -# The L2 norm (Norm_L2) is defined as the square root of the sum of the squared absolute pixel value of the image, i.e.,\f[Norm\_L2 = \sqrt{\sum_{j=0}^{H-1}\sum_{i=0}^{W-1}\left| pSrc(j,i)\right| ^2}\f].
 * 
 * Given two images \f$pSrc1\f$ and \f$pSrc2\f$ both with width \f$W\f$ and height \f$H\f$, 
 * -# The infinity norm of differece (NormDiff_Inf) is defined as the largest absolute difference between pixels of two images.
 * -# The L1 norm of differece (NormDiff_L1) is defined as the sum of the absolute difference between pixels of two images, i.e., \f[NormDiff\_L1 = \sum_{j=0}^{H-1}\sum_{i=0}^{W-1}\left| pSrc1(j,i)-pSrc2(j,i)\right|\f].
 * -# The L2 norm of differece (NormDiff_L2) is defined as the squared root of the sum of the squared absolute difference between pixels of two images, i.e.,\f[NormDiff\_L2 = \sqrt{\sum_{j=0}^{H-1}\sum_{i=0}^{W-1}\left| pSrc1(j,i)-pSrc2(j,i)\right| ^2}\f].
 * 
 * Given two images \f$pSrc1\f$ and \f$pSrc2\f$ both with width \f$W\f$ and height \f$H\f$, 
 * -# The relative error for the infinity norm of differece (NormRel_Inf) is defined as NormDiff_Inf divided by the infinity norm of the second image, i.e., \f[NormRel\_Inf = \frac{NormDiff\_Inf}{Norm\_Inf_{src2}}\f]
 * -# The relative error for the L1 norm of differece (NormRel_L1) is defined as NormDiff_L1 divided by the L1 norm of the second image, i.e., \f[NormRel\_L1 = \frac{NormDiff\_L1}{Norm\_L1_{src2}}\f]
 * -# The relative error for the L2 norm of differece (NormRel_L2) is defined as NormDiff_L2 divided by the L2 norm of the second image, i.e., \f[NormRel\_L2 = \frac{NormDiff\_L2}{Norm\_L2_{src2}}\f]
 * 
 * The norm functions require the addition device scratch buffer for the computations.
 * 
 * \subsection CommonNormParameters Common parameters for nppiNorm functions include:
 *
 * \param pSrc1 \ref source_image_pointer.
 * \param nSrc1Step \ref source_image_line_step.
 * \param pSrc2 \ref source_image_pointer.
 * \param nSrc2Step \ref source_image_line_step.
 * \param oSizeROI \ref roi_specification.
 * \param pMask \ref mask_image_pointer.
 * \param nMaskStep \ref mask_image_line_step.
 * \param nCOI \ref channel_of_interest_number.
 * \param pNorm Pointer to the norm value.
 * \param pNormDiff Pointer to the computed norm of differences. 
 * \param pNormRel Pointer to the computed relative error for the infinity norm of two images.
 * \param pDeviceBuffer Pointer to the required device memory allocation, \ref general_scratch_buffer. 
 *        Use nppiNormInfGetBufferHostSize_XX_XXX to compute the required size (in bytes).
 * \param nppStreamCtx \ref application_managed_stream_context. 
 * \return \ref image_data_error_codes, \ref roi_error_codes, or NPP_COI_ERROR if an invalid channel of interest is specified, or NPP_NOT_EVEN_STEP_ERROR if an invalid floating-point image is specified.
 * 
 * @{
 *
 */ 

/** 
 * \section image_inf_norm Norm_Inf
 * @defgroup image_inf_norm Norm_Inf
 * Primitives for computing the infinity norm of an image. 
 * @{
 */

/** @name Basic Norm_Inf
 *   
 * @{
 */

/**
 * One-channel 8-bit unsigned image Norm_Inf.
 *
 * For common parameter descriptions, see \ref CommonNormParameters.
 *
 */
NppStatus nppiNorm_Inf_8u_C1R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp64f * pNorm, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 16-bit unsigned image Norm_Inf.
 *
 * For common parameter descriptions, see \ref CommonNormParameters.
 *
 */
NppStatus nppiNorm_Inf_16u_C1R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp64f * pNorm, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 16-bit signed image Norm_Inf.
 *
 * For common parameter descriptions, see \ref CommonNormParameters.
 *
 */
NppStatus nppiNorm_Inf_16s_C1R_Ctx(const Npp16s * pSrc, int nSrcStep, NppiSize oSizeROI, Npp64f * pNorm, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 32-bit signed image Norm_Inf.
 *
 * For common parameter descriptions, see \ref CommonNormParameters.
 *
 */
NppStatus nppiNorm_Inf_32s_C1R_Ctx(const Npp32s * pSrc, int nSrcStep, NppiSize oSizeROI, Npp64f * pNorm, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 32-bit floating point image Norm_Inf.
 *
 * For common parameter descriptions, see \ref CommonNormParameters.
 *
 */
NppStatus nppiNorm_Inf_32f_C1R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSizeROI, Npp64f * pNorm, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 8-bit unsigned image Norm_Inf.
 *
 * For common parameter descriptions, see \ref CommonNormParameters.
 *
 */
NppStatus nppiNorm_Inf_8u_C3R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp64f aNorm[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 16-bit unsigned image Norm_Inf.
 *
 * For common parameter descriptions, see \ref CommonNormParameters.
 *
 */
NppStatus nppiNorm_Inf_16u_C3R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp64f aNorm[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 16-bit signed image Norm_Inf.
 *
 * For common parameter descriptions, see \ref CommonNormParameters.
 *
 */
NppStatus nppiNorm_Inf_16s_C3R_Ctx(const Npp16s * pSrc, int nSrcStep, NppiSize oSizeROI, Npp64f aNorm[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 32-bit floating point image Norm_Inf.
 *
 * For common parameter descriptions, see \ref CommonNormParameters.
 *
 */
NppStatus nppiNorm_Inf_32f_C3R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSizeROI, Npp64f aNorm[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image Norm_Inf ignoring alpha channel.
 *
 * For common parameter descriptions, see \ref CommonNormParameters.
 *
 */
NppStatus nppiNorm_Inf_8u_AC4R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp64f aNorm[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit unsigned image Norm_Inf ignoring alpha channel.
 *
 * For common parameter descriptions, see \ref CommonNormParameters.
 *
 */
NppStatus nppiNorm_Inf_16u_AC4R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp64f aNorm[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit signed image Norm_Inf ignoring alpha channel.
 *
 * For common parameter descriptions, see \ref CommonNormParameters.
 *
 */
NppStatus nppiNorm_Inf_16s_AC4R_Ctx(const Npp16s * pSrc, int nSrcStep, NppiSize oSizeROI, Npp64f aNorm[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit floating point image Norm_Inf ignoring alpha channel.
 *
 * For common parameter descriptions, see \ref CommonNormParameters.
 *
 */
NppStatus nppiNorm_Inf_32f_AC4R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSizeROI, Npp64f aNorm[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image Norm_Inf.
 *
 * For common parameter descriptions, see \ref CommonNormParameters.
 *
 */
NppStatus nppiNorm_Inf_8u_C4R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp64f aNorm[4], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit unsigned image Norm_Inf.
 *
 * For common parameter descriptions, see \ref CommonNormParameters.
 *
 */
NppStatus nppiNorm_Inf_16u_C4R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp64f aNorm[4], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit signed image Norm_Inf.
 *
 * For common parameter descriptions, see \ref CommonNormParameters.
 *
 */
NppStatus nppiNorm_Inf_16s_C4R_Ctx(const Npp16s * pSrc, int nSrcStep, NppiSize oSizeROI, Npp64f aNorm[4], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit floating point image Norm_Inf.
 *
 * For common parameter descriptions, see \ref CommonNormParameters.
 *
 */
NppStatus nppiNorm_Inf_32f_C4R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSizeROI, Npp64f aNorm[4], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Masked one-channel 8-bit unsigned image Norm_Inf.
 *
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNorm_Inf_8u_C1MR_Ctx(const Npp8u * pSrc, int nSrcStep, const Npp8u * pMask, int nMaskStep, NppiSize oSizeROI, Npp64f * pNorm, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Masked one-channel 8-bit signed image Norm_Inf.
 *
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNorm_Inf_8s_C1MR_Ctx(const Npp8s * pSrc, int nSrcStep, const Npp8u * pMask, int nMaskStep, NppiSize oSizeROI, Npp64f * pNorm, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Masked one-channel 16-bit unsigned image Norm_Inf.
 *
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNorm_Inf_16u_C1MR_Ctx(const Npp16u * pSrc, int nSrcStep, const Npp8u * pMask, int nMaskStep, NppiSize oSizeROI, Npp64f * pNorm, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Masked one-channel 32-bit floating point image Norm_Inf.
 *
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNorm_Inf_32f_C1MR_Ctx(const Npp32f * pSrc, int nSrcStep, const Npp8u * pMask, int nMaskStep, NppiSize oSizeROI, Npp64f * pNorm, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Masked three-channel 8-bit unsigned image Norm_Inf affecting only single channel.
 *
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNorm_Inf_8u_C3CMR_Ctx(const Npp8u * pSrc, int nSrcStep, const Npp8u * pMask, int nMaskStep, NppiSize oSizeROI, int nCOI, Npp64f * pNorm, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Masked three-channel 8-bit signed image Norm_Inf affecting only single channel.
 *
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNorm_Inf_8s_C3CMR_Ctx(const Npp8s * pSrc, int nSrcStep, const Npp8u * pMask, int nMaskStep, NppiSize oSizeROI, int nCOI, Npp64f * pNorm, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Masked three-channel 16-bit unsigned image Norm_Inf affecting only single channel.
 *
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNorm_Inf_16u_C3CMR_Ctx(const Npp16u * pSrc, int nSrcStep, const Npp8u * pMask, int nMaskStep, NppiSize oSizeROI, int nCOI, Npp64f * pNorm, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Masked three-channel 32-bit floating point image Norm_Inf affecting only single channel.
 *
 * For common parameter descriptions, see \ref CommonNormParameters.
 * 
 */
NppStatus nppiNorm_Inf_32f_C3CMR_Ctx(const Npp32f * pSrc, int nSrcStep, const Npp8u * pMask, int nMaskStep, NppiSize oSizeROI, int nCOI, Npp64f * pNorm, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);

/** @} */

/** @name NormInfGetBufferHostSize
 * Companion primitives for computing the device buffer size (in bytes) required by the Norm_Inf primitives.
 * \section CommonNormInfGetBufferHostSizeParameters
 * Common parameters for nppiNormInfGetBufferHostSize functions include:
 * \param oSizeROI \ref roi_specification.
 * \param hpBufferSize Required buffer size. Important: hpBufferSize is a 
 *        <em>host pointer.</em> \ref general_scratch_buffer.
 * \param nppStreamCtx \ref application_managed_stream_context. 
 * \return NPP_NULL_POINTER_ERROR if hpBufferSize is 0 (NULL),  \ref roi_error_codes.  
 * @{
 */

/** Buffer size for \ref nppiNorm_Inf_8u_C1R_Ctx.
 * 
 * For common parameter descriptions, see \ref CommonNormInfGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiNormInfGetBufferHostSize_8u_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNorm_Inf_16u_C1R_Ctx.
 * 
 * For common parameter descriptions, see \ref CommonNormInfGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiNormInfGetBufferHostSize_16u_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNorm_Inf_16s_C1R_Ctx.
 * 
 * For common parameter descriptions, see \ref CommonNormInfGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiNormInfGetBufferHostSize_16s_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNorm_Inf_32s_C1R_Ctx.
 * 
 * For common parameter descriptions, see \ref CommonNormInfGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiNormInfGetBufferHostSize_32s_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNorm_Inf_32f_C1R_Ctx.
 * 
 * For common parameter descriptions, see \ref CommonNormInfGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiNormInfGetBufferHostSize_32f_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNorm_Inf_8u_C1MR_Ctx.
 * 
 * For common parameter descriptions, see \ref CommonNormInfGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiNormInfGetBufferHostSize_8u_C1MR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNorm_Inf_8s_C1MR_Ctx.
 * 
 * For common parameter descriptions, see \ref CommonNormInfGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiNormInfGetBufferHostSize_8s_C1MR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNorm_Inf_16u_C1MR_Ctx.
 * 
 * For common parameter descriptions, see \ref CommonNormInfGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiNormInfGetBufferHostSize_16u_C1MR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNorm_Inf_32f_C1MR_Ctx.
 * 
 * For common parameter descriptions, see \ref CommonNormInfGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiNormInfGetBufferHostSize_32f_C1MR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNorm_Inf_8u_C3R_Ctx.
 * 
 * For common parameter descriptions, see \ref CommonNormInfGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiNormInfGetBufferHostSize_8u_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNorm_Inf_16u_C3R_Ctx.
 * 
 * For common parameter descriptions, see \ref CommonNormInfGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiNormInfGetBufferHostSize_16u_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNorm_Inf_16s_C3R_Ctx.
 * 
 * For common parameter descriptions, see \ref CommonNormInfGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiNormInfGetBufferHostSize_16s_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNorm_Inf_32f_C3R_Ctx.
 * 
 * For common parameter descriptions, see \ref CommonNormInfGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiNormInfGetBufferHostSize_32f_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNorm_Inf_8u_AC4R_Ctx.
 * 
 * For common parameter descriptions, see \ref CommonNormInfGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiNormInfGetBufferHostSize_8u_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNorm_Inf_16u_AC4R_Ctx.
 * 
 * For common parameter descriptions, see \ref CommonNormInfGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiNormInfGetBufferHostSize_16u_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNorm_Inf_16s_AC4R_Ctx.
 * 
 * For common parameter descriptions, see \ref CommonNormInfGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiNormInfGetBufferHostSize_16s_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNorm_Inf_32f_AC4R_Ctx.
 * 
 * For common parameter descriptions, see \ref CommonNormInfGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiNormInfGetBufferHostSize_32f_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNorm_Inf_8u_C4R_Ctx.
 * 
 * For common parameter descriptions, see \ref CommonNormInfGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiNormInfGetBufferHostSize_8u_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNorm_Inf_16u_C4R_Ctx.
 * 
 * For common parameter descriptions, see \ref CommonNormInfGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiNormInfGetBufferHostSize_16u_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNorm_Inf_16s_C4R_Ctx.
 * 
 * For common parameter descriptions, see \ref CommonNormInfGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiNormInfGetBufferHostSize_16s_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNorm_Inf_32f_C4R_Ctx.
 * 
 * For common parameter descriptions, see \ref CommonNormInfGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiNormInfGetBufferHostSize_32f_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNorm_Inf_8u_C3CMR_Ctx.
 * 
 * For common parameter descriptions, see \ref CommonNormInfGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiNormInfGetBufferHostSize_8u_C3CMR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNorm_Inf_8s_C3CMR_Ctx.
 * 
 * For common parameter descriptions, see \ref CommonNormInfGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiNormInfGetBufferHostSize_8s_C3CMR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNorm_Inf_16u_C3CMR_Ctx.
 * 
 * For common parameter descriptions, see \ref CommonNormInfGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiNormInfGetBufferHostSize_16u_C3CMR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNorm_Inf_32f_C3CMR_Ctx.
 * 
 * For common parameter descriptions, see \ref CommonNormInfGetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiNormInfGetBufferHostSize_32f_C3CMR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);

/** @} */
/** @} image_inf_norm */


/** 
 * \section image_L1_norm Norm_L1
 * @defgroup image_L1_norm Norm_L1
 * Primitives for computing the L1 norm of an image. 
 * @{
 */
/** @name Basic Norm_L1
 * @{
 */ 
/**
 * One-channel 8-bit unsigned image Norm_L1.
 * 
 * For common parameter descriptions, see \ref CommonNormParameters.
 *
 */
NppStatus nppiNorm_L1_8u_C1R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp64f * pNorm, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 16-bit unsigned image Norm_L1.
 * 
 * For common parameter descriptions, see \ref CommonNormParameters.
 *
 */
NppStatus nppiNorm_L1_16u_C1R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp64f * pNorm, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 16-bit signed image Norm_L1.
 * 
 * For common parameter descriptions, see \ref CommonNormParameters.
 *
 */
NppStatus nppiNorm_L1_16s_C1R_Ctx(const Npp16s * pSrc, int nSrcStep, NppiSize oSizeROI, Npp64f * pNorm, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 32-bit floating point image Norm_L1.
 * 
 * For common parameter descriptions, see \ref CommonNormParameters.
 *
 */
NppStatus nppiNorm_L1_32f_C1R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSizeROI, Npp64f * pNorm, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 8-bit unsigned image Norm_L1.
 * 
 * For common parameter descriptions, see \ref CommonNormParameters.
 *
 */
NppStatus nppiNorm_L1_8u_C3R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp64f aNorm[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 16-bit unsigned image Norm_L1.
 * 
 * For common parameter descriptions, see \ref CommonNormParameters.
 *
 */
NppStatus nppiNorm_L1_16u_C3R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp64f aNorm[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 16-bit signed image Norm_L1.
 * 
 * For common parameter descriptions, see \ref CommonNormParameters.
 *
 */
NppStatus nppiNorm_L1_16s_C3R_Ctx(const Npp16s * pSrc, int nSrcStep, NppiSize oSizeROI, Npp64f aNorm[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 32-bit floating point image Norm_L1.
 * 
 * For common parameter descriptions, see \ref CommonNormParameters.
 *
 */
NppStatus nppiNorm_L1_32f_C3R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSizeROI, Npp64f aNorm[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image Norm_L1 ignoring alpha channel.
 * 
 * For common parameter descriptions, see \ref CommonNormParameters.
 *
 */
NppStatus nppiNorm_L1_8u_AC4R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp64f aNorm[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit unsigned image Norm_L1 ignoring alpha channel.
 * 
 * For common parameter descriptions, see \ref CommonNormParameters.
 *
 */
NppStatus nppiNorm_L1_16u_AC4R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp64f aNorm[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit signed image Norm_L1 ignoring alpha channel.
 * 
 * For common parameter descriptions, see \ref CommonNormParameters.
 *
 */
NppStatus nppiNorm_L1_16s_AC4R_Ctx(const Npp16s * pSrc, int nSrcStep, NppiSize oSizeROI, Npp64f aNorm[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit floating point image Norm_L1 ignoring alpha channel.
 * 
 * For common parameter descriptions, see \ref CommonNormParameters.
 *
 */
NppStatus nppiNorm_L1_32f_AC4R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSizeROI, Npp64f aNorm[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image Norm_L1.
 * 
 * For common parameter descriptions, see \ref CommonNormParameters.
 *
 */
NppStatus nppiNorm_L1_8u_C4R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp64f aNorm[4], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit unsigned image Norm_L1.
 * 
 * For common parameter descriptions, see \ref CommonNormParameters.
 *
 */
NppStatus nppiNorm_L1_16u_C4R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp64f aNorm[4], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit signed image Norm_L1.
 * 
 * For common parameter descriptions, see \ref CommonNormParameters.
 *
 */
NppStatus nppiNorm_L1_16s_C4R_Ctx(const Npp16s * pSrc, int nSrcStep, NppiSize oSizeROI, Npp64f aNorm[4], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit floating point image Norm_L1.
 * 
 * For common parameter descriptions, see \ref CommonNormParameters.
 *
 */
NppStatus nppiNorm_L1_32f_C4R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSizeROI, Npp64f aNorm[4], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Masked one-channel 8-bit unsigned image Norm_L1.
 * 
 * For common parameter descriptions, see \ref CommonNormParameters.
 * 
 */
NppStatus nppiNorm_L1_8u_C1MR_Ctx(const Npp8u * pSrc, int nSrcStep, const Npp8u * pMask, int nMaskStep, NppiSize oSizeROI, Npp64f * pNorm, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Masked one-channel 8-bit signed image Norm_L1.
 * 
 * For common parameter descriptions, see \ref CommonNormParameters.
 * 
 */
NppStatus nppiNorm_L1_8s_C1MR_Ctx(const Npp8s * pSrc, int nSrcStep, const Npp8u * pMask, int nMaskStep, NppiSize oSizeROI, Npp64f * pNorm, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Masked one-channel 16-bit unsigned image Norm_L1.
 * 
 * For common parameter descriptions, see \ref CommonNormParameters.
 * 
 * 
 */
NppStatus nppiNorm_L1_16u_C1MR_Ctx(const Npp16u * pSrc, int nSrcStep, const Npp8u * pMask, int nMaskStep, NppiSize oSizeROI, Npp64f * pNorm, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Masked one-channel 32-bit floating point image Norm_L1.
 * 
 * For common parameter descriptions, see \ref CommonNormParameters.
 * 
 * 
 */
NppStatus nppiNorm_L1_32f_C1MR_Ctx(const Npp32f * pSrc, int nSrcStep, const Npp8u * pMask, int nMaskStep, NppiSize oSizeROI, Npp64f * pNorm, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Masked three-channel 8-bit unsigned image Norm_L1 affecting only single channel.
 *
 * For common parameter descriptions, see \ref CommonNormParameters.
 * 
 */
NppStatus nppiNorm_L1_8u_C3CMR_Ctx(const Npp8u * pSrc, int nSrcStep, const Npp8u * pMask, int nMaskStep, NppiSize oSizeROI, int nCOI, Npp64f * pNorm, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Masked three-channel 8-bit signed image Norm_L1 affecting only single channel.
 *
 * For common parameter descriptions, see \ref CommonNormParameters.
 * 
 */
NppStatus nppiNorm_L1_8s_C3CMR_Ctx(const Npp8s * pSrc, int nSrcStep, const Npp8u * pMask, int nMaskStep, NppiSize oSizeROI, int nCOI, Npp64f * pNorm, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Masked three-channel 16-bit unsigned image Norm_L1 affecting only single channel.
 *
 * For common parameter descriptions, see \ref CommonNormParameters.
 * 
 */
NppStatus nppiNorm_L1_16u_C3CMR_Ctx(const Npp16u * pSrc, int nSrcStep, const Npp8u * pMask, int nMaskStep, NppiSize oSizeROI, int nCOI, Npp64f * pNorm, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Masked three-channel 32-bit floating point image Norm_L1 affecting only single channel.
 *
 * For common parameter descriptions, see \ref CommonNormParameters.
 * 
 */
NppStatus nppiNorm_L1_32f_C3CMR_Ctx(const Npp32f * pSrc, int nSrcStep, const Npp8u * pMask, int nMaskStep, NppiSize oSizeROI, int nCOI, Npp64f * pNorm, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);

/** @} */
 
 /** @name NormL1GetBufferHostSize
 * Companion primitives for computing the device buffer size (in bytes) required by the Norm_L1 primitives.
 * \section CommonNormL1GetBufferHostSizeParameters
 * Common parameters for nppiNormL1GetBufferHostSize functions include:
 * \param oSizeROI \ref roi_specification.
 * \param hpBufferSize Required buffer size. Important: hpBufferSize is a 
 *        <em>host pointer.</em> \ref general_scratch_buffer.
 * \param nppStreamCtx \ref application_managed_stream_context. 
 * \return NPP_NULL_POINTER_ERROR if hpBufferSize is 0 (NULL),  \ref roi_error_codes.  
 * 
 * @{
 */
/** 
 * Buffer size for \ref nppiNorm_L1_8u_C1R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonNormL1GetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiNormL1GetBufferHostSize_8u_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNorm_L1_16u_C1R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonNormL1GetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiNormL1GetBufferHostSize_16u_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNorm_L1_16s_C1R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonNormL1GetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiNormL1GetBufferHostSize_16s_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNorm_L1_32f_C1R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonNormL1GetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiNormL1GetBufferHostSize_32f_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNorm_L1_8u_C1MR_Ctx.
 *
 * For common parameter descriptions, see \ref CommonNormL1GetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiNormL1GetBufferHostSize_8u_C1MR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNorm_L1_8s_C1MR_Ctx.
 *
 * For common parameter descriptions, see \ref CommonNormL1GetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiNormL1GetBufferHostSize_8s_C1MR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNorm_L1_16u_C1MR_Ctx.
 *
 * For common parameter descriptions, see \ref CommonNormL1GetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiNormL1GetBufferHostSize_16u_C1MR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNorm_L1_32f_C1MR_Ctx.
 *
 * For common parameter descriptions, see \ref CommonNormL1GetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiNormL1GetBufferHostSize_32f_C1MR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNorm_L1_8u_C3R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonNormL1GetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiNormL1GetBufferHostSize_8u_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNorm_L1_16u_C3R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonNormL1GetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiNormL1GetBufferHostSize_16u_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNorm_L1_16s_C3R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonNormL1GetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiNormL1GetBufferHostSize_16s_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNorm_L1_32f_C3R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonNormL1GetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiNormL1GetBufferHostSize_32f_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNorm_L1_8u_AC4R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonNormL1GetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiNormL1GetBufferHostSize_8u_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNorm_L1_16u_AC4R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonNormL1GetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiNormL1GetBufferHostSize_16u_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNorm_L1_16s_AC4R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonNormL1GetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiNormL1GetBufferHostSize_16s_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNorm_L1_32f_AC4R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonNormL1GetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiNormL1GetBufferHostSize_32f_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNorm_L1_8u_C4R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonNormL1GetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiNormL1GetBufferHostSize_8u_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNorm_L1_16u_C4R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonNormL1GetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiNormL1GetBufferHostSize_16u_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNorm_L1_16s_C4R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonNormL1GetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiNormL1GetBufferHostSize_16s_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNorm_L1_32f_C4R_Ctx.
 *
 * For common parameter descriptions, see \ref CommonNormL1GetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiNormL1GetBufferHostSize_32f_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNorm_L1_8u_C3CMR_Ctx.
 *
 * For common parameter descriptions, see \ref CommonNormL1GetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiNormL1GetBufferHostSize_8u_C3CMR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNorm_L1_8s_C3CMR_Ctx.
 *
 * For common parameter descriptions, see \ref CommonNormL1GetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiNormL1GetBufferHostSize_8s_C3CMR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNorm_L1_16u_C3CMR_Ctx.
 *
 * For common parameter descriptions, see \ref CommonNormL1GetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiNormL1GetBufferHostSize_16u_C3CMR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNorm_L1_32f_C3CMR_Ctx.
 *
 * For common parameter descriptions, see \ref CommonNormL1GetBufferHostSizeParameters.
 *
 */
NppStatus 
nppiNormL1GetBufferHostSize_32f_C3CMR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);

/** @}  */
/** @} image_L1_norm */

/** 
 * \section image_L2_norm Norm_L2
 * @defgroup image_L2_norm Norm_L2
 * Primitives for computing the L2 norm of an image. 
 * @{
 */
/** @name Basic Norm_L2
 * Computes the L2 norm of an image. 
 * 
 * @{
 */
/**
 * One-channel 8-bit unsigned image Norm_L2.
 *
 * For common parameter descriptions, see \ref CommonNormParameters.
 * 
 */
NppStatus nppiNorm_L2_8u_C1R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp64f * pNorm, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 16-bit unsigned image Norm_L2.
 *
 * For common parameter descriptions, see \ref CommonNormParameters.
 * 
 */
NppStatus nppiNorm_L2_16u_C1R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp64f * pNorm, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 16-bit signed image Norm_L2.
 *
 * For common parameter descriptions, see \ref CommonNormParameters.
 * 
 */
NppStatus nppiNorm_L2_16s_C1R_Ctx(const Npp16s * pSrc, int nSrcStep, NppiSize oSizeROI, Npp64f * pNorm, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 32-bit floating point image Norm_L2.
 *
 * For common parameter descriptions, see \ref CommonNormParameters.
 * 
 */
NppStatus nppiNorm_L2_32f_C1R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSizeROI, Npp64f * pNorm, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 8-bit unsigned image Norm_L2.
 *
 * For common parameter descriptions, see \ref CommonNormParameters.
 * 
 */
NppStatus nppiNorm_L2_8u_C3R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp64f aNorm[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 16-bit unsigned image Norm_L2.
 *
 * For common parameter descriptions, see \ref CommonNormParameters.
 * 
 */
NppStatus nppiNorm_L2_16u_C3R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp64f aNorm[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 16-bit signed image Norm_L2.
 *
 * For common parameter descriptions, see \ref CommonNormParameters.
 * 
 */
NppStatus nppiNorm_L2_16s_C3R_Ctx(const Npp16s * pSrc, int nSrcStep, NppiSize oSizeROI, Npp64f aNorm[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 32-bit floating point image Norm_L2.
 *
 * For common parameter descriptions, see \ref CommonNormParameters.
 * 
 */
NppStatus nppiNorm_L2_32f_C3R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSizeROI, Npp64f aNorm[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image Norm_L2 ignoring alpha channel.
 *
 * For common parameter descriptions, see \ref CommonNormParameters.
 * 
 */
NppStatus nppiNorm_L2_8u_AC4R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp64f aNorm[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit unsigned image Norm_L2 ignoring alpha channel.
 *
 * For common parameter descriptions, see \ref CommonNormParameters.
 * 
 */
NppStatus nppiNorm_L2_16u_AC4R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp64f aNorm[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit signed image Norm_L2 ignoring alpha channel.
 *
 * For common parameter descriptions, see \ref CommonNormParameters.
 * 
 */
NppStatus nppiNorm_L2_16s_AC4R_Ctx(const Npp16s * pSrc, int nSrcStep, NppiSize oSizeROI, Npp64f aNorm[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit floating point image Norm_L2 ignoring alpha channel.
 *
 * For common parameter descriptions, see \ref CommonNormParameters.
 * 
 */
NppStatus nppiNorm_L2_32f_AC4R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSizeROI, Npp64f aNorm[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image Norm_L2.
 *
 * For common parameter descriptions, see \ref CommonNormParameters.
 * 
 */
NppStatus nppiNorm_L2_8u_C4R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp64f aNorm[4], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit unsigned image Norm_L2.
 *
 * For common parameter descriptions, see \ref CommonNormParameters.
 * 
 */
NppStatus nppiNorm_L2_16u_C4R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp64f aNorm[4], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit signed image Norm_L2.
 *
 * For common parameter descriptions, see \ref CommonNormParameters.
 * 
 */
NppStatus nppiNorm_L2_16s_C4R_Ctx(const Npp16s * pSrc, int nSrcStep, NppiSize oSizeROI, Npp64f aNorm[4], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit floating point image Norm_L2.
 *
 * For common parameter descriptions, see \ref CommonNormParameters.
 * 
 */
NppStatus nppiNorm_L2_32f_C4R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSizeROI, Npp64f aNorm[4], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Masked one-channel 8-bit unsigned image Norm_L2.
 * For common parameter descriptions, see \ref CommonNormParameters.  
 */
NppStatus nppiNorm_L2_8u_C1MR_Ctx(const Npp8u * pSrc, int nSrcStep, const Npp8u * pMask, int nMaskStep, NppiSize oSizeROI, Npp64f * pNorm, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Masked one-channel 8-bit signed image Norm_L2.
 * For common parameter descriptions, see \ref CommonNormParameters.  
 */
NppStatus nppiNorm_L2_8s_C1MR_Ctx(const Npp8s * pSrc, int nSrcStep, const Npp8u * pMask, int nMaskStep, NppiSize oSizeROI, Npp64f * pNorm, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Masked one-channel 16-bit unsigned image Norm_L2.
 * For common parameter descriptions, see \ref CommonNormParameters.  
 */
NppStatus nppiNorm_L2_16u_C1MR_Ctx(const Npp16u * pSrc, int nSrcStep, const Npp8u * pMask, int nMaskStep, NppiSize oSizeROI, Npp64f * pNorm, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Masked one-channel 32-bit floating point image Norm_L2.
 * For common parameter descriptions, see \ref CommonNormParameters.  
 */
NppStatus nppiNorm_L2_32f_C1MR_Ctx(const Npp32f * pSrc, int nSrcStep, const Npp8u * pMask, int nMaskStep, NppiSize oSizeROI, Npp64f * pNorm, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
 /**
 * Masked three-channel 8-bit unsigned image Norm_L2.
 */
NppStatus nppiNorm_L2_8u_C3CMR_Ctx(const Npp8u * pSrc, int nSrcStep, const Npp8u * pMask, int nMaskStep, NppiSize oSizeROI, int nCOI, Npp64f * pNorm, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Masked three-channel 8-bit signed image Norm_L2.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNorm_L2_8s_C3CMR_Ctx(const Npp8s * pSrc, int nSrcStep, const Npp8u * pMask, int nMaskStep, NppiSize oSizeROI, int nCOI, Npp64f * pNorm, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Masked three-channel 16-bit unsigned image Norm_L2.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNorm_L2_16u_C3CMR_Ctx(const Npp16u * pSrc, int nSrcStep, const Npp8u * pMask, int nMaskStep, NppiSize oSizeROI, int nCOI, Npp64f * pNorm, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Masked three-channel 32-bit floating point image Norm_L2.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNorm_L2_32f_C3CMR_Ctx(const Npp32f * pSrc, int nSrcStep, const Npp8u * pMask, int nMaskStep, NppiSize oSizeROI, int nCOI, Npp64f * pNorm, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);

/** @} */

/** @name NormL2GetBufferHostSize
 * Companion primitives for computing the device buffer size (in bytes) required by the Norm_L2 primitives.
 * \section CommonNormL2GetBufferHostSizeParameters
 * Common parameters for nppiNormL2GetBufferHostSize functions include:
 * \param oSizeROI \ref roi_specification.
 * \param hpBufferSize Required buffer size. Important: hpBufferSize is a 
 *        <em>host pointer.</em> \ref general_scratch_buffer.
 * \param nppStreamCtx \ref application_managed_stream_context. 
 * \return NPP_NULL_POINTER_ERROR if hpBufferSize is 0 (NULL),  \ref roi_error_codes.  
 * @{
 */
/** 
 * Buffer size for \ref nppiNorm_L2_8u_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonNormL2GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormL2GetBufferHostSize_8u_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNorm_L2_16u_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonNormL2GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormL2GetBufferHostSize_16u_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNorm_L2_16s_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonNormL2GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormL2GetBufferHostSize_16s_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNorm_L2_32f_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonNormL2GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormL2GetBufferHostSize_32f_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNorm_L2_8u_C1MR_Ctx.
 * For common parameter descriptions, see \ref CommonNormL2GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormL2GetBufferHostSize_8u_C1MR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNorm_L2_8s_C1MR_Ctx.
 * For common parameter descriptions, see \ref CommonNormL2GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormL2GetBufferHostSize_8s_C1MR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNorm_L2_16u_C1MR_Ctx.
 * For common parameter descriptions, see \ref CommonNormL2GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormL2GetBufferHostSize_16u_C1MR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNorm_L2_32f_C1MR_Ctx.
 * For common parameter descriptions, see \ref CommonNormL2GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormL2GetBufferHostSize_32f_C1MR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNorm_L2_8u_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonNormL2GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormL2GetBufferHostSize_8u_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNorm_L2_16u_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonNormL2GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormL2GetBufferHostSize_16u_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNorm_L2_16s_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonNormL2GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormL2GetBufferHostSize_16s_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNorm_L2_32f_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonNormL2GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormL2GetBufferHostSize_32f_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNorm_L2_8u_AC4R_Ctx.
 * For common parameter descriptions, see \ref CommonNormL2GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormL2GetBufferHostSize_8u_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNorm_L2_16u_AC4R_Ctx.
 * For common parameter descriptions, see \ref CommonNormL2GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormL2GetBufferHostSize_16u_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNorm_L2_16s_AC4R_Ctx.
 * For common parameter descriptions, see \ref CommonNormL2GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormL2GetBufferHostSize_16s_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNorm_L2_32f_AC4R_Ctx.
 * For common parameter descriptions, see \ref CommonNormL2GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormL2GetBufferHostSize_32f_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNorm_L2_8u_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonNormL2GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormL2GetBufferHostSize_8u_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNorm_L2_16u_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonNormL2GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormL2GetBufferHostSize_16u_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNorm_L2_16s_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonNormL2GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormL2GetBufferHostSize_16s_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNorm_L2_32f_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonNormL2GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormL2GetBufferHostSize_32f_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNorm_L2_8u_C3CMR_Ctx.
 * For common parameter descriptions, see \ref CommonNormL2GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormL2GetBufferHostSize_8u_C3CMR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNorm_L2_8s_C3CMR_Ctx.
 * For common parameter descriptions, see \ref CommonNormL2GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormL2GetBufferHostSize_8s_C3CMR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNorm_L2_16u_C3CMR_Ctx.
 * For common parameter descriptions, see \ref CommonNormL2GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormL2GetBufferHostSize_16u_C3CMR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNorm_L2_32f_C3CMR_Ctx.
 * For common parameter descriptions, see \ref CommonNormL2GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormL2GetBufferHostSize_32f_C3CMR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);

/** @} */
/** @} image_L2_norm */

/** 
 * \section image_inf_normdiff NormDiff_Inf
 * @defgroup image_inf_normdiff NormDiff_Inf
 * Primitives for computing the infinity norm of difference of pixels between two images.
 * @{
 */

/** @name Basic NormDiff_Inf
 * @{
 */
/**
 * One-channel 8-bit unsigned image NormDiff_Inf.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormDiff_Inf_8u_C1R_Ctx(const Npp8u * pSrc1, int nSrc1Step, 
                                      const Npp8u * pSrc2, int nSrc2Step, 
                                      NppiSize oSizeROI, Npp64f * pNormDiff, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 16-bit unsigned image NormDiff_Inf.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormDiff_Inf_16u_C1R_Ctx(const Npp16u * pSrc1, int nSrc1Step, 
                                       const Npp16u * pSrc2, int nSrc2Step, 
                                       NppiSize oSizeROI, Npp64f * pNormDiff, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 16-bit signed image NormDiff_Inf.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormDiff_Inf_16s_C1R_Ctx(const Npp16s * pSrc1, int nSrc1Step, 
                                       const Npp16s * pSrc2, int nSrc2Step, 
                                       NppiSize oSizeROI, Npp64f * pNormDiff, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 32-bit floating point image NormDiff_Inf.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormDiff_Inf_32f_C1R_Ctx(const Npp32f * pSrc1, int nSrc1Step, 
                                       const Npp32f * pSrc2, int nSrc2Step, 
                                       NppiSize oSizeROI, Npp64f * pNormDiff, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 8-bit unsigned image NormDiff_Inf.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormDiff_Inf_8u_C3R_Ctx(const Npp8u * pSrc1, int nSrc1Step, 
                                      const Npp8u * pSrc2, int nSrc2Step, 
                                      NppiSize oSizeROI, Npp64f aNormDiff[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 16-bit unsigned image NormDiff_Inf.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormDiff_Inf_16u_C3R_Ctx(const Npp16u * pSrc1, int nSrc1Step, 
                                       const Npp16u * pSrc2, int nSrc2Step, 
                                       NppiSize oSizeROI, Npp64f aNormDiff[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 16-bit signed image NormDiff_Inf.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormDiff_Inf_16s_C3R_Ctx(const Npp16s * pSrc1, int nSrc1Step, 
                                       const Npp16s * pSrc2, int nSrc2Step, 
                                       NppiSize oSizeROI, Npp64f aNormDiff[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 32-bit floating point image NormDiff_Inf.
 *
 * \param pSrc1 \ref source_image_pointer.
 * \param nSrc1Step \ref source_image_line_step.
 * \param pSrc2 \ref source_image_pointer.
 * \param nSrc2Step \ref source_image_line_step.
 * \param oSizeROI \ref roi_specification.
 * \param aNormDiff Array that contains computed Inf-norm of differences. 
 * \param pDeviceBuffer Pointer to the required device memory allocation, \ref general_scratch_buffer. 
 * \param nppStreamCtx \ref application_managed_stream_context. 
 * \return \ref image_data_error_codes, \ref roi_error_codes, or NPP_NOT_EVEN_STEP_ERROR if 
 * an invalid floating-point image is specified.
 */
NppStatus nppiNormDiff_Inf_32f_C3R_Ctx(const Npp32f * pSrc1, int nSrc1Step, 
                                       const Npp32f * pSrc2, int nSrc2Step, 
                                       NppiSize oSizeROI, Npp64f aNormDiff[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image NormDiff_Inf ignoring alpha channel.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormDiff_Inf_8u_AC4R_Ctx(const Npp8u * pSrc1, int nSrc1Step, 
                                       const Npp8u * pSrc2, int nSrc2Step, 
                                       NppiSize oSizeROI, Npp64f aNormDiff[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit unsigned image NormDiff_Inf ignoring alpha channel.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormDiff_Inf_16u_AC4R_Ctx(const Npp16u * pSrc1, int nSrc1Step, 
                                        const Npp16u * pSrc2, int nSrc2Step, 
                                        NppiSize oSizeROI, Npp64f aNormDiff[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit signed image NormDiff_Inf ignoring alpha channel.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormDiff_Inf_16s_AC4R_Ctx(const Npp16s * pSrc1, int nSrc1Step, 
                                        const Npp16s * pSrc2, int nSrc2Step, 
                                        NppiSize oSizeROI, Npp64f aNormDiff[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit floating point image NormDiff_Inf ignoring alpha channel.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormDiff_Inf_32f_AC4R_Ctx(const Npp32f * pSrc1, int nSrc1Step, 
                                        const Npp32f * pSrc2, int nSrc2Step, 
                                        NppiSize oSizeROI, Npp64f aNormDiff[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image NormDiff_Inf.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormDiff_Inf_8u_C4R_Ctx(const Npp8u * pSrc1, int nSrc1Step, 
                                      const Npp8u * pSrc2, int nSrc2Step, 
                                      NppiSize oSizeROI, Npp64f aNormDiff[4], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit unsigned image NormDiff_Inf.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormDiff_Inf_16u_C4R_Ctx(const Npp16u * pSrc1, int nSrc1Step, 
                                       const Npp16u * pSrc2, int nSrc2Step, 
                                       NppiSize oSizeROI, Npp64f aNormDiff[4], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit signed image NormDiff_Inf.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormDiff_Inf_16s_C4R_Ctx(const Npp16s * pSrc1, int nSrc1Step, 
                                       const Npp16s * pSrc2, int nSrc2Step, 
                                       NppiSize oSizeROI, Npp64f aNormDiff[4], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit floating point image NormDiff_Inf.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormDiff_Inf_32f_C4R_Ctx(const Npp32f * pSrc1, int nSrc1Step, 
                                       const Npp32f * pSrc2, int nSrc2Step, 
                                       NppiSize oSizeROI, Npp64f aNormDiff[4], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Masked one-channel 8-bit unsigned images NormDiff_Inf.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormDiff_Inf_8u_C1MR_Ctx(const Npp8u * pSrc1, int nSrc1Step, 
                                       const Npp8u * pSrc2, int nSrc2Step, 
                                       const Npp8u * pMask, int nMaskStep,
                                       NppiSize oSizeROI, Npp64f * pNormDiff, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Masked one-channel 8-bit signed images NormDiff_Inf.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormDiff_Inf_8s_C1MR_Ctx(const Npp8s * pSrc1, int nSrc1Step, 
                                       const Npp8s * pSrc2, int nSrc2Step, 
                                       const Npp8u * pMask, int nMaskStep,
                                       NppiSize oSizeROI, Npp64f * pNormDiff, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Masked one-channel 16-bit unsigned images NormDiff_Inf.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormDiff_Inf_16u_C1MR_Ctx(const Npp16u * pSrc1, int nSrc1Step, 
                                        const Npp16u * pSrc2, int nSrc2Step, 
                                        const Npp8u  * pMask, int nMaskStep,
                                        NppiSize oSizeROI, Npp64f * pNormDiff, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Masked one-channel 32-bit floating point images NormDiff_Inf.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormDiff_Inf_32f_C1MR_Ctx(const Npp32f * pSrc1, int nSrc1Step, 
                                        const Npp32f * pSrc2, int nSrc2Step, 
                                        const Npp8u  * pMask, int nMaskStep,
                                        NppiSize oSizeROI, Npp64f * pNormDiff, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Masked three-channel 8-bit unsigned image NormDiff_Inf affecting only single channel.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormDiff_Inf_8u_C3CMR_Ctx(const Npp8u * pSrc1, int nSrc1Step, 
                                        const Npp8u * pSrc2, int nSrc2Step, 
                                        const Npp8u * pMask, int nMaskStep,
                                        NppiSize oSizeROI, int nCOI, Npp64f * pNormDiff, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Masked three-channel 8-bit signed image NormDiff_Inf affecting only single channel.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormDiff_Inf_8s_C3CMR_Ctx(const Npp8s * pSrc1, int nSrc1Step, 
                                        const Npp8s * pSrc2, int nSrc2Step, 
                                        const Npp8u * pMask, int nMaskStep,
                                        NppiSize oSizeROI, int nCOI, Npp64f * pNormDiff, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Masked three-channel 16-bit unsigned image NormDiff_Inf affecting only single channel.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormDiff_Inf_16u_C3CMR_Ctx(const Npp16u * pSrc1, int nSrc1Step, 
                                         const Npp16u * pSrc2, int nSrc2Step, 
                                         const Npp8u  * pMask, int nMaskStep,
                                         NppiSize oSizeROI, int nCOI, Npp64f * pNormDiff, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Masked three-channel 32-bit floating point image NormDiff_Inf affecting only single channel.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormDiff_Inf_32f_C3CMR_Ctx(const Npp32f * pSrc1, int nSrc1Step, 
                                         const Npp32f * pSrc2, int nSrc2Step, 
                                         const Npp8u  * pMask, int nMaskStep,
                                         NppiSize oSizeROI, int nCOI, Npp64f * pNormDiff, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);

/** @} */
 
/** @name NormDiffInfGetBufferHostSize
 * Companion primitives for computing the device buffer size (in bytes) required by the NormDiff_Inf primitives.
 * \section CommonNormDiffInfGetBufferHostSizeParameters
 * Common parameters for nppiNormDiffInfGetBufferHostSize functions include:
 * \param oSizeROI \ref roi_specification.
 * \param hpBufferSize Required buffer size. Important: hpBufferSize is a 
 *        <em>host pointer.</em> \ref general_scratch_buffer.
 * \param nppStreamCtx \ref application_managed_stream_context. 
 * \return NPP_NULL_POINTER_ERROR if hpBufferSize is 0 (NULL),  \ref roi_error_codes.  
 * @{
 */
/** 
 * Buffer size for \ref nppiNormDiff_Inf_8u_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonNormDiffInfGetBufferHostSizeParameters.
 */
NppStatus 
nppiNormDiffInfGetBufferHostSize_8u_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNormDiff_Inf_16u_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonNormDiffInfGetBufferHostSizeParameters.
 */
NppStatus 
nppiNormDiffInfGetBufferHostSize_16u_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNormDiff_Inf_16s_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonNormDiffInfGetBufferHostSizeParameters.
 */
NppStatus 
nppiNormDiffInfGetBufferHostSize_16s_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNormDiff_Inf_32f_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonNormDiffInfGetBufferHostSizeParameters.
 */
NppStatus 
nppiNormDiffInfGetBufferHostSize_32f_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNormDiff_Inf_8u_C1MR_Ctx.
 * For common parameter descriptions, see \ref CommonNormDiffInfGetBufferHostSizeParameters.
 */
NppStatus 
nppiNormDiffInfGetBufferHostSize_8u_C1MR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNormDiff_Inf_8s_C1MR_Ctx.
 * For common parameter descriptions, see \ref CommonNormDiffInfGetBufferHostSizeParameters.
 */
NppStatus 
nppiNormDiffInfGetBufferHostSize_8s_C1MR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNormDiff_Inf_16u_C1MR_Ctx.
 * For common parameter descriptions, see \ref CommonNormDiffInfGetBufferHostSizeParameters.
 */
NppStatus 
nppiNormDiffInfGetBufferHostSize_16u_C1MR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNormDiff_Inf_32f_C1MR_Ctx.
 * For common parameter descriptions, see \ref CommonNormDiffInfGetBufferHostSizeParameters.
 */
NppStatus 
nppiNormDiffInfGetBufferHostSize_32f_C1MR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNormDiff_Inf_8u_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonNormDiffInfGetBufferHostSizeParameters.
 */
NppStatus 
nppiNormDiffInfGetBufferHostSize_8u_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNormDiff_Inf_16u_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonNormDiffInfGetBufferHostSizeParameters.
 */
NppStatus 
nppiNormDiffInfGetBufferHostSize_16u_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNormDiff_Inf_16s_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonNormDiffInfGetBufferHostSizeParameters.
 */
NppStatus 
nppiNormDiffInfGetBufferHostSize_16s_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNormDiff_Inf_32f_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonNormDiffInfGetBufferHostSizeParameters.
 */
NppStatus 
nppiNormDiffInfGetBufferHostSize_32f_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNormDiff_Inf_8u_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonNormDiffInfGetBufferHostSizeParameters.
 */
NppStatus 
nppiNormDiffInfGetBufferHostSize_8u_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNormDiff_Inf_16u_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonNormDiffInfGetBufferHostSizeParameters.
 */
NppStatus 
nppiNormDiffInfGetBufferHostSize_16u_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNormDiff_Inf_16s_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonNormDiffInfGetBufferHostSizeParameters.
 */
NppStatus 
nppiNormDiffInfGetBufferHostSize_16s_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNormDiff_Inf_32f_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonNormDiffInfGetBufferHostSizeParameters.
 */
NppStatus 
nppiNormDiffInfGetBufferHostSize_32f_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNormDiff_Inf_8u_AC4R_Ctx.
 * For common parameter descriptions, see \ref CommonNormDiffInfGetBufferHostSizeParameters.
 */
NppStatus 
nppiNormDiffInfGetBufferHostSize_8u_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNormDiff_Inf_16u_AC4R_Ctx.
 * For common parameter descriptions, see \ref CommonNormDiffInfGetBufferHostSizeParameters.
 */
NppStatus 
nppiNormDiffInfGetBufferHostSize_16u_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNormDiff_Inf_16s_AC4R_Ctx.
 * For common parameter descriptions, see \ref CommonNormDiffInfGetBufferHostSizeParameters.
 */
NppStatus 
nppiNormDiffInfGetBufferHostSize_16s_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNormDiff_Inf_32f_AC4R_Ctx.
 * For common parameter descriptions, see \ref CommonNormDiffInfGetBufferHostSizeParameters.
 */
NppStatus 
nppiNormDiffInfGetBufferHostSize_32f_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNormDiff_Inf_8u_C3CMR_Ctx.
 * For common parameter descriptions, see \ref CommonNormDiffInfGetBufferHostSizeParameters.
 */
NppStatus 
nppiNormDiffInfGetBufferHostSize_8u_C3CMR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNormDiff_Inf_8s_C3CMR_Ctx.
 * For common parameter descriptions, see \ref CommonNormDiffInfGetBufferHostSizeParameters.
 */
NppStatus 
nppiNormDiffInfGetBufferHostSize_8s_C3CMR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNormDiff_Inf_16u_C3CMR_Ctx.
 * For common parameter descriptions, see \ref CommonNormDiffInfGetBufferHostSizeParameters.
 */
NppStatus 
nppiNormDiffInfGetBufferHostSize_16u_C3CMR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiNormDiff_Inf_32f_C3CMR_Ctx.
 * For common parameter descriptions, see \ref CommonNormDiffInfGetBufferHostSizeParameters.
 */
NppStatus 
nppiNormDiffInfGetBufferHostSize_32f_C3CMR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);

/** @} */
/** @} image_inf_normdiff */

/** 
 * \section image_L1_normdiff NormDiff_L1
 * @defgroup image_L1_normdiff NormDiff_L1
 * Primitives for computing the L1 norm of difference of pixels between two images.
 * @{
 */

/** @name Basic NormDiff_L1
 * @{
 */
/**
 * One-channel 8-bit unsigned image NormDiff_L1.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormDiff_L1_8u_C1R_Ctx(const Npp8u * pSrc1, int nSrc1Step, 
                                     const Npp8u * pSrc2, int nSrc2Step, 
                                     NppiSize oSizeROI, Npp64f * pNormDiff, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 16-bit unsigned image NormDiff_L1.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormDiff_L1_16u_C1R_Ctx(const Npp16u * pSrc1, int nSrc1Step, 
                                      const Npp16u * pSrc2, int nSrc2Step, 
                                      NppiSize oSizeROI, Npp64f * pNormDiff, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 16-bit signed image NormDiff_L1.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormDiff_L1_16s_C1R_Ctx(const Npp16s * pSrc1, int nSrc1Step, 
                                      const Npp16s * pSrc2, int nSrc2Step, 
                                      NppiSize oSizeROI, Npp64f * pNormDiff, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 32-bit floating point image NormDiff_L1.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormDiff_L1_32f_C1R_Ctx(const Npp32f * pSrc1, int nSrc1Step, 
                                      const Npp32f * pSrc2, int nSrc2Step, 
                                      NppiSize oSizeROI, Npp64f * pNormDiff, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 8-bit unsigned image NormDiff_L1.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormDiff_L1_8u_C3R_Ctx(const Npp8u * pSrc1, int nSrc1Step, 
                                     const Npp8u * pSrc2, int nSrc2Step, 
                                     NppiSize oSizeROI, Npp64f aNormDiff[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 16-bit unsigned image NormDiff_L1.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormDiff_L1_16u_C3R_Ctx(const Npp16u * pSrc1, int nSrc1Step, 
                                      const Npp16u * pSrc2, int nSrc2Step, 
                                      NppiSize oSizeROI, Npp64f aNormDiff[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 16-bit signed image NormDiff_L1.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormDiff_L1_16s_C3R_Ctx(const Npp16s * pSrc1, int nSrc1Step, 
                                      const Npp16s * pSrc2, int nSrc2Step, 
                                      NppiSize oSizeROI, Npp64f aNormDiff[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 32-bit floating point image NormDiff_L1.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormDiff_L1_32f_C3R_Ctx(const Npp32f * pSrc1, int nSrc1Step, 
                                      const Npp32f * pSrc2, int nSrc2Step, 
                                      NppiSize oSizeROI, Npp64f aNormDiff[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image NormDiff_L1 ignoring alpha channel.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormDiff_L1_8u_AC4R_Ctx(const Npp8u * pSrc1, int nSrc1Step, 
                                      const Npp8u * pSrc2, int nSrc2Step, 
                                      NppiSize oSizeROI, Npp64f aNormDiff[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit unsigned image NormDiff_L1 ignoring alpha channel.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormDiff_L1_16u_AC4R_Ctx(const Npp16u * pSrc1, int nSrc1Step, 
                                       const Npp16u * pSrc2, int nSrc2Step, 
                                       NppiSize oSizeROI, Npp64f aNormDiff[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit signed image NormDiff_L1 ignoring alpha channel.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormDiff_L1_16s_AC4R_Ctx(const Npp16s * pSrc1, int nSrc1Step, 
                                       const Npp16s * pSrc2, int nSrc2Step, 
                                       NppiSize oSizeROI, Npp64f aNormDiff[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit floating point image NormDiff_L1 ignoring alpha channel.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormDiff_L1_32f_AC4R_Ctx(const Npp32f * pSrc1, int nSrc1Step, 
                                       const Npp32f * pSrc2, int nSrc2Step, 
                                       NppiSize oSizeROI, Npp64f aNormDiff[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image NormDiff_L1.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormDiff_L1_8u_C4R_Ctx(const Npp8u * pSrc1, int nSrc1Step, 
                                     const Npp8u * pSrc2, int nSrc2Step, 
                                     NppiSize oSizeROI, Npp64f aNormDiff[4], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit unsigned image NormDiff_L1.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormDiff_L1_16u_C4R_Ctx(const Npp16u * pSrc1, int nSrc1Step, 
                                      const Npp16u * pSrc2, int nSrc2Step, 
                                      NppiSize oSizeROI, Npp64f aNormDiff[4], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit signed image NormDiff_L1.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormDiff_L1_16s_C4R_Ctx(const Npp16s * pSrc1, int nSrc1Step, 
                                      const Npp16s * pSrc2, int nSrc2Step, 
                                      NppiSize oSizeROI, Npp64f aNormDiff[4], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit floating point image NormDiff_L1.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormDiff_L1_32f_C4R_Ctx(const Npp32f * pSrc1, int nSrc1Step, 
                                      const Npp32f * pSrc2, int nSrc2Step, 
                                      NppiSize oSizeROI, Npp64f aNormDiff[4], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Masked one-channel 8-bit unsigned image NormDiff_L1.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormDiff_L1_8u_C1MR_Ctx(const Npp8u * pSrc1, int nSrc1Step, 
                                      const Npp8u * pSrc2, int nSrc2Step, 
                                      const Npp8u * pMask, int nMaskStep,
                                      NppiSize oSizeROI, Npp64f * pNormDiff, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Masked one-channel 8-bit signed image NormDiff_L1.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormDiff_L1_8s_C1MR_Ctx(const Npp8s * pSrc1, int nSrc1Step, 
                                      const Npp8s * pSrc2, int nSrc2Step, 
                                      const Npp8u * pMask, int nMaskStep,
                                      NppiSize oSizeROI, Npp64f * pNormDiff, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Masked one-channel 16-bit unsigned image NormDiff_L1.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormDiff_L1_16u_C1MR_Ctx(const Npp16u * pSrc1, int nSrc1Step, 
                                       const Npp16u * pSrc2, int nSrc2Step, 
                                       const Npp8u  * pMask, int nMaskStep,
                                       NppiSize oSizeROI, Npp64f * pNormDiff, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Masked one-channel 32-bit floating point image NormDiff_L1.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormDiff_L1_32f_C1MR_Ctx(const Npp32f * pSrc1, int nSrc1Step, 
                                       const Npp32f * pSrc2, int nSrc2Step, 
                                       const Npp8u  * pMask, int nMaskStep,
                                       NppiSize oSizeROI, Npp64f * pNormDiff, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Masked three-channel 8-bit unsigned image NormDiff_L1 affecting only single channel.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormDiff_L1_8u_C3CMR_Ctx(const Npp8u * pSrc1, int nSrc1Step, 
                                       const Npp8u * pSrc2, int nSrc2Step, 
                                       const Npp8u * pMask, int nMaskStep,
                                       NppiSize oSizeROI, int nCOI, Npp64f * pNormDiff, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Masked three-channel 8-bit signed image NormDiff_L1 affecting only single channel.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormDiff_L1_8s_C3CMR_Ctx(const Npp8s * pSrc1, int nSrc1Step, 
                                       const Npp8s * pSrc2, int nSrc2Step, 
                                       const Npp8u * pMask, int nMaskStep,
                                       NppiSize oSizeROI, int nCOI, Npp64f * pNormDiff, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Masked three-channel 16-bit unsigned image NormDiff_L1 affecting only single channel.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormDiff_L1_16u_C3CMR_Ctx(const Npp16u * pSrc1, int nSrc1Step, 
                                        const Npp16u * pSrc2, int nSrc2Step, 
                                        const Npp8u  * pMask, int nMaskStep,
                                        NppiSize oSizeROI, int nCOI, Npp64f * pNormDiff, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Masked three-channel 32-bit floating point image NormDiff_L1 affecting only single channel.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormDiff_L1_32f_C3CMR_Ctx(const Npp32f * pSrc1, int nSrc1Step, 
                                        const Npp32f * pSrc2, int nSrc2Step, 
                                        const Npp8u  * pMask, int nMaskStep,
                                        NppiSize oSizeROI, int nCOI, Npp64f * pNormDiff, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/** @} */
 
 /** @name NormDiffL1GetBufferHostSize
 * Companion primitives for computing the device buffer size (in bytes) required by the NormDiff_L1 primitives.
 * \section CommonNormDiffL1GetBufferHostSizeParameters
 * Common parameters for nppiNormDiffL1GetBufferHostSize functions include:
 * \param oSizeROI \ref roi_specification.
 * \param hpBufferSize Required buffer size. Important: hpBufferSize is a 
 *        <em>host pointer.</em> \ref general_scratch_buffer.
 * \param nppStreamCtx \ref application_managed_stream_context. 
 * \return NPP_NULL_POINTER_ERROR if hpBufferSize is 0 (NULL),  \ref roi_error_codes.  
 * @{
 */
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormDiff_L1_8u_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonNormDiffL1GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormDiffL1GetBufferHostSize_8u_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormDiff_L1_16u_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonNormDiffL1GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormDiffL1GetBufferHostSize_16u_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormDiff_L1_16s_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonNormDiffL1GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormDiffL1GetBufferHostSize_16s_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormDiff_L1_32f_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonNormDiffL1GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormDiffL1GetBufferHostSize_32f_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormDiff_L1_8u_C1MR_Ctx.
 * For common parameter descriptions, see \ref CommonNormDiffL1GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormDiffL1GetBufferHostSize_8u_C1MR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormDiff_L1_8s_C1MR_Ctx.
 * For common parameter descriptions, see \ref CommonNormDiffL1GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormDiffL1GetBufferHostSize_8s_C1MR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormDiff_L1_16u_C1MR_Ctx.
 * For common parameter descriptions, see \ref CommonNormDiffL1GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormDiffL1GetBufferHostSize_16u_C1MR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormDiff_L1_32f_C1MR_Ctx.
 * For common parameter descriptions, see \ref CommonNormDiffL1GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormDiffL1GetBufferHostSize_32f_C1MR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormDiff_L1_8u_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonNormDiffL1GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormDiffL1GetBufferHostSize_8u_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormDiff_L1_16u_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonNormDiffL1GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormDiffL1GetBufferHostSize_16u_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormDiff_L1_16s_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonNormDiffL1GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormDiffL1GetBufferHostSize_16s_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormDiff_L1_32f_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonNormDiffL1GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormDiffL1GetBufferHostSize_32f_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormDiff_L1_8u_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonNormDiffL1GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormDiffL1GetBufferHostSize_8u_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormDiff_L1_16u_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonNormDiffL1GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormDiffL1GetBufferHostSize_16u_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormDiff_L1_16s_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonNormDiffL1GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormDiffL1GetBufferHostSize_16s_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormDiff_L1_32f_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonNormDiffL1GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormDiffL1GetBufferHostSize_32f_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormDiff_L1_8u_AC4R_Ctx.
 * For common parameter descriptions, see \ref CommonNormDiffL1GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormDiffL1GetBufferHostSize_8u_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormDiff_L1_16u_AC4R_Ctx.
 * For common parameter descriptions, see \ref CommonNormDiffL1GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormDiffL1GetBufferHostSize_16u_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormDiff_L1_16s_AC4R_Ctx.
 * For common parameter descriptions, see \ref CommonNormDiffL1GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormDiffL1GetBufferHostSize_16s_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormDiff_L1_32f_AC4R_Ctx.
 * For common parameter descriptions, see \ref CommonNormDiffL1GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormDiffL1GetBufferHostSize_32f_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormDiff_L1_8u_C3CMR_Ctx.
 * For common parameter descriptions, see \ref CommonNormDiffL1GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormDiffL1GetBufferHostSize_8u_C3CMR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormDiff_L1_8s_C3CMR_Ctx.
 * For common parameter descriptions, see \ref CommonNormDiffL1GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormDiffL1GetBufferHostSize_8s_C3CMR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormDiff_L1_16u_C3CMR_Ctx.
 * For common parameter descriptions, see \ref CommonNormDiffL1GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormDiffL1GetBufferHostSize_16u_C3CMR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormDiff_L1_32f_C3CMR_Ctx.
 * For common parameter descriptions, see \ref CommonNormDiffL1GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormDiffL1GetBufferHostSize_32f_C3CMR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);

/** @} */
/** @} image_L1_normdiff */

/** 
 * \section image_L2_normdiff NormDiff_L2
 * @defgroup image_L2_normdiff NormDiff_L2
 * Primitives for computing the L2 norm of difference of pixels between two images.
 * @{
 */
/** @name Basic NormDiff_L2
 * @{
 */
/**
 * One-channel 8-bit unsigned image NormDiff_L2.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormDiff_L2_8u_C1R_Ctx(const Npp8u * pSrc1, int nSrc1Step, 
                                     const Npp8u * pSrc2, int nSrc2Step, 
                                     NppiSize oSizeROI, Npp64f * pNormDiff, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 16-bit unsigned image NormDiff_L2.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormDiff_L2_16u_C1R_Ctx(const Npp16u * pSrc1, int nSrc1Step, 
                                      const Npp16u * pSrc2, int nSrc2Step, 
                                      NppiSize oSizeROI, Npp64f * pNormDiff, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 16-bit signed image NormDiff_L2.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormDiff_L2_16s_C1R_Ctx(const Npp16s * pSrc1, int nSrc1Step, 
                                      const Npp16s * pSrc2, int nSrc2Step, 
                                      NppiSize oSizeROI, Npp64f * pNormDiff, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 32-bit floating point image NormDiff_L2.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormDiff_L2_32f_C1R_Ctx(const Npp32f * pSrc1, int nSrc1Step, 
                                      const Npp32f * pSrc2, int nSrc2Step, 
                                      NppiSize oSizeROI, Npp64f * pNormDiff, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 8-bit unsigned image NormDiff_L2.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormDiff_L2_8u_C3R_Ctx(const Npp8u * pSrc1, int nSrc1Step, 
                                     const Npp8u * pSrc2, int nSrc2Step, 
                                     NppiSize oSizeROI, Npp64f aNormDiff[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 16-bit unsigned image NormDiff_L2.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormDiff_L2_16u_C3R_Ctx(const Npp16u * pSrc1, int nSrc1Step, 
                                      const Npp16u * pSrc2, int nSrc2Step, 
                                      NppiSize oSizeROI, Npp64f aNormDiff[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 16-bit signed image NormDiff_L2.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormDiff_L2_16s_C3R_Ctx(const Npp16s * pSrc1, int nSrc1Step, 
                                      const Npp16s * pSrc2, int nSrc2Step, 
                                      NppiSize oSizeROI, Npp64f aNormDiff[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 32-bit floating point image NormDiff_L2.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormDiff_L2_32f_C3R_Ctx(const Npp32f * pSrc1, int nSrc1Step, 
                                      const Npp32f * pSrc2, int nSrc2Step, 
                                      NppiSize oSizeROI, Npp64f aNormDiff[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image NormDiff_L2 ignoring alpha channel.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormDiff_L2_8u_AC4R_Ctx(const Npp8u * pSrc1, int nSrc1Step, 
                                      const Npp8u * pSrc2, int nSrc2Step, 
                                      NppiSize oSizeROI, Npp64f aNormDiff[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit unsigned image NormDiff_L2 ignoring alpha channel.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormDiff_L2_16u_AC4R_Ctx(const Npp16u * pSrc1, int nSrc1Step, 
                                       const Npp16u * pSrc2, int nSrc2Step, 
                                       NppiSize oSizeROI, Npp64f aNormDiff[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit signed image NormDiff_L2 ignoring alpha channel.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormDiff_L2_16s_AC4R_Ctx(const Npp16s * pSrc1, int nSrc1Step, 
                                       const Npp16s * pSrc2, int nSrc2Step, 
                                       NppiSize oSizeROI, Npp64f aNormDiff[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit floating point image NormDiff_L2 ignoring alpha channel.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormDiff_L2_32f_AC4R_Ctx(const Npp32f * pSrc1, int nSrc1Step, 
                                       const Npp32f * pSrc2, int nSrc2Step, 
                                       NppiSize oSizeROI, Npp64f aNormDiff[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image NormDiff_L2.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormDiff_L2_8u_C4R_Ctx(const Npp8u * pSrc1, int nSrc1Step, 
                                     const Npp8u * pSrc2, int nSrc2Step, 
                                     NppiSize oSizeROI, Npp64f aNormDiff[4], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit unsigned image NormDiff_L2.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormDiff_L2_16u_C4R_Ctx(const Npp16u * pSrc1, int nSrc1Step, 
                                      const Npp16u * pSrc2, int nSrc2Step, 
                                      NppiSize oSizeROI, Npp64f aNormDiff[4], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit signed image NormDiff_L2.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormDiff_L2_16s_C4R_Ctx(const Npp16s * pSrc1, int nSrc1Step, 
                                      const Npp16s * pSrc2, int nSrc2Step, 
                                      NppiSize oSizeROI, Npp64f aNormDiff[4], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit floating point image NormDiff_L2.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormDiff_L2_32f_C4R_Ctx(const Npp32f * pSrc1, int nSrc1Step, 
                                      const Npp32f * pSrc2, int nSrc2Step, 
                                      NppiSize oSizeROI, Npp64f aNormDiff[4], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Masked one-channel 8-bit unsigned image NormDiff_L2.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormDiff_L2_8u_C1MR_Ctx(const Npp8u * pSrc1, int nSrc1Step, 
                                      const Npp8u * pSrc2, int nSrc2Step, 
                                      const Npp8u * pMask, int nMaskStep,
                                      NppiSize oSizeROI, Npp64f * pNormDiff, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Masked one-channel 8-bit signed image NormDiff_L2.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormDiff_L2_8s_C1MR_Ctx(const Npp8s * pSrc1, int nSrc1Step, 
                                      const Npp8s * pSrc2, int nSrc2Step, 
                                      const Npp8u * pMask, int nMaskStep,
                                      NppiSize oSizeROI, Npp64f * pNormDiff, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Masked one-channel 16-bit unsigned image NormDiff_L2.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormDiff_L2_16u_C1MR_Ctx(const Npp16u * pSrc1, int nSrc1Step, 
                                       const Npp16u * pSrc2, int nSrc2Step, 
                                       const Npp8u  * pMask, int nMaskStep,
                                       NppiSize oSizeROI, Npp64f * pNormDiff, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Masked one-channel 32-bit floating point image NormDiff_L2.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormDiff_L2_32f_C1MR_Ctx(const Npp32f * pSrc1, int nSrc1Step, 
                                       const Npp32f * pSrc2, int nSrc2Step, 
                                       const Npp8u  * pMask, int nMaskStep,
                                       NppiSize oSizeROI, Npp64f * pNormDiff, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Masked three-channel 8-bit unsigned image NormDiff_L2 affecting only single channel.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormDiff_L2_8u_C3CMR_Ctx(const Npp8u * pSrc1, int nSrc1Step, 
                                       const Npp8u * pSrc2, int nSrc2Step, 
                                       const Npp8u * pMask, int nMaskStep,
                                       NppiSize oSizeROI, int nCOI, Npp64f * pNormDiff, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Masked three-channel 8-bit signed image NormDiff_L2 affecting only single channel.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormDiff_L2_8s_C3CMR_Ctx(const Npp8s * pSrc1, int nSrc1Step, 
                                       const Npp8s * pSrc2, int nSrc2Step, 
                                       const Npp8u * pMask, int nMaskStep,
                                       NppiSize oSizeROI, int nCOI, Npp64f * pNormDiff, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Masked three-channel 16-bit unsigned image NormDiff_L2 affecting only single channel.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormDiff_L2_16u_C3CMR_Ctx(const Npp16u * pSrc1, int nSrc1Step, 
                                        const Npp16u * pSrc2, int nSrc2Step, 
                                        const Npp8u  * pMask, int nMaskStep,
                                        NppiSize oSizeROI, int nCOI, Npp64f * pNormDiff, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Masked three-channel 32-bit floating point image NormDiff_L2 affecting only single channel.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormDiff_L2_32f_C3CMR_Ctx(const Npp32f * pSrc1, int nSrc1Step, 
                                        const Npp32f * pSrc2, int nSrc2Step, 
                                        const Npp8u  * pMask, int nMaskStep,
                                        NppiSize oSizeROI, int nCOI, Npp64f * pNormDiff, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);

/** @} */

/** @name NormDiffL2GetBufferHostSize
 * Companion primitives for computing the device buffer size (in bytes) required by the NormDiff_L2 primitives.
 * \section CommonNormDiffL2GetBufferHostSizeParameters
 * Common parameters for nppiNormDiffL2GetBufferHostSize functions include:
 * \param oSizeROI \ref roi_specification.
 * \param hpBufferSize Required buffer size. Important: hpBufferSize is a 
 *        <em>host pointer.</em> \ref general_scratch_buffer.
 * \param nppStreamCtx \ref application_managed_stream_context. 
 * \return NPP_NULL_POINTER_ERROR if hpBufferSize is 0 (NULL),  \ref roi_error_codes.  
 * @{
 */
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormDiff_L2_8u_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonNormDiffL2GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormDiffL2GetBufferHostSize_8u_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormDiff_L2_16u_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonNormDiffL2GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormDiffL2GetBufferHostSize_16u_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormDiff_L2_16s_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonNormDiffL2GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormDiffL2GetBufferHostSize_16s_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormDiff_L2_32f_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonNormDiffL2GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormDiffL2GetBufferHostSize_32f_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormDiff_L2_8u_C1MR_Ctx.
 * For common parameter descriptions, see \ref CommonNormDiffL2GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormDiffL2GetBufferHostSize_8u_C1MR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormDiff_L2_8s_C1MR_Ctx.
 * For common parameter descriptions, see \ref CommonNormDiffL2GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormDiffL2GetBufferHostSize_8s_C1MR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormDiff_L2_16u_C1MR_Ctx.
 * For common parameter descriptions, see \ref CommonNormDiffL2GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormDiffL2GetBufferHostSize_16u_C1MR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormDiff_L2_32f_C1MR_Ctx.
 * For common parameter descriptions, see \ref CommonNormDiffL2GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormDiffL2GetBufferHostSize_32f_C1MR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormDiff_L2_8u_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonNormDiffL2GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormDiffL2GetBufferHostSize_8u_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormDiff_L2_16u_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonNormDiffL2GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormDiffL2GetBufferHostSize_16u_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormDiff_L2_16s_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonNormDiffL2GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormDiffL2GetBufferHostSize_16s_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormDiff_L2_32f_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonNormDiffL2GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormDiffL2GetBufferHostSize_32f_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormDiff_L2_8u_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonNormDiffL2GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormDiffL2GetBufferHostSize_8u_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormDiff_L2_16u_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonNormDiffL2GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormDiffL2GetBufferHostSize_16u_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormDiff_L2_16s_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonNormDiffL2GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormDiffL2GetBufferHostSize_16s_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormDiff_L2_32f_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonNormDiffL2GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormDiffL2GetBufferHostSize_32f_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormDiff_L2_8u_AC4R_Ctx.
 * For common parameter descriptions, see \ref CommonNormDiffL2GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormDiffL2GetBufferHostSize_8u_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormDiff_L2_16u_AC4R_Ctx.
 * For common parameter descriptions, see \ref CommonNormDiffL2GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormDiffL2GetBufferHostSize_16u_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormDiff_L2_16s_AC4R_Ctx.
 * For common parameter descriptions, see \ref CommonNormDiffL2GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormDiffL2GetBufferHostSize_16s_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormDiff_L2_32f_AC4R_Ctx.
 * For common parameter descriptions, see \ref CommonNormDiffL2GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormDiffL2GetBufferHostSize_32f_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormDiff_L2_8u_C3CMR_Ctx.
 * For common parameter descriptions, see \ref CommonNormDiffL2GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormDiffL2GetBufferHostSize_8u_C3CMR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormDiff_L2_8s_C3CMR_Ctx.
 * For common parameter descriptions, see \ref CommonNormDiffL2GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormDiffL2GetBufferHostSize_8s_C3CMR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormDiff_L2_16u_C3CMR_Ctx.
 * For common parameter descriptions, see \ref CommonNormDiffL2GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormDiffL2GetBufferHostSize_16u_C3CMR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormDiff_L2_32f_C3CMR_Ctx.
 * For common parameter descriptions, see \ref CommonNormDiffL2GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormDiffL2GetBufferHostSize_32f_C3CMR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx); 

/** @} */
/** @} image_L2_normdiff */

/** 
 * \section image_inf_normrel NormRel_Inf
 * @defgroup image_inf_normrel NormRel_Inf
 * Primitives for computing the relative error of infinity norm between two images.
 * @{
 */
/** @name Basic NormRel_Inf
 * @{
 */
/**
 * One-channel 8-bit unsigned image NormRel_Inf.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormRel_Inf_8u_C1R_Ctx(const Npp8u * pSrc1, int nSrc1Step, 
                                     const Npp8u * pSrc2, int nSrc2Step, 
                                     NppiSize oSizeROI, Npp64f * pNormRel, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 16-bit unsigned image NormRel_Inf.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormRel_Inf_16u_C1R_Ctx(const Npp16u * pSrc1, int nSrc1Step, 
                                      const Npp16u * pSrc2, int nSrc2Step, 
                                      NppiSize oSizeROI, Npp64f * pNormRel, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 16-bit signed image NormRel_Inf.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormRel_Inf_16s_C1R_Ctx(const Npp16s * pSrc1, int nSrc1Step, 
                                      const Npp16s * pSrc2, int nSrc2Step, 
                                      NppiSize oSizeROI, Npp64f * pNormRel, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 32-bit floating point image NormRel_Inf.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormRel_Inf_32f_C1R_Ctx(const Npp32f * pSrc1, int nSrc1Step, 
                                      const Npp32f * pSrc2, int nSrc2Step, 
                                      NppiSize oSizeROI, Npp64f * pNormRel, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 8-bit unsigned image NormRel_Inf.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormRel_Inf_8u_C3R_Ctx(const Npp8u * pSrc1, int nSrc1Step, 
                                     const Npp8u * pSrc2, int nSrc2Step, 
                                     NppiSize oSizeROI, Npp64f aNormRel[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 16-bit unsigned image NormRel_Inf.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormRel_Inf_16u_C3R_Ctx(const Npp16u * pSrc1, int nSrc1Step, 
                                      const Npp16u * pSrc2, int nSrc2Step, 
                                      NppiSize oSizeROI, Npp64f aNormRel[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 16-bit signed image NormRel_Inf.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormRel_Inf_16s_C3R_Ctx(const Npp16s * pSrc1, int nSrc1Step, 
                                      const Npp16s * pSrc2, int nSrc2Step, 
                                      NppiSize oSizeROI, Npp64f aNormRel[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 32-bit floating point image NormRel_Inf.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormRel_Inf_32f_C3R_Ctx(const Npp32f * pSrc1, int nSrc1Step, 
                                      const Npp32f * pSrc2, int nSrc2Step, 
                                      NppiSize oSizeROI, Npp64f aNormRel[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image NormRel_Inf ignoring alpha channel.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormRel_Inf_8u_AC4R_Ctx(const Npp8u * pSrc1, int nSrc1Step, 
                                      const Npp8u * pSrc2, int nSrc2Step, 
                                      NppiSize oSizeROI, Npp64f aNormRel[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit unsigned image NormRel_Inf ignoring alpha channel.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormRel_Inf_16u_AC4R_Ctx(const Npp16u * pSrc1, int nSrc1Step, 
                                       const Npp16u * pSrc2, int nSrc2Step, 
                                       NppiSize oSizeROI, Npp64f aNormRel[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit signed image NormRel_Inf ignoring alpha channel.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormRel_Inf_16s_AC4R_Ctx(const Npp16s * pSrc1, int nSrc1Step, 
                                       const Npp16s * pSrc2, int nSrc2Step, 
                                       NppiSize oSizeROI, Npp64f aNormRel[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit floating point image NormRel_Inf ignoring alpha channel.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormRel_Inf_32f_AC4R_Ctx(const Npp32f * pSrc1, int nSrc1Step, 
                                       const Npp32f * pSrc2, int nSrc2Step, 
                                       NppiSize oSizeROI, Npp64f aNormRel[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image NormRel_Inf.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormRel_Inf_8u_C4R_Ctx(const Npp8u * pSrc1, int nSrc1Step, 
                                     const Npp8u * pSrc2, int nSrc2Step, 
                                     NppiSize oSizeROI, Npp64f aNormRel[4], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit unsigned image NormRel_Inf.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormRel_Inf_16u_C4R_Ctx(const Npp16u * pSrc1, int nSrc1Step, 
                                      const Npp16u * pSrc2, int nSrc2Step, 
                                      NppiSize oSizeROI, Npp64f aNormRel[4], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit signed image NormRel_Inf.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormRel_Inf_16s_C4R_Ctx(const Npp16s * pSrc1, int nSrc1Step, 
                                      const Npp16s * pSrc2, int nSrc2Step, 
                                      NppiSize oSizeROI, Npp64f aNormRel[4], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit floating point image NormRel_Inf.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormRel_Inf_32f_C4R_Ctx(const Npp32f * pSrc1, int nSrc1Step, 
                                      const Npp32f * pSrc2, int nSrc2Step, 
                                      NppiSize oSizeROI, Npp64f aNormRel[4], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Masked one-channel 8-bit unsigned image NormRel_Inf.
 * For common parameter descriptions, see \ref CommonNormParameters. 
 */
NppStatus nppiNormRel_Inf_8u_C1MR_Ctx(const Npp8u * pSrc1, int nSrc1Step, 
                                      const Npp8u * pSrc2, int nSrc2Step, 
                                      const Npp8u * pMask, int nMaskStep,
                                      NppiSize oSizeROI, Npp64f * pNormRel, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Masked one-channel 8-bit signed image NormRel_Inf.
 * For common parameter descriptions, see \ref CommonNormParameters. 
 */
NppStatus nppiNormRel_Inf_8s_C1MR_Ctx(const Npp8s * pSrc1, int nSrc1Step, 
                                      const Npp8s * pSrc2, int nSrc2Step, 
                                      const Npp8u * pMask, int nMaskStep,
                                      NppiSize oSizeROI, Npp64f * pNormRel, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Masked one-channel 16-bit unsigned image NormRel_Inf.
 * For common parameter descriptions, see \ref CommonNormParameters. 
 */
NppStatus nppiNormRel_Inf_16u_C1MR_Ctx(const Npp16u * pSrc1, int nSrc1Step, 
                                       const Npp16u * pSrc2, int nSrc2Step, 
                                       const Npp8u  * pMask, int nMaskStep,
                                       NppiSize oSizeROI, Npp64f * pNormRel, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Masked one-channel 32-bit floating point image NormRel_Inf.
 * For common parameter descriptions, see \ref CommonNormParameters. 
 */
NppStatus nppiNormRel_Inf_32f_C1MR_Ctx(const Npp32f * pSrc1, int nSrc1Step, 
                                       const Npp32f * pSrc2, int nSrc2Step, 
                                       const Npp8u  * pMask, int nMaskStep,
                                       NppiSize oSizeROI, Npp64f * pNormRel, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Masked three-channel 8-bit unsigned image NormRel_Inf affecting only signle channel.
 * For common parameter descriptions, see \ref CommonNormParameters. 
 */
NppStatus nppiNormRel_Inf_8u_C3CMR_Ctx(const Npp8u * pSrc1, int nSrc1Step, 
                                       const Npp8u * pSrc2, int nSrc2Step, 
                                       const Npp8u * pMask, int nMaskStep,
                                       NppiSize oSizeROI, int nCOI, Npp64f * pNormRel, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Masked three-channel 8-bit signed image NormRel_Inf affecting only signle channel.
 * For common parameter descriptions, see \ref CommonNormParameters. 
 */
NppStatus nppiNormRel_Inf_8s_C3CMR_Ctx(const Npp8s * pSrc1, int nSrc1Step, 
                                       const Npp8s * pSrc2, int nSrc2Step, 
                                       const Npp8u * pMask, int nMaskStep,
                                       NppiSize oSizeROI, int nCOI, Npp64f * pNormRel, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Masked three-channel 16-bit unsigned image NormRel_Inf affecting only signle channel.
 * For common parameter descriptions, see \ref CommonNormParameters. 
 */
NppStatus nppiNormRel_Inf_16u_C3CMR_Ctx(const Npp16u * pSrc1, int nSrc1Step, 
                                        const Npp16u * pSrc2, int nSrc2Step, 
                                        const Npp8u  * pMask, int nMaskStep,
                                        NppiSize oSizeROI, int nCOI, Npp64f * pNormRel, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Masked three-channel 32-bit floating point image NormRel_Inf affecting only signle channel.
 * For common parameter descriptions, see \ref CommonNormParameters. 
 */
NppStatus nppiNormRel_Inf_32f_C3CMR_Ctx(const Npp32f * pSrc1, int nSrc1Step, 
                                        const Npp32f * pSrc2, int nSrc2Step, 
                                        const Npp8u  * pMask, int nMaskStep,
                                        NppiSize oSizeROI, int nCOI, Npp64f * pNormRel, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);

/** @} */

/** @name NormRelInfGetBufferHostSize
 * Companion primitives for computing the device buffer size (in bytes) required by the NormRel_Inf primitives.
 * \section CommonNormRelInfGetBufferHostSizeParameters
 * Common parameters for nppiNormRelInfGetBufferHostSize functions include:
 * \param oSizeROI \ref roi_specification.
 * \param hpBufferSize Required buffer size. Important: hpBufferSize is a 
 *        <em>host pointer.</em> \ref general_scratch_buffer.
 * \param nppStreamCtx \ref application_managed_stream_context. 
 * \return NPP_NULL_POINTER_ERROR if hpBufferSize is 0 (NULL),  \ref roi_error_codes.  
 * @{
 */
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormRel_Inf_8u_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonNormRelInfGetBufferHostSizeParameters.
 */
NppStatus 
nppiNormRelInfGetBufferHostSize_8u_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormRel_Inf_16u_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonNormRelInfGetBufferHostSizeParameters.
 */
NppStatus 
nppiNormRelInfGetBufferHostSize_16u_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormRel_Inf_16s_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonNormRelInfGetBufferHostSizeParameters.
 */
NppStatus 
nppiNormRelInfGetBufferHostSize_16s_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormRel_Inf_32f_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonNormRelInfGetBufferHostSizeParameters.
 */
NppStatus 
nppiNormRelInfGetBufferHostSize_32f_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormRel_Inf_8u_C1MR_Ctx.
 * For common parameter descriptions, see \ref CommonNormRelInfGetBufferHostSizeParameters.
 */
NppStatus 
nppiNormRelInfGetBufferHostSize_8u_C1MR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormRel_Inf_8s_C1MR_Ctx.
 * For common parameter descriptions, see \ref CommonNormRelInfGetBufferHostSizeParameters.
 */
NppStatus 
nppiNormRelInfGetBufferHostSize_8s_C1MR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormRel_Inf_16u_C1MR_Ctx.
 * For common parameter descriptions, see \ref CommonNormRelInfGetBufferHostSizeParameters.
 */
NppStatus 
nppiNormRelInfGetBufferHostSize_16u_C1MR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormRel_Inf_32f_C1MR_Ctx.
 * For common parameter descriptions, see \ref CommonNormRelInfGetBufferHostSizeParameters.
 */
NppStatus 
nppiNormRelInfGetBufferHostSize_32f_C1MR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormRel_Inf_8u_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonNormRelInfGetBufferHostSizeParameters.
 */
NppStatus 
nppiNormRelInfGetBufferHostSize_8u_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormRel_Inf_16u_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonNormRelInfGetBufferHostSizeParameters.
 */
NppStatus 
nppiNormRelInfGetBufferHostSize_16u_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormRel_Inf_16s_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonNormRelInfGetBufferHostSizeParameters.
 */
NppStatus 
nppiNormRelInfGetBufferHostSize_16s_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormRel_Inf_32f_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonNormRelInfGetBufferHostSizeParameters.
 */
NppStatus 
nppiNormRelInfGetBufferHostSize_32f_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormRel_Inf_8u_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonNormRelInfGetBufferHostSizeParameters.
 */
NppStatus 
nppiNormRelInfGetBufferHostSize_8u_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormRel_Inf_16u_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonNormRelInfGetBufferHostSizeParameters.
 */
NppStatus 
nppiNormRelInfGetBufferHostSize_16u_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormRel_Inf_16s_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonNormRelInfGetBufferHostSizeParameters.
 */
NppStatus 
nppiNormRelInfGetBufferHostSize_16s_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormRel_Inf_32f_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonNormRelInfGetBufferHostSizeParameters.
 */
NppStatus 
nppiNormRelInfGetBufferHostSize_32f_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormRel_Inf_8u_AC4R_Ctx.
 * For common parameter descriptions, see \ref CommonNormRelInfGetBufferHostSizeParameters.
 */
NppStatus 
nppiNormRelInfGetBufferHostSize_8u_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormRel_Inf_16u_AC4R_Ctx.
 * For common parameter descriptions, see \ref CommonNormRelInfGetBufferHostSizeParameters.
 */
NppStatus 
nppiNormRelInfGetBufferHostSize_16u_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormRel_Inf_16s_AC4R_Ctx.
 * For common parameter descriptions, see \ref CommonNormRelInfGetBufferHostSizeParameters.
 */
NppStatus 
nppiNormRelInfGetBufferHostSize_16s_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormRel_Inf_32f_AC4R_Ctx.
 * For common parameter descriptions, see \ref CommonNormRelInfGetBufferHostSizeParameters.
 */
NppStatus 
nppiNormRelInfGetBufferHostSize_32f_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormRel_Inf_8u_C3CMR_Ctx.
 * For common parameter descriptions, see \ref CommonNormRelInfGetBufferHostSizeParameters.
 */
NppStatus 
nppiNormRelInfGetBufferHostSize_8u_C3CMR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormRel_Inf_8s_C3CMR_Ctx.
 * For common parameter descriptions, see \ref CommonNormRelInfGetBufferHostSizeParameters.
 */
NppStatus 
nppiNormRelInfGetBufferHostSize_8s_C3CMR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormRel_Inf_16u_C3CMR_Ctx.
 * For common parameter descriptions, see \ref CommonNormRelInfGetBufferHostSizeParameters.
 */
NppStatus 
nppiNormRelInfGetBufferHostSize_16u_C3CMR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormRel_Inf_32f_C3CMR_Ctx.
 * For common parameter descriptions, see \ref CommonNormRelInfGetBufferHostSizeParameters.
 */
NppStatus 
nppiNormRelInfGetBufferHostSize_32f_C3CMR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);

/** @} */

/** @} image_inf_normrel */

/** 
 * \section image_L1_normrel NormRel_L1
 * @defgroup image_L1_normrel NormRel_L1
 * Primitives for computing the relative error of L1 norm between two images.
 * @{
 */

/** @name Basic NormRel_L1
 * @{
 */
/**
 * One-channel 8-bit unsigned image NormRel_L1.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormRel_L1_8u_C1R_Ctx(const Npp8u * pSrc1, int nSrc1Step, 
                                    const Npp8u * pSrc2, int nSrc2Step, 
                                    NppiSize oSizeROI, Npp64f * pNormRel, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 16-bit unsigned image NormRel_L1.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormRel_L1_16u_C1R_Ctx(const Npp16u * pSrc1, int nSrc1Step, 
                                     const Npp16u * pSrc2, int nSrc2Step, 
                                     NppiSize oSizeROI, Npp64f * pNormRel, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 16-bit signed image NormRel_L1.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormRel_L1_16s_C1R_Ctx(const Npp16s * pSrc1, int nSrc1Step, 
                                     const Npp16s * pSrc2, int nSrc2Step, 
                                     NppiSize oSizeROI, Npp64f * pNormRel, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 32-bit floating point image NormRel_L1.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormRel_L1_32f_C1R_Ctx(const Npp32f * pSrc1, int nSrc1Step, 
                                     const Npp32f * pSrc2, int nSrc2Step, 
                                     NppiSize oSizeROI, Npp64f * pNormRel, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 8-bit unsigned image NormRel_L1.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormRel_L1_8u_C3R_Ctx(const Npp8u * pSrc1, int nSrc1Step, 
                                    const Npp8u * pSrc2, int nSrc2Step, 
                                    NppiSize oSizeROI, Npp64f aNormRel[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 16-bit unsigned image NormRel_L1.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormRel_L1_16u_C3R_Ctx(const Npp16u * pSrc1, int nSrc1Step, 
                                     const Npp16u * pSrc2, int nSrc2Step, 
                                     NppiSize oSizeROI, Npp64f aNormRel[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 16-bit signed image NormRel_L1.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormRel_L1_16s_C3R_Ctx(const Npp16s * pSrc1, int nSrc1Step, 
                                     const Npp16s * pSrc2, int nSrc2Step, 
                                     NppiSize oSizeROI, Npp64f aNormRel[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 32-bit floating point image NormRel_L1.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormRel_L1_32f_C3R_Ctx(const Npp32f * pSrc1, int nSrc1Step, 
                                     const Npp32f * pSrc2, int nSrc2Step, 
                                     NppiSize oSizeROI, Npp64f aNormRel[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit signed image NormRel_L1 ignoring alpha channel.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormRel_L1_8u_AC4R_Ctx(const Npp8u * pSrc1, int nSrc1Step, 
                                     const Npp8u * pSrc2, int nSrc2Step, 
                                     NppiSize oSizeROI, Npp64f aNormRel[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit unsigned image NormRel_L1 ignoring alpha channel.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormRel_L1_16u_AC4R_Ctx(const Npp16u * pSrc1, int nSrc1Step, 
                                      const Npp16u * pSrc2, int nSrc2Step, 
                                      NppiSize oSizeROI, Npp64f aNormRel[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit signed image NormRel_L1 ignoring alpha channel.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormRel_L1_16s_AC4R_Ctx(const Npp16s * pSrc1, int nSrc1Step, 
                                      const Npp16s * pSrc2, int nSrc2Step, 
                                      NppiSize oSizeROI, Npp64f aNormRel[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit floating point image NormRel_L1 ignoring alpha channel.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormRel_L1_32f_AC4R_Ctx(const Npp32f * pSrc1, int nSrc1Step, 
                                      const Npp32f * pSrc2, int nSrc2Step, 
                                      NppiSize oSizeROI, Npp64f aNormRel[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image NormRel_L1.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormRel_L1_8u_C4R_Ctx(const Npp8u * pSrc1, int nSrc1Step, 
                                    const Npp8u * pSrc2, int nSrc2Step, 
                                    NppiSize oSizeROI, Npp64f aNormRel[4], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit unsigned image NormRel_L1.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormRel_L1_16u_C4R_Ctx(const Npp16u * pSrc1, int nSrc1Step, 
                                     const Npp16u * pSrc2, int nSrc2Step, 
                                     NppiSize oSizeROI, Npp64f aNormRel[4], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit signed image NormRel_L1.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormRel_L1_16s_C4R_Ctx(const Npp16s * pSrc1, int nSrc1Step, 
                                     const Npp16s * pSrc2, int nSrc2Step, 
                                     NppiSize oSizeROI, Npp64f aNormRel[4], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit floating point image NormRel_L1.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormRel_L1_32f_C4R_Ctx(const Npp32f * pSrc1, int nSrc1Step, 
                                     const Npp32f * pSrc2, int nSrc2Step, 
                                     NppiSize oSizeROI, Npp64f aNormRel[4], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 8-bit unsigned image NormRel_L1.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormRel_L1_8u_C1MR_Ctx(const Npp8u * pSrc1, int nSrc1Step, 
                                     const Npp8u * pSrc2, int nSrc2Step, 
                                     const Npp8u * pMask, int nMaskStep,
                                     NppiSize oSizeROI, Npp64f * pNormRel, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 8-bit signed image NormRel_L1.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormRel_L1_8s_C1MR_Ctx(const Npp8s * pSrc1, int nSrc1Step, 
                                     const Npp8s * pSrc2, int nSrc2Step, 
                                     const Npp8u * pMask, int nMaskStep,
                                     NppiSize oSizeROI, Npp64f * pNormRel, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 16-bit unsigned image NormRel_L1.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormRel_L1_16u_C1MR_Ctx(const Npp16u * pSrc1, int nSrc1Step, 
                                      const Npp16u * pSrc2, int nSrc2Step, 
                                      const Npp8u  * pMask, int nMaskStep,
                                      NppiSize oSizeROI, Npp64f * pNormRel, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 32-bit floating point image NormRel_L1.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormRel_L1_32f_C1MR_Ctx(const Npp32f * pSrc1, int nSrc1Step, 
                                      const Npp32f * pSrc2, int nSrc2Step, 
                                      const Npp8u  * pMask, int nMaskStep,
                                      NppiSize oSizeROI, Npp64f * pNormRel, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Masked three-channel 8-bit unsigned image NormRel_L1 affecting only single channel.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormRel_L1_8u_C3CMR_Ctx(const Npp8u * pSrc1, int nSrc1Step, 
                                      const Npp8u * pSrc2, int nSrc2Step, 
                                      const Npp8u * pMask, int nMaskStep,
                                      NppiSize oSizeROI, int nCOI, Npp64f * pNormRel, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Masked three-channel 8-bit signed image NormRel_L1 affecting only single channel.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormRel_L1_8s_C3CMR_Ctx(const Npp8s * pSrc1, int nSrc1Step, 
                                      const Npp8s * pSrc2, int nSrc2Step, 
                                      const Npp8u * pMask, int nMaskStep,
                                      NppiSize oSizeROI, int nCOI, Npp64f * pNormRel, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Masked three-channel 16-bit unsigned image NormRel_L1 affecting only single channel.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormRel_L1_16u_C3CMR_Ctx(const Npp16u * pSrc1, int nSrc1Step, 
                                       const Npp16u * pSrc2, int nSrc2Step, 
                                       const Npp8u  * pMask, int nMaskStep,
                                       NppiSize oSizeROI, int nCOI, Npp64f * pNormRel, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Masked three-channel 32-bit floating point image NormRel_L1 affecting only single channel.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormRel_L1_32f_C3CMR_Ctx(const Npp32f * pSrc1, int nSrc1Step, 
                                       const Npp32f * pSrc2, int nSrc2Step, 
                                       const Npp8u  * pMask, int nMaskStep,
                                       NppiSize oSizeROI, int nCOI, Npp64f * pNormRel, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);

/** @} */

/** @name NormRelL1GetBufferHostSize
 * Companion primitives for computing the device buffer size (in bytes) required by the NormRel_L1 primitives.
 * \section CommonNormRelL1GetBufferHostSizeParameters
 * Common parameters for nppiNormRelL1GetBufferHostSize functions include:
 * \param oSizeROI \ref roi_specification.
 * \param hpBufferSize Required buffer size. Important: hpBufferSize is a 
 *        <em>host pointer.</em> \ref general_scratch_buffer.
 * \param nppStreamCtx \ref application_managed_stream_context. 
 * \return NPP_NULL_POINTER_ERROR if hpBufferSize is 0 (NULL),  \ref roi_error_codes.  
 * @{
 */
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormRel_L1_8u_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonNormRelL1GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormRelL1GetBufferHostSize_8u_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormRel_L1_16u_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonNormRelL1GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormRelL1GetBufferHostSize_16u_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormRel_L1_16s_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonNormRelL1GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormRelL1GetBufferHostSize_16s_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormRel_L1_32f_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonNormRelL1GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormRelL1GetBufferHostSize_32f_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormRel_L1_8u_C1MR_Ctx.
 * For common parameter descriptions, see \ref CommonNormRelL1GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormRelL1GetBufferHostSize_8u_C1MR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormRel_L1_8s_C1MR_Ctx.
 * For common parameter descriptions, see \ref CommonNormRelL1GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormRelL1GetBufferHostSize_8s_C1MR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormRel_L1_16u_C1MR_Ctx.
 * For common parameter descriptions, see \ref CommonNormRelL1GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormRelL1GetBufferHostSize_16u_C1MR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormRel_L1_32f_C1MR_Ctx.
 * For common parameter descriptions, see \ref CommonNormRelL1GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormRelL1GetBufferHostSize_32f_C1MR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormRel_L1_8u_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonNormRelL1GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormRelL1GetBufferHostSize_8u_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormRel_L1_16u_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonNormRelL1GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormRelL1GetBufferHostSize_16u_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormRel_L1_16s_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonNormRelL1GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormRelL1GetBufferHostSize_16s_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormRel_L1_32f_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonNormRelL1GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormRelL1GetBufferHostSize_32f_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormRel_L1_8u_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonNormRelL1GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormRelL1GetBufferHostSize_8u_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormRel_L1_16u_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonNormRelL1GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormRelL1GetBufferHostSize_16u_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormRel_L1_16s_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonNormRelL1GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormRelL1GetBufferHostSize_16s_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormRel_L1_32f_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonNormRelL1GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormRelL1GetBufferHostSize_32f_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormRel_L1_8u_AC4R_Ctx.
 * For common parameter descriptions, see \ref CommonNormRelL1GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormRelL1GetBufferHostSize_8u_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormRel_L1_16u_AC4R_Ctx.
 * For common parameter descriptions, see \ref CommonNormRelL1GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormRelL1GetBufferHostSize_16u_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormRel_L1_16s_AC4R_Ctx.
 * For common parameter descriptions, see \ref CommonNormRelL1GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormRelL1GetBufferHostSize_16s_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormRel_L1_32f_AC4R_Ctx.
 * For common parameter descriptions, see \ref CommonNormRelL1GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormRelL1GetBufferHostSize_32f_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormRel_L1_8u_C3CMR_Ctx.
 * For common parameter descriptions, see \ref CommonNormRelL1GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormRelL1GetBufferHostSize_8u_C3CMR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormRel_L1_8s_C3CMR_Ctx.
 * For common parameter descriptions, see \ref CommonNormRelL1GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormRelL1GetBufferHostSize_8s_C3CMR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormRel_L1_16u_C3CMR_Ctx.
 * For common parameter descriptions, see \ref CommonNormRelL1GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormRelL1GetBufferHostSize_16u_C3CMR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormRel_L1_32f_C3CMR_Ctx.
 * For common parameter descriptions, see \ref CommonNormRelL1GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormRelL1GetBufferHostSize_32f_C3CMR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);

/** @} */

/** @} image_L1_normrel */

/** 
 * \section image_L2_normrel NormRel_L2
 * @defgroup image_L2_normrel NormRel_L2
 * Primitives for computing the relative error of L2 norm between two images.
 * @{
 */

/** @name Basic NormRel_L2
 * @{
 */
/**
 * One-channel 8-bit unsigned image NormRel_L2.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormRel_L2_8u_C1R_Ctx(const Npp8u * pSrc1, int nSrc1Step, 
                                    const Npp8u * pSrc2, int nSrc2Step, 
                                    NppiSize oSizeROI, Npp64f * pNormRel, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 16-bit unsigned image NormRel_L2.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormRel_L2_16u_C1R_Ctx(const Npp16u * pSrc1, int nSrc1Step, 
                                     const Npp16u * pSrc2, int nSrc2Step, 
                                     NppiSize oSizeROI, Npp64f * pNormRel, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 16-bit signed image NormRel_L2.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormRel_L2_16s_C1R_Ctx(const Npp16s * pSrc1, int nSrc1Step, 
                                     const Npp16s * pSrc2, int nSrc2Step, 
                                     NppiSize oSizeROI, Npp64f * pNormRel, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 32-bit floating point image NormRel_L2.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormRel_L2_32f_C1R_Ctx(const Npp32f * pSrc1, int nSrc1Step, 
                                     const Npp32f * pSrc2, int nSrc2Step, 
                                     NppiSize oSizeROI, Npp64f * pNormRel, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 8-bit unsigned image NormRel_L2.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormRel_L2_8u_C3R_Ctx(const Npp8u * pSrc1, int nSrc1Step, 
                                    const Npp8u * pSrc2, int nSrc2Step, 
                                    NppiSize oSizeROI, Npp64f aNormRel[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 16-bit unsigned image NormRel_L2.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormRel_L2_16u_C3R_Ctx(const Npp16u * pSrc1, int nSrc1Step, 
                                     const Npp16u * pSrc2, int nSrc2Step, 
                                     NppiSize oSizeROI, Npp64f aNormRel[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 16-bit signed image NormRel_L2.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormRel_L2_16s_C3R_Ctx(const Npp16s * pSrc1, int nSrc1Step, 
                                     const Npp16s * pSrc2, int nSrc2Step, 
                                     NppiSize oSizeROI, Npp64f aNormRel[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 32-bit floating point image NormRel_L2.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormRel_L2_32f_C3R_Ctx(const Npp32f * pSrc1, int nSrc1Step, 
                                     const Npp32f * pSrc2, int nSrc2Step, 
                                     NppiSize oSizeROI, Npp64f aNormRel[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image NormRel_L2 ignoring alpha channel.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormRel_L2_8u_AC4R_Ctx(const Npp8u * pSrc1, int nSrc1Step, 
                                     const Npp8u * pSrc2, int nSrc2Step, 
                                     NppiSize oSizeROI, Npp64f aNormRel[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit unsigned image NormRel_L2 ignoring alpha channel.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormRel_L2_16u_AC4R_Ctx(const Npp16u * pSrc1, int nSrc1Step, 
                                      const Npp16u * pSrc2, int nSrc2Step, 
                                      NppiSize oSizeROI, Npp64f aNormRel[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit signed image NormRel_L2 ignoring alpha channel.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormRel_L2_16s_AC4R_Ctx(const Npp16s * pSrc1, int nSrc1Step, 
                                      const Npp16s * pSrc2, int nSrc2Step, 
                                      NppiSize oSizeROI, Npp64f aNormRel[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit floating point image NormRel_L2 ignoring alpha channel.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormRel_L2_32f_AC4R_Ctx(const Npp32f * pSrc1, int nSrc1Step, 
                                      const Npp32f * pSrc2, int nSrc2Step, 
                                      NppiSize oSizeROI, Npp64f aNormRel[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image NormRel_L2.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormRel_L2_8u_C4R_Ctx(const Npp8u * pSrc1, int nSrc1Step, 
                                    const Npp8u * pSrc2, int nSrc2Step, 
                                    NppiSize oSizeROI, Npp64f aNormRel[4], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit unsigned image NormRel_L2.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormRel_L2_16u_C4R_Ctx(const Npp16u * pSrc1, int nSrc1Step, 
                                     const Npp16u * pSrc2, int nSrc2Step, 
                                     NppiSize oSizeROI, Npp64f aNormRel[4], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit signed image NormRel_L2.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormRel_L2_16s_C4R_Ctx(const Npp16s * pSrc1, int nSrc1Step, 
                                     const Npp16s * pSrc2, int nSrc2Step, 
                                     NppiSize oSizeROI, Npp64f aNormRel[4], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit floating point image NormRel_L2.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormRel_L2_32f_C4R_Ctx(const Npp32f * pSrc1, int nSrc1Step, 
                                     const Npp32f * pSrc2, int nSrc2Step, 
                                     NppiSize oSizeROI, Npp64f aNormRel[4], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Masked one-channel 8-bit unsigned image NormRel_L2.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormRel_L2_8u_C1MR_Ctx(const Npp8u * pSrc1, int nSrc1Step, 
                                     const Npp8u * pSrc2, int nSrc2Step, 
                                     const Npp8u * pMask, int nMaskStep,
                                     NppiSize oSizeROI, Npp64f * pNormRel, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Masked one-channel 8-bit signed image NormRel_L2.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormRel_L2_8s_C1MR_Ctx(const Npp8s * pSrc1, int nSrc1Step, 
                                     const Npp8s * pSrc2, int nSrc2Step, 
                                     const Npp8u * pMask, int nMaskStep,
                                     NppiSize oSizeROI, Npp64f * pNormRel, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Masked one-channel 16-bit unsigned image NormRel_L2.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormRel_L2_16u_C1MR_Ctx(const Npp16u * pSrc1, int nSrc1Step, 
                                      const Npp16u * pSrc2, int nSrc2Step, 
                                      const Npp8u  * pMask, int nMaskStep,
                                      NppiSize oSizeROI, Npp64f * pNormRel, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Masked one-channel 32-bit floating point image NormRel_L2.
 * For common parameter descriptions, see \ref CommonNormParameters .
 */
NppStatus nppiNormRel_L2_32f_C1MR_Ctx(const Npp32f * pSrc1, int nSrc1Step, 
                                      const Npp32f * pSrc2, int nSrc2Step, 
                                      const Npp8u  * pMask, int nMaskStep,
                                      NppiSize oSizeROI, Npp64f * pNormRel, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx); 
 
/**
 * Masked three-channel 8-bit unsigned image NormRel_L2 affecting only single channel.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormRel_L2_8u_C3CMR_Ctx(const Npp8u * pSrc1, int nSrc1Step, 
                                      const Npp8u * pSrc2, int nSrc2Step, 
                                      const Npp8u * pMask, int nMaskStep,
                                      NppiSize oSizeROI, int nCOI, Npp64f * pNormRel, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Masked three-channel 8-bit signed image NormRel_L2 affecting only single channel.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormRel_L2_8s_C3CMR_Ctx(const Npp8s * pSrc1, int nSrc1Step, 
                                      const Npp8s * pSrc2, int nSrc2Step, 
                                      const Npp8u * pMask, int nMaskStep,
                                      NppiSize oSizeROI, int nCOI, Npp64f * pNormRel, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Masked three-channel 16-bit unsigned image NormRel_L2 affecting only single channel.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormRel_L2_16u_C3CMR_Ctx(const Npp16u * pSrc1, int nSrc1Step, 
                                       const Npp16u * pSrc2, int nSrc2Step, 
                                       const Npp8u  * pMask, int nMaskStep,
                                       NppiSize oSizeROI, int nCOI, Npp64f * pNormRel, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Masked three-channel 32-bit floating point image NormRel_L2 affecting only single channel.
 * For common parameter descriptions, see \ref CommonNormParameters.
 */
NppStatus nppiNormRel_L2_32f_C3CMR_Ctx(const Npp32f * pSrc1, int nSrc1Step, 
                                       const Npp32f * pSrc2, int nSrc2Step, 
                                       const Npp8u  * pMask, int nMaskStep,
                                       NppiSize oSizeROI, int nCOI, Npp64f * pNormRel, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);

 /** @} */

/** @name NormRelL2GetBufferHostSize
 * Companion primitives for computing the device buffer size (in bytes) required by the NormRel_L2 primitives.
 * \section CommonNormRelL2GetBufferHostSizeParameters
 * Common parameters for nppiNormRelL2GetBufferHostSize functions include:
 * \param oSizeROI \ref roi_specification.
 * \param hpBufferSize Required buffer size. Important: hpBufferSize is a 
 *        <em>host pointer.</em> \ref general_scratch_buffer.
 * \param nppStreamCtx \ref application_managed_stream_context. 
 * \return NPP_NULL_POINTER_ERROR if hpBufferSize is 0 (NULL),  \ref roi_error_codes.  
 * @{
 */
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormRel_L2_8u_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonNormRelL2GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormRelL2GetBufferHostSize_8u_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormRel_L2_16u_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonNormRelL2GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormRelL2GetBufferHostSize_16u_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormRel_L2_16s_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonNormRelL2GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormRelL2GetBufferHostSize_16s_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormRel_L2_32f_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonNormRelL2GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormRelL2GetBufferHostSize_32f_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormRel_L2_8u_C1MR_Ctx.
 * For common parameter descriptions, see \ref CommonNormRelL2GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormRelL2GetBufferHostSize_8u_C1MR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormRel_L2_8s_C1MR_Ctx.
 * For common parameter descriptions, see \ref CommonNormRelL2GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormRelL2GetBufferHostSize_8s_C1MR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormRel_L2_16u_C1MR_Ctx.
 * For common parameter descriptions, see \ref CommonNormRelL2GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormRelL2GetBufferHostSize_16u_C1MR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormRel_L2_32f_C1MR_Ctx.
 * For common parameter descriptions, see \ref CommonNormRelL2GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormRelL2GetBufferHostSize_32f_C1MR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormRel_L2_8u_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonNormRelL2GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormRelL2GetBufferHostSize_8u_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormRel_L2_16u_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonNormRelL2GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormRelL2GetBufferHostSize_16u_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormRel_L2_16s_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonNormRelL2GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormRelL2GetBufferHostSize_16s_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormRel_L2_32f_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonNormRelL2GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormRelL2GetBufferHostSize_32f_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormRel_L2_8u_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonNormRelL2GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormRelL2GetBufferHostSize_8u_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormRel_L2_16u_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonNormRelL2GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormRelL2GetBufferHostSize_16u_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormRel_L2_16s_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonNormRelL2GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormRelL2GetBufferHostSize_16s_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormRel_L2_32f_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonNormRelL2GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormRelL2GetBufferHostSize_32f_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormRel_L2_8u_AC4R_Ctx.
 * For common parameter descriptions, see \ref CommonNormRelL2GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormRelL2GetBufferHostSize_8u_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormRel_L2_16u_AC4R_Ctx.
 * For common parameter descriptions, see \ref CommonNormRelL2GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormRelL2GetBufferHostSize_16u_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormRel_L2_16s_AC4R_Ctx.
 * For common parameter descriptions, see \ref CommonNormRelL2GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormRelL2GetBufferHostSize_16s_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormRel_L2_32f_AC4R_Ctx.
 * For common parameter descriptions, see \ref CommonNormRelL2GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormRelL2GetBufferHostSize_32f_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormRel_L2_8u_C3CMR_Ctx.
 * For common parameter descriptions, see \ref CommonNormRelL2GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormRelL2GetBufferHostSize_8u_C3CMR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormRel_L2_8s_C3CMR_Ctx.
 * For common parameter descriptions, see \ref CommonNormRelL2GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormRelL2GetBufferHostSize_8s_C3CMR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormRel_L2_16u_C3CMR_Ctx.
 * For common parameter descriptions, see \ref CommonNormRelL2GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormRelL2GetBufferHostSize_16u_C3CMR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Computes the device scratch buffer size (in bytes) for nppiNormRel_L2_32f_C3CMR_Ctx.
 * For common parameter descriptions, see \ref CommonNormRelL2GetBufferHostSizeParameters.
 */
NppStatus 
nppiNormRelL2GetBufferHostSize_32f_C3CMR_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);

/** @} */
/** @} image_L2_normrel */
/** @} image_norm */

/** 
 * \section image_dot_prod DotProd
 * @defgroup image_dot_prod DotProd
 * Primitives for computing the dot product of two images.
 * @{
 *
 */

/** @name DotProd
 * Given two images \f$pSrc1\f$ and \f$pSrc2\f$ both with width \f$W\f$ and height \f$H\f$, 
 * the dot product will be computed as
 * \f[DotProd = \sum_{j=0}^{H-1}\sum_{i=0}^{W-1}[pSrc1(j,i)\cdot pSrc2(j,i)]\f]
 * The functions require additional scratch buffer for computations.
 * \section CommonDotProdParameters Common parameters for nppiDotProd functions include:
 * \param pSrc1 \ref source_image_pointer.
 * \param nSrc1Step \ref source_image_line_step.
 * \param pSrc2 \ref source_image_pointer.
 * \param nSrc2Step \ref source_image_line_step.
 * \param oSizeROI \ref roi_specification.
 * \param pDp Pointer to the computed dot product of the two images. 
 * \param pDeviceBuffer Pointer to the required device memory allocation, \ref general_scratch_buffer. 
 * \param nppStreamCtx \ref application_managed_stream_context. 
 * \return \ref image_data_error_codes, \ref roi_error_codes.
 * @{
 */
/**
 * One-channel 8-bit unsigned image DotProd.
 * For common parameter descriptions, see \ref CommonDotProdParameters.
 */
NppStatus nppiDotProd_8u64f_C1R_Ctx(const Npp8u * pSrc1, int nSrc1Step, 
                                    const Npp8u * pSrc2, int nSrc2Step, 
                                    NppiSize oSizeROI, Npp64f * pDp, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 8-bit signed image DotProd.
 * For common parameter descriptions, see \ref CommonDotProdParameters.
 */
NppStatus nppiDotProd_8s64f_C1R_Ctx(const Npp8s * pSrc1, int nSrc1Step, 
                                    const Npp8s * pSrc2, int nSrc2Step, 
                                    NppiSize oSizeROI, Npp64f * pDp, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 16-bit unsigned image DotProd.
 * For common parameter descriptions, see \ref CommonDotProdParameters.
 */
NppStatus nppiDotProd_16u64f_C1R_Ctx(const Npp16u * pSrc1, int nSrc1Step, 
                                     const Npp16u * pSrc2, int nSrc2Step, 
                                     NppiSize oSizeROI, Npp64f * pDp, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 16-bit signed image DotProd.
 * For common parameter descriptions, see \ref CommonDotProdParameters.
 */
NppStatus nppiDotProd_16s64f_C1R_Ctx(const Npp16s * pSrc1, int nSrc1Step, 
                                     const Npp16s * pSrc2, int nSrc2Step, 
                                     NppiSize oSizeROI, Npp64f * pDp, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 32-bit unsigned image DotProd.
 * For common parameter descriptions, see \ref CommonDotProdParameters.
 */
NppStatus nppiDotProd_32u64f_C1R_Ctx(const Npp32u * pSrc1, int nSrc1Step, 
                                     const Npp32u * pSrc2, int nSrc2Step, 
                                     NppiSize oSizeROI, Npp64f * pDp, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 32-bit signed image DotProd.
 * For common parameter descriptions, see \ref CommonDotProdParameters.
 */
NppStatus nppiDotProd_32s64f_C1R_Ctx(const Npp32s * pSrc1, int nSrc1Step, 
                                     const Npp32s * pSrc2, int nSrc2Step, 
                                     NppiSize oSizeROI, Npp64f * pDp, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 32-bit floating point image DotProd.
 * For common parameter descriptions, see \ref CommonDotProdParameters.
 */
NppStatus nppiDotProd_32f64f_C1R_Ctx(const Npp32f * pSrc1, int nSrc1Step, 
                                     const Npp32f * pSrc2, int nSrc2Step, 
                                     NppiSize oSizeROI, Npp64f * pDp, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 8-bit unsigned image DotProd.
 * For common parameter descriptions, see \ref CommonDotProdParameters.
 */
NppStatus nppiDotProd_8u64f_C3R_Ctx(const Npp8u * pSrc1, int nSrc1Step, 
                                    const Npp8u * pSrc2, int nSrc2Step, 
                                    NppiSize oSizeROI, Npp64f aDp[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 8-bit signed image DotProd.
 * For common parameter descriptions, see \ref CommonDotProdParameters.
 */
NppStatus nppiDotProd_8s64f_C3R_Ctx(const Npp8s * pSrc1, int nSrc1Step, 
                                    const Npp8s * pSrc2, int nSrc2Step, 
                                    NppiSize oSizeROI, Npp64f aDp[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 16-bit unsigned image DotProd.
 * For common parameter descriptions, see \ref CommonDotProdParameters.
 */
NppStatus nppiDotProd_16u64f_C3R_Ctx(const Npp16u * pSrc1, int nSrc1Step, 
                                     const Npp16u * pSrc2, int nSrc2Step, 
                                     NppiSize oSizeROI, Npp64f aDp[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 16-bit signed image DotProd.
 * For common parameter descriptions, see \ref CommonDotProdParameters.
 */
NppStatus nppiDotProd_16s64f_C3R_Ctx(const Npp16s * pSrc1, int nSrc1Step, 
                                     const Npp16s * pSrc2, int nSrc2Step, 
                                     NppiSize oSizeROI, Npp64f aDp[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 32-bit unsigned image DotProd.
 * For common parameter descriptions, see \ref CommonDotProdParameters.
 */
NppStatus nppiDotProd_32u64f_C3R_Ctx(const Npp32u * pSrc1, int nSrc1Step, 
                                     const Npp32u * pSrc2, int nSrc2Step, 
                                     NppiSize oSizeROI, Npp64f aDp[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 32-bit signed image DotProd.
 * For common parameter descriptions, see \ref CommonDotProdParameters.
 */
NppStatus nppiDotProd_32s64f_C3R_Ctx(const Npp32s * pSrc1, int nSrc1Step, 
                                     const Npp32s * pSrc2, int nSrc2Step, 
                                     NppiSize oSizeROI, Npp64f aDp[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 32-bit floating point image DotProd.
 * For common parameter descriptions, see \ref CommonDotProdParameters.
 */
NppStatus nppiDotProd_32f64f_C3R_Ctx(const Npp32f * pSrc1, int nSrc1Step, 
                                     const Npp32f * pSrc2, int nSrc2Step, 
                                     NppiSize oSizeROI, Npp64f aDp[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image DotProd.
 * For common parameter descriptions, see \ref CommonDotProdParameters.
 */
NppStatus nppiDotProd_8u64f_C4R_Ctx(const Npp8u * pSrc1, int nSrc1Step, 
                                    const Npp8u * pSrc2, int nSrc2Step, 
                                    NppiSize oSizeROI, Npp64f aDp[4], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit signed image DotProd.
 * For common parameter descriptions, see \ref CommonDotProdParameters.
 */
NppStatus nppiDotProd_8s64f_C4R_Ctx(const Npp8s * pSrc1, int nSrc1Step, 
                                    const Npp8s * pSrc2, int nSrc2Step, 
                                    NppiSize oSizeROI, Npp64f aDp[4], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit unsigned image DotProd.
 * For common parameter descriptions, see \ref CommonDotProdParameters.
 */
NppStatus nppiDotProd_16u64f_C4R_Ctx(const Npp16u * pSrc1, int nSrc1Step, 
                                     const Npp16u * pSrc2, int nSrc2Step, 
                                     NppiSize oSizeROI, Npp64f aDp[4], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit signed image DotProd.
 * For common parameter descriptions, see \ref CommonDotProdParameters.
 */
NppStatus nppiDotProd_16s64f_C4R_Ctx(const Npp16s * pSrc1, int nSrc1Step, 
                                     const Npp16s * pSrc2, int nSrc2Step, 
                                     NppiSize oSizeROI, Npp64f aDp[4], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit unsigned image DotProd.
 * For common parameter descriptions, see \ref CommonDotProdParameters.
 */
NppStatus nppiDotProd_32u64f_C4R_Ctx(const Npp32u * pSrc1, int nSrc1Step, 
                                     const Npp32u * pSrc2, int nSrc2Step, 
                                     NppiSize oSizeROI, Npp64f aDp[4], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit signed image DotProd.
 * For common parameter descriptions, see \ref CommonDotProdParameters.
 */
NppStatus nppiDotProd_32s64f_C4R_Ctx(const Npp32s * pSrc1, int nSrc1Step, 
                                     const Npp32s * pSrc2, int nSrc2Step, 
                                     NppiSize oSizeROI, Npp64f aDp[4], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit floating point image DotProd.
 * For common parameter descriptions, see \ref CommonDotProdParameters.
 */
NppStatus nppiDotProd_32f64f_C4R_Ctx(const Npp32f * pSrc1, int nSrc1Step, 
                                     const Npp32f * pSrc2, int nSrc2Step, 
                                     NppiSize oSizeROI, Npp64f aDp[4], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image DotProd ignoring alpha channel.
 * For common parameter descriptions, see \ref CommonDotProdParameters.
 */
NppStatus nppiDotProd_8u64f_AC4R_Ctx(const Npp8u * pSrc1, int nSrc1Step, 
                                     const Npp8u * pSrc2, int nSrc2Step, 
                                     NppiSize oSizeROI, Npp64f aDp[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit signed image DotProd ignoring alpha channel.
 * For common parameter descriptions, see \ref CommonDotProdParameters.
 */
NppStatus nppiDotProd_8s64f_AC4R_Ctx(const Npp8s * pSrc1, int nSrc1Step, 
                                     const Npp8s * pSrc2, int nSrc2Step, 
                                     NppiSize oSizeROI, Npp64f aDp[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit unsigned image DotProd ignoring alpha channel.
 * For common parameter descriptions, see \ref CommonDotProdParameters.
 */
NppStatus nppiDotProd_16u64f_AC4R_Ctx(const Npp16u * pSrc1, int nSrc1Step, 
                                      const Npp16u * pSrc2, int nSrc2Step, 
                                      NppiSize oSizeROI, Npp64f aDp[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit signed image DotProd ignoring alpha channel.
 * For common parameter descriptions, see \ref CommonDotProdParameters.
 */
NppStatus nppiDotProd_16s64f_AC4R_Ctx(const Npp16s * pSrc1, int nSrc1Step, 
                                      const Npp16s * pSrc2, int nSrc2Step, 
                                      NppiSize oSizeROI, Npp64f aDp[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit unsigned image DotProd ignoring alpha channel.
 * For common parameter descriptions, see \ref CommonDotProdParameters.
 */
NppStatus nppiDotProd_32u64f_AC4R_Ctx(const Npp32u * pSrc1, int nSrc1Step, 
                                      const Npp32u * pSrc2, int nSrc2Step, 
                                      NppiSize oSizeROI, Npp64f aDp[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit signed image DotProd ignoring alpha channel.
 * For common parameter descriptions, see \ref CommonDotProdParameters.
 */
NppStatus nppiDotProd_32s64f_AC4R_Ctx(const Npp32s * pSrc1, int nSrc1Step, 
                                      const Npp32s * pSrc2, int nSrc2Step, 
                                      NppiSize oSizeROI, Npp64f aDp[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit floating point image DotProd ignoring alpha channel.
 * For common parameter descriptions, see \ref CommonDotProdParameters.
 */
NppStatus nppiDotProd_32f64f_AC4R_Ctx(const Npp32f * pSrc1, int nSrc1Step, 
                                      const Npp32f * pSrc2, int nSrc2Step, 
                                      NppiSize oSizeROI, Npp64f aDp[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);

/** @} */

/** @name DotProdGetBufferHostSize
 * Companion primitives for computing the device buffer size (in bytes) required by the Mean_StdDev primitives.
 * \section CommonDotProdGetBufferHostSizeParameters
 * Common parameters for nppiDotProdGetBufferHostSize functions include:
 * \param oSizeROI \ref roi_specification.
 * \param hpBufferSize Required buffer size. Important: hpBufferSize is a 
 *        <em>host pointer.</em> \ref general_scratch_buffer.
 * \param nppStreamCtx \ref application_managed_stream_context. 
 * \return NPP_NULL_POINTER_ERROR if hpBufferSize is 0 (NULL),  \ref roi_error_codes.  
 * @{
 */ 
/** 
 * Device scratch buffer size (in bytes) for nppiDotProd_8u64f_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonDotProdGetBufferHostSizeParameters.
 */
NppStatus 
nppiDotProdGetBufferHostSize_8u64f_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Device scratch buffer size (in bytes) for nppiDotProd_8s64f_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonDotProdGetBufferHostSizeParameters.
 */
NppStatus 
nppiDotProdGetBufferHostSize_8s64f_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Device scratch buffer size (in bytes) for nppiDotProd_16u64f_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonDotProdGetBufferHostSizeParameters.
 */
NppStatus 
nppiDotProdGetBufferHostSize_16u64f_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Device scratch buffer size (in bytes) for nppiDotProd_16s64f_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonDotProdGetBufferHostSizeParameters.
 */
NppStatus 
nppiDotProdGetBufferHostSize_16s64f_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Device scratch buffer size (in bytes) for nppiDotProd_32u64f_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonDotProdGetBufferHostSizeParameters.
 */
NppStatus 
nppiDotProdGetBufferHostSize_32u64f_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Device scratch buffer size (in bytes) for nppiDotProd_32s64f_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonDotProdGetBufferHostSizeParameters.
 */
NppStatus 
nppiDotProdGetBufferHostSize_32s64f_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Device scratch buffer size (in bytes) for nppiDotProd_32f64f_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonDotProdGetBufferHostSizeParameters.
 */
NppStatus 
nppiDotProdGetBufferHostSize_32f64f_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Device scratch buffer size (in bytes) for nppiDotProd_8u64f_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonDotProdGetBufferHostSizeParameters.
 */
NppStatus 
nppiDotProdGetBufferHostSize_8u64f_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Device scratch buffer size (in bytes) for nppiDotProd_8s64f_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonDotProdGetBufferHostSizeParameters.
 */
NppStatus 
nppiDotProdGetBufferHostSize_8s64f_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Device scratch buffer size (in bytes) for nppiDotProd_16u64f_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonDotProdGetBufferHostSizeParameters.
 */
NppStatus 
nppiDotProdGetBufferHostSize_16u64f_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Device scratch buffer size (in bytes) for nppiDotProd_16s64f_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonDotProdGetBufferHostSizeParameters.
 */
NppStatus 
nppiDotProdGetBufferHostSize_16s64f_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Device scratch buffer size (in bytes) for nppiDotProd_32u64f_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonDotProdGetBufferHostSizeParameters.
 */
NppStatus 
nppiDotProdGetBufferHostSize_32u64f_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Device scratch buffer size (in bytes) for nppiDotProd_32s64f_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonDotProdGetBufferHostSizeParameters.
 */
NppStatus 
nppiDotProdGetBufferHostSize_32s64f_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Device scratch buffer size (in bytes) for nppiDotProd_32f64f_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonDotProdGetBufferHostSizeParameters.
 */
NppStatus 
nppiDotProdGetBufferHostSize_32f64f_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Device scratch buffer size (in bytes) for nppiDotProd_8u64f_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonDotProdGetBufferHostSizeParameters.
 */
NppStatus 
nppiDotProdGetBufferHostSize_8u64f_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Device scratch buffer size (in bytes) for nppiDotProd_8s64f_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonDotProdGetBufferHostSizeParameters.
 */
NppStatus 
nppiDotProdGetBufferHostSize_8s64f_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Device scratch buffer size (in bytes) for nppiDotProd_16u64f_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonDotProdGetBufferHostSizeParameters.
 */
NppStatus 
nppiDotProdGetBufferHostSize_16u64f_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Device scratch buffer size (in bytes) for nppiDotProd_16s64f_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonDotProdGetBufferHostSizeParameters.
 */
NppStatus 
nppiDotProdGetBufferHostSize_16s64f_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Device scratch buffer size (in bytes) for nppiDotProd_32u64f_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonDotProdGetBufferHostSizeParameters.
 */
NppStatus 
nppiDotProdGetBufferHostSize_32u64f_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Device scratch buffer size (in bytes) for nppiDotProd_32s64f_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonDotProdGetBufferHostSizeParameters.
 */
NppStatus 
nppiDotProdGetBufferHostSize_32s64f_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Device scratch buffer size (in bytes) for nppiDotProd_32f64f_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonDotProdGetBufferHostSizeParameters.
 */
NppStatus 
nppiDotProdGetBufferHostSize_32f64f_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Device scratch buffer size (in bytes) for nppiDotProd_8u64f_AC4R_Ctx.
 * For common parameter descriptions, see \ref CommonDotProdGetBufferHostSizeParameters.
 */
NppStatus 
nppiDotProdGetBufferHostSize_8u64f_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Device scratch buffer size (in bytes) for nppiDotProd_8s64f_AC4R_Ctx.
 * For common parameter descriptions, see \ref CommonDotProdGetBufferHostSizeParameters.
 */
NppStatus 
nppiDotProdGetBufferHostSize_8s64f_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Device scratch buffer size (in bytes) for nppiDotProd_16u64f_AC4R_Ctx.
 * For common parameter descriptions, see \ref CommonDotProdGetBufferHostSizeParameters.
 */
NppStatus 
nppiDotProdGetBufferHostSize_16u64f_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Device scratch buffer size (in bytes) for nppiDotProd_16s64f_AC4R_Ctx.
 * For common parameter descriptions, see \ref CommonDotProdGetBufferHostSizeParameters.
 */
NppStatus 
nppiDotProdGetBufferHostSize_16s64f_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Device scratch buffer size (in bytes) for nppiDotProd_32u64f_AC4R_Ctx.
 * For common parameter descriptions, see \ref CommonDotProdGetBufferHostSizeParameters.
 */
NppStatus 
nppiDotProdGetBufferHostSize_32u64f_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Device scratch buffer size (in bytes) for nppiDotProd_32s64f_AC4R_Ctx.
 * For common parameter descriptions, see \ref CommonDotProdGetBufferHostSizeParameters.
 */
NppStatus 
nppiDotProdGetBufferHostSize_32s64f_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Device scratch buffer size (in bytes) for nppiDotProd_32f64f_AC4R_Ctx.
 * For common parameter descriptions, see \ref CommonDotProdGetBufferHostSizeParameters.
 */
NppStatus 
nppiDotProdGetBufferHostSize_32f64f_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);

/** @} */
/** @} image_dot_prod */

/** 
 * \section image_count_in_range CountInRange.
 * @defgroup image_count_in_range CountInRange
 * Primitives for computing the amount of pixels that fall into the specified intensity range.
 * @{
 *
 */

/** @name CountInRange
 * The lower bound and the upper bound are inclusive. The functions require additional
 * scratch buffer for computations. 
 * \section CommonCountInRangeParameters Common parameters for nppiCountInRange functions include:
 * \param pSrc \ref source_image_pointer.
 * \param nSrcStep \ref source_image_line_step.
 * \param oSizeROI \ref roi_specification.
 * \param pCounts Pointer to the number of pixels that fall into the specified range.
 * \param nLowerBound Lower bound of the specified range.
 * \param nUpperBound Upper bound of the specified range.
 * \param pDeviceBuffer Pointer to the required device memory allocation, \ref general_scratch_buffer. 
 * \param nppStreamCtx \ref application_managed_stream_context. 
 * \return \ref image_data_error_codes, \ref roi_error_codes, or NPP_RANGE_ERROR if the lower bound
 * is larger than the upper bound.
 * @{
 *
 */

/**
 * One-channel 8-bit unsigned image CountInRange.
 * For common parameter descriptions, see \ref CommonCountInRangeParameters.
 */
NppStatus nppiCountInRange_8u_C1R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSizeROI, 
                                      int * pCounts, Npp8u nLowerBound, Npp8u nUpperBound, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 32-bit floating point image CountInRange.
 * For common parameter descriptions, see \ref CommonCountInRangeParameters.
 */
NppStatus nppiCountInRange_32f_C1R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSizeROI, 
                                       int * pCounts, Npp32f nLowerBound, Npp32f nUpperBound, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 8-bit unsigned image CountInRange.
 * For common parameter descriptions, see \ref CommonCountInRangeParameters.
 */
NppStatus nppiCountInRange_8u_C3R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSizeROI, 
                                      int aCounts[3], Npp8u aLowerBound[3], Npp8u aUpperBound[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 32-bit floating point image CountInRange.
 * For common parameter descriptions, see \ref CommonCountInRangeParameters.
 */
NppStatus nppiCountInRange_32f_C3R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSizeROI, 
                                       int aCounts[3], Npp32f aLowerBound[3], Npp32f aUpperBound[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image CountInRange ignoring alpha channel.
 * For common parameter descriptions, see \ref CommonCountInRangeParameters.
 */
NppStatus nppiCountInRange_8u_AC4R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSizeROI, 
                                       int aCounts[3], Npp8u aLowerBound[3], Npp8u aUpperBound[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit floating point image CountInRange ignoring alpha channel.
 * For common parameter descriptions, see \ref CommonCountInRangeParameters.
 */
NppStatus nppiCountInRange_32f_AC4R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSizeROI, 
                                        int aCounts[3], Npp32f aLowerBound[3], Npp32f aUpperBound[3], Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);

/** @} */
 
 /** @name CountInRangeGetBufferHostSize
 * Companion primitives for computing the device buffer size (in bytes) required by the CountInRange primitives.
 * \section CommonCountInRangeGetBufferHostSizeParameters
 * Common parameters for nppiCountInRangeGetBufferHostSize functions include:
 * \param oSizeROI \ref roi_specification.
 * \param hpBufferSize Required buffer size. Important: hpBufferSize is a 
 *        <em>host pointer.</em> \ref general_scratch_buffer.
 * \param nppStreamCtx \ref application_managed_stream_context. 
 * \return NPP_NULL_POINTER_ERROR if hpBufferSize is 0 (NULL),  \ref roi_error_codes.  
 * @{
 */

/** 
 * Device scratch buffer size (in bytes) for nppiCountInRange_8u_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonCountInRangeGetBufferHostSizeParameters.
 */
NppStatus 
nppiCountInRangeGetBufferHostSize_8u_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Device scratch buffer size (in bytes) for nppiCountInRange_32f_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonCountInRangeGetBufferHostSizeParameters.
 */
NppStatus 
nppiCountInRangeGetBufferHostSize_32f_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Device scratch buffer size (in bytes) for nppiCountInRange_8u_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonCountInRangeGetBufferHostSizeParameters.
 */
NppStatus 
nppiCountInRangeGetBufferHostSize_8u_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Device scratch buffer size (in bytes) for nppiCountInRange_32f_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonCountInRangeGetBufferHostSizeParameters.
 */
NppStatus 
nppiCountInRangeGetBufferHostSize_32f_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Device scratch buffer size (in bytes) for nppiCountInRange_8u_AC4R_Ctx.
 * For common parameter descriptions, see \ref CommonCountInRangeGetBufferHostSizeParameters.
 */
NppStatus 
nppiCountInRangeGetBufferHostSize_8u_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Device scratch buffer size (in bytes) for nppiCountInRange_32f_AC4R_Ctx.
 * For common parameter descriptions, see \ref CommonCountInRangeGetBufferHostSizeParameters.
 */
NppStatus 
nppiCountInRangeGetBufferHostSize_32f_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);

/** @} */
/** @} image_count_in_range */

/** 
 * \section image_maxevery MaxEvery
 * @defgroup image_maxevery MaxEvery
 * Primitives for computing the maximal value of the pixel pair from two images.
 * @{
 *
 */

/** @name MaxEvery
 * The maximum is stored into the second image.
 * \section CommonMaxEveryParameters Common parameters for nppiMaxEvery functions include: 
 * \param pSrc \ref source_image_pointer.
 * \param nSrcStep \ref source_image_line_step.
 * \param pSrcDst \ref in_place_image_pointer.
 * \param nSrcDstStep \ref source_image_line_step.
 * \param oSizeROI \ref roi_specification.
 * \param nppStreamCtx \ref application_managed_stream_context. 
 * \return \ref image_data_error_codes, \ref roi_error_codes.
 * @{
 *
 */
/**
 * One-channel 8-bit unsigned image MaxEvery.
 * For common parameter descriptions, see \ref CommonMaxEveryParameters.
 */
NppStatus nppiMaxEvery_8u_C1IR_Ctx(const Npp8u * pSrc, int nSrcStep, Npp8u * pSrcDst, int nSrcDstStep, NppiSize oSizeROI, NppStreamContext nppStreamCtx);
/**
 * One-channel 16-bit unsigned image MaxEvery.
 * For common parameter descriptions, see \ref CommonMaxEveryParameters.
 */
NppStatus nppiMaxEvery_16u_C1IR_Ctx(const Npp16u * pSrc, int nSrcStep, Npp16u * pSrcDst, int nSrcDstStep, NppiSize oSizeROI, NppStreamContext nppStreamCtx);
/**
 * One-channel 16-bit signed image MaxEvery.
 * For common parameter descriptions, see \ref CommonMaxEveryParameters.
 */
NppStatus nppiMaxEvery_16s_C1IR_Ctx(const Npp16s * pSrc, int nSrcStep, Npp16s * pSrcDst, int nSrcDstStep, NppiSize oSizeROI, NppStreamContext nppStreamCtx);
/**
 * One-channel 32-bit floating point image MaxEvery.
 * For common parameter descriptions, see \ref CommonMaxEveryParameters.
 */
NppStatus nppiMaxEvery_32f_C1IR_Ctx(const Npp32f * pSrc, int nSrcStep, Npp32f * pSrcDst, int nSrcDstStep, NppiSize oSizeROI, NppStreamContext nppStreamCtx);
/**
 * Three-channel 8-bit unsigned image MaxEvery.
 * For common parameter descriptions, see \ref CommonMaxEveryParameters.
 */
NppStatus nppiMaxEvery_8u_C3IR_Ctx(const Npp8u * pSrc, int nSrcStep, Npp8u * pSrcDst, int nSrcDstStep, NppiSize oSizeROI, NppStreamContext nppStreamCtx);
/**
 * Three-channel 16-bit unsigned image MaxEvery.
 * For common parameter descriptions, see \ref CommonMaxEveryParameters.
 */
NppStatus nppiMaxEvery_16u_C3IR_Ctx(const Npp16u * pSrc, int nSrcStep, Npp16u * pSrcDst, int nSrcDstStep, NppiSize oSizeROI, NppStreamContext nppStreamCtx);
/**
 * Three-channel 16-bit signed image MaxEvery.
 * For common parameter descriptions, see \ref CommonMaxEveryParameters.
 */
NppStatus nppiMaxEvery_16s_C3IR_Ctx(const Npp16s * pSrc, int nSrcStep, Npp16s * pSrcDst, int nSrcDstStep, NppiSize oSizeROI, NppStreamContext nppStreamCtx);
/**
 * Three-channel 32-bit floating point image MaxEvery.
 * For common parameter descriptions, see \ref CommonMaxEveryParameters.
 */
NppStatus nppiMaxEvery_32f_C3IR_Ctx(const Npp32f * pSrc, int nSrcStep, Npp32f * pSrcDst, int nSrcDstStep, NppiSize oSizeROI, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image MaxEvery.
 * For common parameter descriptions, see \ref CommonMaxEveryParameters.
 */
NppStatus nppiMaxEvery_8u_C4IR_Ctx(const Npp8u * pSrc, int nSrcStep, Npp8u * pSrcDst, int nSrcDstStep, NppiSize oSizeROI, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit unsigned image MaxEvery.
 * For common parameter descriptions, see \ref CommonMaxEveryParameters.
 */
NppStatus nppiMaxEvery_16u_C4IR_Ctx(const Npp16u * pSrc, int nSrcStep, Npp16u * pSrcDst, int nSrcDstStep, NppiSize oSizeROI, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit signed image MaxEvery.
 * For common parameter descriptions, see \ref CommonMaxEveryParameters.
 */
NppStatus nppiMaxEvery_16s_C4IR_Ctx(const Npp16s * pSrc, int nSrcStep, Npp16s * pSrcDst, int nSrcDstStep, NppiSize oSizeROI, NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit floating point image MaxEvery.
 * For common parameter descriptions, see \ref CommonMaxEveryParameters.
 */
NppStatus nppiMaxEvery_32f_C4IR_Ctx(const Npp32f * pSrc, int nSrcStep, Npp32f * pSrcDst, int nSrcDstStep, NppiSize oSizeROI, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image MaxEvery ignoring alpha channel.
 * For common parameter descriptions, see \ref CommonMaxEveryParameters.
 */
NppStatus nppiMaxEvery_8u_AC4IR_Ctx(const Npp8u * pSrc, int nSrcStep, Npp8u * pSrcDst, int nSrcDstStep, NppiSize oSizeROI, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit unsigned image MaxEvery ignoring alpha channel.
 * For common parameter descriptions, see \ref CommonMaxEveryParameters.
 */
NppStatus nppiMaxEvery_16u_AC4IR_Ctx(const Npp16u * pSrc, int nSrcStep, Npp16u * pSrcDst, int nSrcDstStep, NppiSize oSizeROI, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit signed image MaxEvery ignoring alpha channel.
 * For common parameter descriptions, see \ref CommonMaxEveryParameters.
 */
NppStatus nppiMaxEvery_16s_AC4IR_Ctx(const Npp16s * pSrc, int nSrcStep, Npp16s * pSrcDst, int nSrcDstStep, NppiSize oSizeROI, NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit floating point image MaxEvery ignoring alpha channel.
 * For common parameter descriptions, see \ref CommonMaxEveryParameters.
 */
NppStatus nppiMaxEvery_32f_AC4IR_Ctx(const Npp32f * pSrc, int nSrcStep, Npp32f * pSrcDst, int nSrcDstStep, NppiSize oSizeROI, NppStreamContext nppStreamCtx);

/** @} */
/** @} image_maxevery*/

/** 
 * \section image_minevery MinEvery
 * @defgroup image_minevery MinEvery
 * Primitives for computing the minimal value of the pixel pair from two images.
 * @{
 *
 */

/** @name MinEvery
 * The minimum is stored into the second image.
 * \section CommonMinEveryParameters Common parameters for nppiMinEvery functions include:
 * \param pSrc \ref source_image_pointer.
 * \param nSrcStep \ref source_image_line_step.
 * \param pSrcDst \ref in_place_image_pointer.
 * \param nSrcDstStep \ref source_image_line_step.
 * \param oSizeROI \ref roi_specification.
 * \param nppStreamCtx \ref application_managed_stream_context. 
 * \return \ref image_data_error_codes, \ref roi_error_codes.
 * @{
 *
 */
/**
 * One-channel 8-bit unsigned image MinEvery.
 * For common parameter descriptions, see \ref CommonMinEveryParameters.
 */
NppStatus nppiMinEvery_8u_C1IR_Ctx(const Npp8u * pSrc, int nSrcStep, Npp8u * pSrcDst, int nSrcDstStep, NppiSize oSizeROI, NppStreamContext nppStreamCtx);
/**
 * One-channel 16-bit unsigned image MinEvery.
 * For common parameter descriptions, see \ref CommonMinEveryParameters.
 */
NppStatus nppiMinEvery_16u_C1IR_Ctx(const Npp16u * pSrc, int nSrcStep, Npp16u * pSrcDst, int nSrcDstStep, NppiSize oSizeROI, NppStreamContext nppStreamCtx);
/**
 * One-channel 16-bit signed image MinEvery.
 * For common parameter descriptions, see \ref CommonMinEveryParameters.
 */
NppStatus nppiMinEvery_16s_C1IR_Ctx(const Npp16s * pSrc, int nSrcStep, Npp16s * pSrcDst, int nSrcDstStep, NppiSize oSizeROI, NppStreamContext nppStreamCtx);
/**
 * One-channel 32-bit floating point image MinEvery.
 * For common parameter descriptions, see \ref CommonMinEveryParameters.
 */
NppStatus nppiMinEvery_32f_C1IR_Ctx(const Npp32f * pSrc, int nSrcStep, Npp32f * pSrcDst, int nSrcDstStep, NppiSize oSizeROI, NppStreamContext nppStreamCtx);
/**
 * Three-channel 8-bit unsigned image MinEvery.
 * For common parameter descriptions, see \ref CommonMinEveryParameters.
 */
NppStatus nppiMinEvery_8u_C3IR_Ctx(const Npp8u * pSrc, int nSrcStep, Npp8u * pSrcDst, int nSrcDstStep, NppiSize oSizeROI, NppStreamContext nppStreamCtx);
/**
 * Three-channel 16-bit unsigned image MinEvery.
 * For common parameter descriptions, see \ref CommonMinEveryParameters.
 */
NppStatus nppiMinEvery_16u_C3IR_Ctx(const Npp16u * pSrc, int nSrcStep, Npp16u * pSrcDst, int nSrcDstStep, NppiSize oSizeROI, NppStreamContext nppStreamCtx);
/**
 * Three-channel 16-bit signed image MinEvery.
 * For common parameter descriptions, see \ref CommonMinEveryParameters.
 */
NppStatus nppiMinEvery_16s_C3IR_Ctx(const Npp16s * pSrc, int nSrcStep, Npp16s * pSrcDst, int nSrcDstStep, NppiSize oSizeROI, NppStreamContext nppStreamCtx);
/**
 * Three-channel 32-bit floating point image MinEvery.
 * For common parameter descriptions, see \ref CommonMinEveryParameters.
 */
NppStatus nppiMinEvery_32f_C3IR_Ctx(const Npp32f * pSrc, int nSrcStep, Npp32f * pSrcDst, int nSrcDstStep, NppiSize oSizeROI, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image MinEvery.
 * For common parameter descriptions, see \ref CommonMinEveryParameters.
 */
NppStatus nppiMinEvery_8u_C4IR_Ctx(const Npp8u * pSrc, int nSrcStep, Npp8u * pSrcDst, int nSrcDstStep, NppiSize oSizeROI, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit unsigned image MinEvery.
 * For common parameter descriptions, see \ref CommonMinEveryParameters.
 */
NppStatus nppiMinEvery_16u_C4IR_Ctx(const Npp16u * pSrc, int nSrcStep, Npp16u * pSrcDst, int nSrcDstStep, NppiSize oSizeROI, NppStreamContext nppStreamCtx);

/**
 * Four-channel 16-bit signed image MinEvery.
 * For common parameter descriptions, see \ref CommonMinEveryParameters.
 */
NppStatus nppiMinEvery_16s_C4IR_Ctx(const Npp16s * pSrc, int nSrcStep, Npp16s * pSrcDst, int nSrcDstStep, NppiSize oSizeROI, NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit floating point image MinEvery.
 * For common parameter descriptions, see \ref CommonMinEveryParameters.
 */
NppStatus nppiMinEvery_32f_C4IR_Ctx(const Npp32f * pSrc, int nSrcStep, Npp32f * pSrcDst, int nSrcDstStep, NppiSize oSizeROI, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image MinEvery ignoring alpha channel.
 * For common parameter descriptions, see \ref CommonMinEveryParameters.
 */
NppStatus nppiMinEvery_8u_AC4IR_Ctx(const Npp8u * pSrc, int nSrcStep, Npp8u * pSrcDst, int nSrcDstStep, NppiSize oSizeROI, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit unsigned image MinEvery ignoring alpha channel.
 * For common parameter descriptions, see \ref CommonMinEveryParameters.
 */
NppStatus nppiMinEvery_16u_AC4IR_Ctx(const Npp16u * pSrc, int nSrcStep, Npp16u * pSrcDst, int nSrcDstStep, NppiSize oSizeROI, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit signed image MinEvery ignoring alpha channel.
 * For common parameter descriptions, see \ref CommonMinEveryParameters.
 */
NppStatus nppiMinEvery_16s_AC4IR_Ctx(const Npp16s * pSrc, int nSrcStep, Npp16s * pSrcDst, int nSrcDstStep, NppiSize oSizeROI, NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit floating point image MinEvery ignoring alpha channel.
 * For common parameter descriptions, see \ref CommonMinEveryParameters.
 */
NppStatus nppiMinEvery_32f_AC4IR_Ctx(const Npp32f * pSrc, int nSrcStep, Npp32f * pSrcDst, int nSrcDstStep, NppiSize oSizeROI, NppStreamContext nppStreamCtx);

/** @} */
/** @} image_minevery */

/** 
 * \section image_integral Integral
 * @defgroup image_integral Integral
 * Primitives for computing the integral image of a given image.
 * @{
 *
 */
                                  
/** @name Integral
 * Given an input image \f$pSrc\f$ and the specified value \f$nVal\f$, 
 * the pixel value of the integral image \f$pDst\f$ at coordinate 
 * (i, j) will be computed as
 * \f[pDst(j,i) = nVal + \sum_{l=0}^{j-1}\sum_{k=0}^{i-1}pSrc(l,k)\f]
 * If the size of the input image is \f$W \times H\f$, the size of 
 * the integral image will be \f$(W+1) \times (H+1)\f$.
 * @{
 */
/**
 * One-channel 8-bit unsigned image Integral with 32-bit signed output.
 *
 * \param pSrc \ref source_image_pointer.
 * \param nSrcStep \ref source_image_line_step.
 * \param pDst \ref destination_image_pointer.
 * \param nDstStep \ref destination_image_line_step.
 * \param oROI \ref roi_specification.
 * \param nVal The value to add to pDst image pixels
 * \param nppStreamCtx \ref application_managed_stream_context. 
 * \return \ref image_data_error_codes, \ref roi_error_codes.
 */
NppStatus 
nppiIntegral_8u32s_C1R_Ctx(const Npp8u  * pSrc, int nSrcStep, 
                                 Npp32s * pDst, int nDstStep, 
                           NppiSize oROI, Npp32s nVal, NppStreamContext nppStreamCtx);

/**
 * One-channel 8-bit unsigned image Integral with 32-bit floating point output.
 * \param pSrc \ref source_image_pointer.
 * \param nSrcStep \ref source_image_line_step.
 * \param pDst \ref destination_image_pointer.
 * \param nDstStep \ref destination_image_line_step.
 * \param oROI \ref roi_specification.
 * \param nVal The value to add to pDst image pixels
 * \param nppStreamCtx \ref application_managed_stream_context. 
 * \return \ref image_data_error_codes, \ref roi_error_codes.
 */
NppStatus 
nppiIntegral_8u32f_C1R_Ctx(const Npp8u  * pSrc, int nSrcStep, 
                                 Npp32f * pDst, int nDstStep, 
                           NppiSize oROI, Npp32f nVal, NppStreamContext nppStreamCtx);

/** @} */
/** @} Integral */

/** 
 * \section image_sqrintegral SqrIntegral
 * @defgroup image_sqrintegral SqrIntegral
 * Primitives for computing both the integral and the squared integral images of a given image.
 * @{
 *
 */

/** @name SqrIntegral
 *
 * Given an input image \f$pSrc\f$ and the specified value \f$nVal\f$, 
 * the pixel value of the integral image \f$pDst\f$ at coordinate 
 * (i, j) will be computed as
 * \f[pDst(j,i) = nVal + \sum_{l=0}^{j-1}\sum_{k=0}^{i-1}pSrc(l,k)\f]
 * Given an input image \f$pSrc\f$ and the specified value \f$nValSqr\f$, 
 * the pixel value of the squared integral image \f$pSqr\f$ at coordinate 
 * (i, j) will be computed as
 * \f[pSqr(j,i) = nValSqr + \sum_{l=0}^{j-1}\sum_{k=0}^{i-1}{pSrc(l,k)}^2\f]
 * If the size of the input image is \f$W \times H\f$, the size of 
 * the squared integral image will be \f$(W+1) \times (H+1)\f$.
 * @{
 */

 /**
 * One-channel 8-bit unsigned image SqrIntegral. 
 * Destination integral image and square integral image are 32-bit signed int.
 *
 * \param pSrc \ref source_image_pointer.
 * \param nSrcStep \ref source_image_line_step.
 * \param pDst \ref destination_image_pointer.
 * \param nDstStep \ref destination_image_line_step.
 * \param pSqr \ref destination_image_pointer.
 * \param nSqrStep \ref destination_image_line_step.
 * \param oSrcROI \ref roi_specification.
 * \param nVal The value to add to pDst image pixels
 * \param nValSqr The value to add to pSqr image pixels
 * \param nppStreamCtx \ref application_managed_stream_context. 
 * \return \ref image_data_error_codes, \ref roi_error_codes
 */
NppStatus 
nppiSqrIntegral_8u32s_C1R_Ctx(const Npp8u  * pSrc, int nSrcStep, 
                                    Npp32s * pDst, int nDstStep, 
                                    Npp32s * pSqr, int nSqrStep,
                              NppiSize oSrcROI, Npp32s nVal, Npp32s nValSqr, NppStreamContext nppStreamCtx);
 /**
 * One-channel 8-bit unsigned image SqrIntegral. 
 * Destination integral image is 32-bit signed int. 
 * Destination square integral image is 64-bit double floating point.
 *
 * \param pSrc \ref source_image_pointer.
 * \param nSrcStep \ref source_image_line_step.
 * \param pDst \ref destination_image_pointer.
 * \param nDstStep \ref destination_image_line_step.
 * \param pSqr \ref destination_image_pointer.
 * \param nSqrStep \ref destination_image_line_step.
 * \param oSrcROI \ref roi_specification.
 * \param nVal The value to add to pDst image pixels
 * \param nValSqr The value to add to pSqr image pixels
 * \param nppStreamCtx \ref application_managed_stream_context. 
 * \return \ref image_data_error_codes, \ref roi_error_codes
 */
NppStatus 
nppiSqrIntegral_8u32s64f_C1R_Ctx(const Npp8u  * pSrc, int nSrcStep, 
                                       Npp32s * pDst, int nDstStep, 
                                       Npp64f * pSqr, int nSqrStep,
                                 NppiSize oSrcROI, Npp32s nVal, Npp64f nValSqr, NppStreamContext nppStreamCtx);
/**
 * One-channel 8-bit unsigned image SqrIntegral. 
 * Destination integral image is 32-bit floating point.
 * Destination square integral image is 64-bit double floating point.
 *
 * \param pSrc \ref source_image_pointer.
 * \param nSrcStep \ref source_image_line_step.
 * \param pDst \ref destination_image_pointer.
 * \param nDstStep \ref destination_image_line_step.
 * \param pSqr \ref destination_image_pointer.
 * \param nSqrStep \ref destination_image_line_step.
 * \param oSrcROI \ref roi_specification.
 * \param nVal The value to add to pDst image pixels
 * \param nValSqr The value to add to pSqr image pixels
 * \param nppStreamCtx \ref application_managed_stream_context. 
 * \return \ref image_data_error_codes, \ref roi_error_codes
 */
NppStatus 
nppiSqrIntegral_8u32f64f_C1R_Ctx(const Npp8u  * pSrc, int nSrcStep, 
                                       Npp32f * pDst, int nDstStep, 
                                       Npp64f * pSqr, int nSqrStep,
                                 NppiSize oSrcROI, Npp32f nVal, Npp64f nValSqr, NppStreamContext nppStreamCtx);

/** @}  */
/** @} image_sqrintegral */

/** 
 * \section image_rectstddev RectStdDev
 * @defgroup image_rectstddev RectStdDev
 * Primitives for computing the standard deviation of the integral images.
 * The function computes the standard deviation of the pixel in the rectangular window with the 
 * integral image \f$pSrc\f$ and the squared integral image \f$pSqr\f$, which can be obtained by calling
 * \ref image_integral and \ref image_sqrintegral.
 * 
 * The standard deviation of the pixel \f$(j, i)\f$ can be computed using the formula:
 * \f[pDst(j, i) = \sqrt{max(0, \frac{\sum(SqrIntegral)\cdot N - (\sum(Integral))^2}{N^2})}\f]
 * where \f$\sum(SqrIntegral) = pSqr[j+oRect.y+oRect.height, i+oRect.x+oRect.width] - pSqr[j+oRect.y,i+oRect.x+oRect.width]
 * - pSqr[j+oRect.y+oRect.height, i+oRect.x] + pSqr[j+oRect.y, i+oRect.x]\f$,
 * \f$\sum(Integral) = pSrc[j+oRect.y+oRect.height, i+oRect.x+oRect.width] - pSrc[j+oRect.y,i+oRect.x+oRect.width]
 * - pSrc[j+oRect.y+oRect.height, i+oRect.x] + pSrc[j+oRect.y, i+oRect.x]\f$,
 * \f$N = oRect.width \cdot oRect.height\f$.
 *
 * The size of the \f$pSrc\f$ and \f$pSqr\f$ should be \f$(oSizeROI.width + oRect.x + oRect.width, 
 * oSizeROI.height + oRect.y + oRect.height).\f$
 * @{
 *
 */
/** @name RectStdDev
 * @{
 */

/**
 * One-channel 32-bit floating point image RectStdDev.
 *
 * \param pSrc \ref source_image_pointer.
 * \param nSrcStep \ref source_image_line_step.
 * \param pSqr \ref destination_image_pointer.
 * \param nSqrStep \ref destination_image_line_step.
 * \param pDst \ref destination_image_pointer.
 * \param nDstStep \ref destination_image_line_step.
 * \param oSizeROI \ref roi_specification.
 * \param oRect rectangular window
 * \param nppStreamCtx \ref application_managed_stream_context. 
 * \return \ref image_data_error_codes, \ref roi_error_codes
 */
NppStatus 
nppiRectStdDev_32f_C1R_Ctx(const Npp32f *pSrc, int nSrcStep, const Npp64f *pSqr, int nSqrStep, 
                                 Npp32f *pDst, int nDstStep, NppiSize oSizeROI, NppiRect oRect, NppStreamContext nppStreamCtx);
/**
 * One-channel 32-bit signed image RectStdDev, scaled by \f$2^(-nScaleFactor)\f$. 
 *
 * \param pSrc \ref source_image_pointer.
 * \param nSrcStep \ref source_image_line_step.
 * \param pSqr \ref destination_image_pointer.
 * \param nSqrStep \ref destination_image_line_step.
 * \param pDst \ref destination_image_pointer.
 * \param nDstStep \ref destination_image_line_step.
 * \param oSizeROI \ref roi_specification.
 * \param oRect rectangular window
 * \param nScaleFactor \ref integer_result_scaling.
 * \param nppStreamCtx \ref application_managed_stream_context. 
 * \return \ref image_data_error_codes, \ref roi_error_codes
 */
NppStatus 
nppiRectStdDev_32s_C1RSfs_Ctx(const Npp32s *pSrc, int nSrcStep, const Npp32s *pSqr, int nSqrStep, 
                                    Npp32s *pDst, int nDstStep, NppiSize oSizeROI, NppiRect oRect, int nScaleFactor, NppStreamContext nppStreamCtx);
 /**
 * One-channel 32-bit signed image RectStdDev. 
 *
 * \param pSrc \ref source_image_pointer.
 * \param nSrcStep \ref source_image_line_step.
 * \param pSqr \ref destination_image_pointer.
 * \param nSqrStep \ref destination_image_line_step.
 * \param pDst \ref destination_image_pointer.
 * \param nDstStep \ref destination_image_line_step.
 * \param oSizeROI \ref roi_specification.
 * \param oRect rectangular window
 * \param nppStreamCtx \ref application_managed_stream_context. 
 * \return \ref image_data_error_codes, \ref roi_error_codes
 */
NppStatus 
nppiRectStdDev_32s32f_C1R_Ctx(const Npp32s *pSrc, int nSrcStep, const Npp64f *pSqr, int nSqrStep, 
                                    Npp32f *pDst, int nDstStep, NppiSize oSizeROI, NppiRect oRect, NppStreamContext nppStreamCtx);

/** @}  */
/** @} image_rectstddev */


/** 
 * \section image_histogrameven HistogramEven
 * @defgroup image_histogrameven HistogramEven
 * Primitives for computing the histogram of an image with evenly distributed bins.
 * @{
 *
 */

/** @name HistogramEven
 * The \f$nLowerLevel\f$ (inclusive) and \f$nUpperLevel\f$ (exclusive) define the boundaries of the range, 
 * which are evenly segmented into \f$nLevel - 1\f$ bins. The computed histogram is stored in \f$pHist\f$. 
 * The levels are calculated by another primitive \ref nppiEvenLevelsHost_32s and are stored 
 * in a host pointer \f$hpLevels\f$. The number of levels is also \f$nLevel - 1\f$.
 * The histogram \f$pHist[k]\f$ is defined as the total number of pixels that fall into the range:
 * \f$hpLevels[k] <= pSrc(j, i) < hpLevels[k+1]\f$.
 * The functions require additional scratch buffer for computations.
 * \section CommonHistogramEvenParameters Common parameters for nppiHistogramEven functions include:
 * \param pSrc \ref source_image_pointer.
 * \param nSrcStep \ref source_image_line_step.
 * \param oSizeROI \ref roi_specification.
 * \param pHist Pointer to array that receives the computed histogram. 
 *      The array must be of size nLevels-1. 
 * \param nLevels Number of levels.
 * \param nLowerLevel Lower boundary of lowest level bin.
 * \param nUpperLevel Upper boundary of highest level bin.
 * \param pBuffer Pointer to appropriately sized scratch buffer.
 * \param nppStreamCtx \ref application_managed_stream_context. 
 * \return \ref image_data_error_codes, \ref roi_error_codes.
 * @{
 */

/**
 * Compute levels with even distribution.
 *
 * \param hpLevels A host pointer to array which receives the levels being
 *        computed. The array needs to be of size nLevels. 
 * \param nLevels The number of levels being computed. nLevels must be at least 2.
 * \param nLowerLevel Lower boundary value of the lowest level.
 * \param nUpperLevel Upper boundary value of the greatest level.
 * \return image_data_error_codes, or NPP_HISTO_NUMBER_OF_LEVELS_ERROR if an invalid nLevels is specified.
 */
NppStatus
nppiEvenLevelsHost_32s(Npp32s * hpLevels, int nLevels, Npp32s nLowerLevel, Npp32s nUpperLevel);

/**
 * One-channel 8-bit unsigned HistogramEven.
 * For common parameter descriptions, see \ref CommonHistogramEvenParameters.
 */
NppStatus 
nppiHistogramEven_8u_C1R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp32s * pHist, 
                             int nLevels, Npp32s nLowerLevel, Npp32s nUpperLevel, Npp8u * pBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 8-bit unsigned HistogramEven.
 * For common parameter descriptions, see \ref CommonHistogramEvenParameters.
 */
NppStatus 
nppiHistogramEven_8u_C3R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp32s * pHist[3], 
                             int nLevels[3], Npp32s nLowerLevel[3], Npp32s nUpperLevel[3], Npp8u * pBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned HistogramEven.
 * For common parameter descriptions, see \ref CommonHistogramEvenParameters.
 */
NppStatus 
nppiHistogramEven_8u_C4R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSizeROI, 
                                   Npp32s * pHist[4], 
                             int nLevels[4], Npp32s nLowerLevel[4], Npp32s nUpperLevel[4], Npp8u * pBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned HistogramEven ignoring alpha channel.
 * For common parameter descriptions, see \ref CommonHistogramEvenParameters.
 */
NppStatus 
nppiHistogramEven_8u_AC4R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSizeROI, 
                                    Npp32s * pHist[3], 
                              int nLevels[3], Npp32s nLowerLevel[3], Npp32s nUpperLevel[3], Npp8u * pBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 16-bit unsigned HistogramEven.
 * For common parameter descriptions, see \ref CommonHistogramEvenParameters.
 */
NppStatus 
nppiHistogramEven_16u_C1R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp32s * pHist, 
                              int nLevels, Npp32s nLowerLevel, Npp32s nUpperLevel, Npp8u * pBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 16-bit unsigned HistogramEven.
 * For common parameter descriptions, see \ref CommonHistogramEvenParameters.
 */
NppStatus 
nppiHistogramEven_16u_C3R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSizeROI, 
                                    Npp32s * pHist[3], 
                              int nLevels[3], Npp32s nLowerLevel[3], Npp32s nUpperLevel[3], Npp8u * pBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit unsigned HistogramEven.
 * For common parameter descriptions, see \ref CommonHistogramEvenParameters.
 */
NppStatus 
nppiHistogramEven_16u_C4R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSizeROI, 
                                    Npp32s * pHist[4], 
                              int nLevels[4], Npp32s nLowerLevel[4], Npp32s nUpperLevel[4], Npp8u * pBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit unsigned HistogramEven ignoring alpha channel.
 * For common parameter descriptions, see \ref CommonHistogramEvenParameters.
 */
NppStatus 
nppiHistogramEven_16u_AC4R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSizeROI, 
                                     Npp32s * pHist[3], 
                               int nLevels[3], Npp32s nLowerLevel[3], Npp32s nUpperLevel[3], Npp8u * pBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 16-bit signed HistogramEven.
 * For common parameter descriptions, see \ref CommonHistogramEvenParameters.
 */
NppStatus 
nppiHistogramEven_16s_C1R_Ctx(const Npp16s * pSrc, int nSrcStep, NppiSize oSizeROI, Npp32s * pHist, 
                              int nLevels, Npp32s nLowerLevel, Npp32s nUpperLevel, Npp8u * pBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 16-bit signed HistogramEven.
 * For common parameter descriptions, see \ref CommonHistogramEvenParameters.
 */
NppStatus 
nppiHistogramEven_16s_C3R_Ctx(const Npp16s * pSrc, int nSrcStep, NppiSize oSizeROI, 
                                    Npp32s * pHist[3], 
                              int nLevels[3], Npp32s nLowerLevel[3], Npp32s nUpperLevel[3], Npp8u * pBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit signed HistogramEven.
 * For common parameter descriptions, see \ref CommonHistogramEvenParameters.
 */
NppStatus 
nppiHistogramEven_16s_C4R_Ctx(const Npp16s * pSrc, int nSrcStep, NppiSize oSizeROI, 
                                    Npp32s * pHist[4], 
                              int nLevels[4], Npp32s nLowerLevel[4], Npp32s nUpperLevel[4], Npp8u * pBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit signed HistogramEven ignoring alpha channel.
 * For common parameter descriptions, see \ref CommonHistogramEvenParameters.
 */
NppStatus 
nppiHistogramEven_16s_AC4R_Ctx(const Npp16s * pSrc, int nSrcStep, NppiSize oSizeROI, 
                                     Npp32s * pHist[3], 
                               int nLevels[3], Npp32s nLowerLevel[3], Npp32s nUpperLevel[3], Npp8u * pBuffer, NppStreamContext nppStreamCtx);

/** @} */
 
/** @name HistogramEvenGetBufferSize
 * Companion primitives for computing the device buffer size (in bytes) required by the HistogramEven primitives.
 * \section CommonHistogramEvenGetBufferSizeParameters Common parameters for nppiHistogramEvenGetBufferSize functions include: 
 * \param oSizeROI \ref roi_specification.
 * \param nLevels Number of levels in the histogram.
 * \param hpBufferSize Required buffer size. Important: hpBufferSize is a 
 *        <em>host pointer.</em> \ref general_scratch_buffer.
 * \param nppStreamCtx \ref application_managed_stream_context. 
 * \return NPP_NULL_POINTER_ERROR if hpBufferSize is 0 (NULL),  \ref roi_error_codes..
 * @{
 */

/**
 * Buffer size for \ref nppiHistogramEven_8u_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonHistogramEvenGetBufferSizeParameters.
 */
NppStatus 
nppiHistogramEvenGetBufferSize_8u_C1R_Ctx(NppiSize oSizeROI, int nLevels ,size_t * hpBufferSize, NppStreamContext nppStreamCtx);
/**
 * Buffer size for \ref nppiHistogramEven_8u_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonHistogramEvenGetBufferSizeParameters.
 */
NppStatus 
nppiHistogramEvenGetBufferSize_8u_C3R_Ctx(NppiSize oSizeROI, int nLevels[3] ,size_t * hpBufferSize, NppStreamContext nppStreamCtx);
/**
 * Buffer size for \ref nppiHistogramEven_8u_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonHistogramEvenGetBufferSizeParameters.
 */
NppStatus 
nppiHistogramEvenGetBufferSize_8u_C4R_Ctx(NppiSize oSizeROI, int nLevels[4] ,size_t * hpBufferSize, NppStreamContext nppStreamCtx);
/**
 * Buffer size for \ref nppiHistogramEven_8u_AC4R_Ctx.
 * For common parameter descriptions, see \ref CommonHistogramEvenGetBufferSizeParameters.
 */
NppStatus 
nppiHistogramEvenGetBufferSize_8u_AC4R_Ctx(NppiSize oSizeROI, int nLevels[3] ,size_t * hpBufferSize, NppStreamContext nppStreamCtx);
/**
 * Buffer size for \ref nppiHistogramEven_16u_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonHistogramEvenGetBufferSizeParameters.
 */
NppStatus 
nppiHistogramEvenGetBufferSize_16u_C1R_Ctx(NppiSize oSizeROI, int nLevels ,size_t * hpBufferSize, NppStreamContext nppStreamCtx);
/**
 * Buffer size for \ref nppiHistogramEven_16u_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonHistogramEvenGetBufferSizeParameters.
 */
NppStatus 
nppiHistogramEvenGetBufferSize_16u_C3R_Ctx(NppiSize oSizeROI, int nLevels[3] , size_t * hpBufferSize, NppStreamContext nppStreamCtx);
/**
 * Buffer size for \ref nppiHistogramEven_16u_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonHistogramEvenGetBufferSizeParameters.
 */
NppStatus 
nppiHistogramEvenGetBufferSize_16u_C4R_Ctx(NppiSize oSizeROI, int nLevels[4] ,size_t * hpBufferSize, NppStreamContext nppStreamCtx);
/**
 * Buffer size for \ref nppiHistogramEven_16u_AC4R_Ctx.
 * For common parameter descriptions, see \ref CommonHistogramEvenGetBufferSizeParameters.
 */
NppStatus 
nppiHistogramEvenGetBufferSize_16u_AC4R_Ctx(NppiSize oSizeROI, int nLevels[3] , size_t * hpBufferSize, NppStreamContext nppStreamCtx);
/**
 * Buffer size for \ref nppiHistogramEven_16s_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonHistogramEvenGetBufferSizeParameters.
 */
NppStatus 
nppiHistogramEvenGetBufferSize_16s_C1R_Ctx(NppiSize oSizeROI, int nLevels ,size_t * hpBufferSize, NppStreamContext nppStreamCtx);
/**
 * Buffer size for \ref nppiHistogramEven_16s_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonHistogramEvenGetBufferSizeParameters.
 */
NppStatus 
nppiHistogramEvenGetBufferSize_16s_C3R_Ctx(NppiSize oSizeROI, int nLevels[3] ,size_t * hpBufferSize, NppStreamContext nppStreamCtx);
/**
 * Buffer size for \ref nppiHistogramEven_16s_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonHistogramEvenGetBufferSizeParameters.
 */
NppStatus 
nppiHistogramEvenGetBufferSize_16s_C4R_Ctx(NppiSize oSizeROI, int nLevels[4] ,size_t * hpBufferSize, NppStreamContext nppStreamCtx);
/**
 * Buffer size for \ref nppiHistogramEven_16s_AC4R_Ctx.
 * For common parameter descriptions, see \ref CommonHistogramEvenGetBufferSizeParameters.
 */
NppStatus 
nppiHistogramEvenGetBufferSize_16s_AC4R_Ctx(NppiSize oSizeROI, int nLevels[3] ,size_t * hpBufferSize, NppStreamContext nppStreamCtx);

/** @} */

/** @} image_histogrameven */


/** 
 * \section image_histogramrange HistogramRange
 * @defgroup image_histogramrange HistogramRange
 * Primitives for computing the histogram of an image within specified ranges.
 * @{
 *
 */

/** @name HistogramRange
 * The histogram is computed according to the ranges provided in \f$pLevels\f$. 
 * The histogram \f$pHist[k]\f$ is defined as the total number of pixels that fall into the range:
 * \f$pLevels[k] <= pSrc(j, i) < pLevels[k+1]\f$. The number of the histogram bins is \f$nLevel - 1\f$.
 * The functions require additional scratch buffer for computations.
 * \section CommonHistogramRangeParameters Common parameters for nppiHistogramRange functions include:
 * \param pSrc \ref source_image_pointer.
 * \param nSrcStep \ref source_image_line_step.
 * \param oSizeROI \ref roi_specification.
 * \param pHist Pointer to array that receives the computed histogram. 
 *      The array must be of size nLevels-1. 
 * \param pLevels Pointer to array containing the level sizes of the bins.
        The array must be of size nLevels.
 * \param nLevels Number of levels in histogram.
 * \param pBuffer Pointer to appropriately sized (nppiHistogramRangeGetBufferSize_XX_XXX) 
 *      scratch buffer.
 * \param nppStreamCtx \ref application_managed_stream_context. 
 * \return \ref image_data_error_codes, \ref roi_error_codes
 * @{
 */
/**
 * One-channel 8-bit unsigned HistogramRange.
 * For common parameter descriptions, see \ref CommonHistogramEvenParameters.
 */
NppStatus 
nppiHistogramRange_8u_C1R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp32s * pHist, 
                              const Npp32s * pLevels, int nLevels, Npp8u * pBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 8-bit unsigned HistogramRange.
 * For common parameter descriptions, see \ref CommonHistogramEvenParameters.
 */
NppStatus 
nppiHistogramRange_8u_C3R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp32s * pHist[3], 
                              const Npp32s * pLevels[3], int nLevels[3], Npp8u * pBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned HistogramRange.
 * For common parameter descriptions, see \ref CommonHistogramEvenParameters.
 */
NppStatus 
nppiHistogramRange_8u_C4R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp32s * pHist[4], 
                              const Npp32s * pLevels[4], int nLevels[4], Npp8u * pBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned HistogramRange ignoring alpha channel.
 * For common parameter descriptions, see \ref CommonHistogramEvenParameters.
 */
NppStatus 
nppiHistogramRange_8u_AC4R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp32s * pHist[3], 
                               const Npp32s * pLevels[3], int nLevels[3], Npp8u * pBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 16-bit unsigned HistogramRange.
 * For common parameter descriptions, see \ref CommonHistogramEvenParameters.
 */
NppStatus 
nppiHistogramRange_16u_C1R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp32s * pHist, 
                               const Npp32s * pLevels, int nLevels, Npp8u * pBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 16-bit unsigned HistogramRange.
 * For common parameter descriptions, see \ref CommonHistogramEvenParameters.
 */
NppStatus 
nppiHistogramRange_16u_C3R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp32s * pHist[3], 
                               const Npp32s * pLevels[3], int nLevels[3], Npp8u * pBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit unsigned HistogramRange.
 * For common parameter descriptions, see \ref CommonHistogramEvenParameters.
 */
NppStatus 
nppiHistogramRange_16u_C4R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp32s * pHist[4], 
                               const Npp32s * pLevels[4], int nLevels[4], Npp8u * pBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit unsigned HistogramRange ignoring alpha channel.
 * For common parameter descriptions, see \ref CommonHistogramEvenParameters.
 */
NppStatus 
nppiHistogramRange_16u_AC4R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSizeROI, Npp32s * pHist[3], 
                                const Npp32s * pLevels[3], int nLevels[3], Npp8u * pBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 16-bit signed HistogramRange.
 * For common parameter descriptions, see \ref CommonHistogramEvenParameters.
 */
NppStatus 
nppiHistogramRange_16s_C1R_Ctx(const Npp16s * pSrc, int nSrcStep, NppiSize oSizeROI, Npp32s * pHist, 
                               const Npp32s * pLevels, int nLevels, Npp8u * pBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 16-bit signed HistogramRange.
 * For common parameter descriptions, see \ref CommonHistogramEvenParameters.
 */
NppStatus 
nppiHistogramRange_16s_C3R_Ctx(const Npp16s * pSrc, int nSrcStep, NppiSize oSizeROI, Npp32s * pHist[3], 
                               const Npp32s * pLevels[3], int nLevels[3], Npp8u * pBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit signed HistogramRange.
 * For common parameter descriptions, see \ref CommonHistogramEvenParameters.
 */
NppStatus 
nppiHistogramRange_16s_C4R_Ctx(const Npp16s * pSrc, int nSrcStep, NppiSize oSizeROI, Npp32s * pHist[4], 
                               const Npp32s * pLevels[4], int nLevels[4], Npp8u * pBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit signed HistogramRange.
 * For common parameter descriptions, see \ref CommonHistogramEvenParameters.
 */
NppStatus 
nppiHistogramRange_16s_AC4R_Ctx(const Npp16s * pSrc, int nSrcStep, NppiSize oSizeROI, Npp32s * pHist[3], 
                                const Npp32s * pLevels[3], int nLevels[3], Npp8u * pBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 32-bit floating point HistogramRange.
 * For common parameter descriptions, see \ref CommonHistogramEvenParameters.
 */
NppStatus 
nppiHistogramRange_32f_C1R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSizeROI, Npp32s * pHist, 
                               const Npp32f * pLevels, int nLevels, Npp8u * pBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 32-bit floating point HistogramRange.
 * For common parameter descriptions, see \ref CommonHistogramEvenParameters.
 */
NppStatus 
nppiHistogramRange_32f_C3R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSizeROI, Npp32s * pHist[3], 
                               const Npp32f * pLevels[3], int nLevels[3], Npp8u * pBuffer, NppStreamContext nppStreamCtx);

/**
 * Four-channel 32-bit floating point HistogramRange.
 * For common parameter descriptions, see \ref CommonHistogramEvenParameters.
 */
NppStatus 
nppiHistogramRange_32f_C4R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSizeROI, Npp32s * pHist[4], 
                               const Npp32f * pLevels[4], int nLevels[4], Npp8u * pBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit floating point HistogramRange ignoring alpha channel.
 * For common parameter descriptions, see \ref CommonHistogramEvenParameters.
 */
NppStatus 
nppiHistogramRange_32f_AC4R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSizeROI, Npp32s * pHist[3], 
                                const Npp32f * pLevels[3], int nLevels[3], Npp8u * pBuffer, NppStreamContext nppStreamCtx);

/** @} */

/** @name HistogramRangeGetBufferSize
 * Companion primitives for computing the device buffer size (in bytes) required by the HistogramRange primitives.
 * @{
 */
 
/**
 * Scratch-buffer size for nppiHistogramRange_8u_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonHistogramEvenGetBufferSizeParameters.
 */
NppStatus 
nppiHistogramRangeGetBufferSize_8u_C1R_Ctx(NppiSize oSizeROI, int nLevels ,size_t * hpBufferSize, NppStreamContext nppStreamCtx);
/**
 * Scratch-buffer size for nppiHistogramRange_8u_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonHistogramEvenGetBufferSizeParameters.
 */
NppStatus 
nppiHistogramRangeGetBufferSize_8u_C3R_Ctx(NppiSize oSizeROI, int nLevels[3] ,size_t * hpBufferSize, NppStreamContext nppStreamCtx);
/**
 * Scratch-buffer size for nppiHistogramRange_8u_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonHistogramEvenGetBufferSizeParameters.
 */
NppStatus 
nppiHistogramRangeGetBufferSize_8u_C4R_Ctx(NppiSize oSizeROI, int nLevels[4] ,size_t * hpBufferSize, NppStreamContext nppStreamCtx);
/**
 * Scratch-buffer size for nppiHistogramRange_8u_AC4R_Ctx.
 * For common parameter descriptions, see \ref CommonHistogramEvenGetBufferSizeParameters.
 */
NppStatus 
nppiHistogramRangeGetBufferSize_8u_AC4R_Ctx(NppiSize oSizeROI, int nLevels[3] ,size_t * hpBufferSize, NppStreamContext nppStreamCtx);
/**
 * Scratch-buffer size for nppiHistogramRange_16u_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonHistogramEvenGetBufferSizeParameters.
 */
NppStatus 
nppiHistogramRangeGetBufferSize_16u_C1R_Ctx(NppiSize oSizeROI, int nLevels ,size_t * hpBufferSize, NppStreamContext nppStreamCtx);
/**
 * Scratch-buffer size for nppiHistogramRange_16u_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonHistogramEvenGetBufferSizeParameters.
 */
NppStatus 
nppiHistogramRangeGetBufferSize_16u_C3R_Ctx(NppiSize oSizeROI, int nLevels[3], size_t * hpBufferSize, NppStreamContext nppStreamCtx);
/**
 * Scratch-buffer size for nppiHistogramRange_16u_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonHistogramEvenGetBufferSizeParameters.
 */
NppStatus 
nppiHistogramRangeGetBufferSize_16u_C4R_Ctx(NppiSize oSizeROI, int nLevels[4], size_t * hpBufferSize, NppStreamContext nppStreamCtx);
/**
 * Scratch-buffer size for nppiHistogramRange_16u_AC4R_Ctx.
 * For common parameter descriptions, see \ref CommonHistogramEvenGetBufferSizeParameters.
 */
NppStatus 
nppiHistogramRangeGetBufferSize_16u_AC4R_Ctx(NppiSize oSizeROI, int nLevels[3], size_t * hpBufferSize, NppStreamContext nppStreamCtx);
/**
 * Scratch-buffer size for nppiHistogramRange_16s_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonHistogramEvenGetBufferSizeParameters.
 */
NppStatus 
nppiHistogramRangeGetBufferSize_16s_C1R_Ctx(NppiSize oSizeROI, int nLevels, size_t * hpBufferSize, NppStreamContext nppStreamCtx);
/**
 * Scratch-buffer size for nppiHistogramRange_16s_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonHistogramEvenGetBufferSizeParameters.
 */
NppStatus 
nppiHistogramRangeGetBufferSize_16s_C3R_Ctx(NppiSize oSizeROI, int nLevels[3], size_t * hpBufferSize, NppStreamContext nppStreamCtx);
/**
 * Scratch-buffer size for nppiHistogramRange_16s_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonHistogramEvenGetBufferSizeParameters.
 */
NppStatus 
nppiHistogramRangeGetBufferSize_16s_C4R_Ctx(NppiSize oSizeROI, int nLevels[4] ,size_t * hpBufferSize, NppStreamContext nppStreamCtx);
/**
 * Scratch-buffer size for nppiHistogramRange_16s_AC4R_Ctx.
 * For common parameter descriptions, see \ref CommonHistogramEvenGetBufferSizeParameters.
 */
NppStatus 
nppiHistogramRangeGetBufferSize_16s_AC4R_Ctx(NppiSize oSizeROI, int nLevels[3], size_t * hpBufferSize, NppStreamContext nppStreamCtx);
/**
 * Scratch-buffer size for nppiHistogramRange_32f_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonHistogramEvenGetBufferSizeParameters.
 */
NppStatus 
nppiHistogramRangeGetBufferSize_32f_C1R_Ctx(NppiSize oSizeROI, int nLevels, size_t * hpBufferSize, NppStreamContext nppStreamCtx);
/**
 * Scratch-buffer size for nppiHistogramRange_32f_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonHistogramEvenGetBufferSizeParameters.
 */
NppStatus 
nppiHistogramRangeGetBufferSize_32f_C3R_Ctx(NppiSize oSizeROI, int nLevels[3], size_t * hpBufferSize, NppStreamContext nppStreamCtx);
/**
 * Scratch-buffer size for nppiHistogramRange_32f_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonHistogramEvenGetBufferSizeParameters.
 */
NppStatus 
nppiHistogramRangeGetBufferSize_32f_C4R_Ctx(NppiSize oSizeROI, int nLevels[4], size_t * hpBufferSize, NppStreamContext nppStreamCtx);
/**
 * Scratch-buffer size for nppiHistogramRange_32f_AC4R_Ctx.
 * For common parameter descriptions, see \ref CommonHistogramEvenGetBufferSizeParameters.
 */
NppStatus 
nppiHistogramRangeGetBufferSize_32f_AC4R_Ctx(NppiSize oSizeROI, int nLevels[3], size_t * hpBufferSize, NppStreamContext nppStreamCtx);

/** @} */
/** @} image_histogramrange */

/** 
 * \section image_proximity Image Proximity
 * @defgroup image_proximity Image Proximity
 * Primitives for computing the proximity measure between a source image and a template image.
 *
 * \subsection general_introduction General Introduction
 * There are basically two approaches to compute the proximity measure for template matching, 
 * Euclidean distance and the cross correlation.
 * -# Euclidean distance computes the sum of the squared distance (SSD) between the corresponding 
 * pixels of the source image and the template image. The smaller the distance is, the more 
 * similar the source image and the template image is around the pixel. The anchor of the template  
 * image is used during the computations, which always lies in the geometric center of the image. 
 * Given a source image \f$pSrc\f$ (\f$W_s \times H_s\f$) and a template image \f$pTpl\f$ (\f$W_t \times H_t\f$), 
 * the Euclidean distance \f$D_{st}(c,r)\f$ between two images at pixel in row \f$r\f$ and column 
 * \f$c\f$ is computed as (\f$s\f$ stands for source image and \f$t\f$ for template image for short):
 * \f[D_{st}(c,r)=\sum_{j=0}^{H_t-1}\sum_{i=0}^{W_t-1}[pTpl(j,i)-pSrc(j+c-\frac{H_t}{2}, i+r-\frac{W_t}{2})]^2 \f]
 * -# Cross correlation computes the sum of the product between the corresponding pixels of the source 
 * image and the template image. The cross correlation \f$R_{st}(c,r)\f$ is calculated as:
 * \f[R_{st}(c,r)=\sum_{j=0}^{H_t-1}\sum_{i=0}^{W_t-1}[pTpl(j,i)\cdot pSrc(j+c-\frac{H_t}{2}, i+r-\frac{W_t}{2})] \f]
 * The larger the cross correlation value is, the more similar the source image and the template image is around the pixel. 
 * -# The cross correlation \f$R_{st}(c,r)\f$ is affected by the brightness of the images which may vary due to the lighting 
 * and exposure conditions. Therefore, NPP computes the cross correlation coefficient to circumvent this dependence. 
 * This is typically done at every step by subtracting the mean from every pixel value, i.e.,
 * \f[\tilde{R}_{st}(c,r)=\sum_{j=0}^{H_t-1}\sum_{i=0}^{W_t-1}[pTpl(j,i)-Mean_t]\cdot [pSrc(j+c-\frac{H_t}{2}, i+r-\frac{W_t}{2})-Mean_s] \f]
 * 
 * NPP computes the normalized values of Euclidean distance, cross correlation and the cross correlation coefficient.
 * -# The normalized Euclidean distance \f$\sigma_{st}(c,r)\f$ is defined as:
 * \f[\sigma_{st}(c,r) = \frac{D_{st}(c,r)}{\sqrt{R_{ss}(c,r)\cdot R_{tt}(\frac{H_t}{2},\frac{W_t}{2})}} \f]
 * -# The normalized cross correlation \f$\rho_{st}(c,r)\f$ is defined as:
 * \f[\rho_{st}(c,r) = \frac{R_{st}(c,r)}{\sqrt{R_{ss}(c,r)\cdot R_{tt}(\frac{H_t}{2},\frac{W_t}{2})}} \f]
 * The \f$R_{ss}(c,r)\f$ and \f$R_{tt}(\frac{H_t}{2}, \frac{W_t}{2}\f$ denote the auto correlation of the source
 * image and the template image individually. They are defined as:
 * \f[R_{ss}(c,r)=\sum_{j=c-\frac{H_t}{2}}^{c+\frac{H_t}{2}}\sum_{i=r-\frac{W_t}{2}}^{r+\frac{W_t}{2}}pSrc(j, i) \f]
 * \f[R_{tt}(\frac{H_t}{2},\frac{W_t}{2})=\sum_{j=0}^{H_t-1}\sum_{i=0}^{W_t-1}pTpl(j,i) \f]
 * -# Similarly, the normalized cross correlation coefficient \f$\gamma_{st}(c,r)\f$ is calculated as:
 * \f[\gamma_{st}(c,r) = \frac{\tilde{R}_{st}(c,r)}{\sqrt{\tilde{R}_{ss}(c,r)\cdot \tilde{R}_{tt}(\frac{H_t}{2},\frac{W_t}{2})}} \f]
 * The \f$\tilde{R}_{ss}(c,r)\f$ and \f$\tilde{R}_{tt}(\frac{H_t}{2}, \frac{W_t}{2}\f$ are defined as:
 * \f[\tilde{R}_{ss}(c,r)=\sum_{j=c-\frac{H_t}{2}}^{c+\frac{H_t}{2}}\sum_{i=r-\frac{W_t}{2}}^{r+\frac{W_t}{2}}[pSrc(j, i)-Mean_s] \f]
 * \f[\tilde{R}_{tt}(\frac{H_t}{2},\frac{W_t}{2})=\sum_{j=0}^{H_t-1}\sum_{i=0}^{W_t-1}[pTpl(j,i)-Mean_t] \f] 
 * where \f$Mean_t\f$ is the template mean minus the mean of the image in the region just under the template. 
 *
 * \subsection category Categorizations
 * The Euclidean distance and the cross correlation are categorized into three types, full, same, and valid. 
 * -# Full mode indicates that the anchor of the template image starts from the outside of the source image, 
 * assuming the out-of-boundary pixels are zeor-padded. The size of the destination
 * image is \f$(W_s + W_t - 1) \times (H_s + H_t - 1)\f$.
 * -# Same mode means that the anchor of the template image starts from the top left pixel
 * of the source image. All the out-of-boundary pixels are also zero-padded.
 * The size of the destination image is the same as the source one, i.e., \f$W_s \times H_s\f$.
 * -# Valid mode indicates that there are no out-of-boudnary readings from the source image.
 * The anchor of the template image starts from the inside of the source image. The 
 * size of the destination image is \f$(W_s - W_t + 1) \times (H_s - H_t + 1)\f$.
 * 
 * @{
 */

/** 
 * \section sqrdistancefullnorm SqrDistanceFull_Norm 
 * @defgroup sqrdistancefullnorm SqrDistanceFull_Norm
 * Primitives for computing the normalized Euclidean distance between two images with full mode.
 * @{
 */
/**
 * @name SqrDistanceFull_Norm
 * The functions compute the \f$\sigma_{st}(c,r)\f$ in \ref general_introduction
 * with full mode (see \ref category).
 * \section CommonSqrDistanceFullNormParameters Common parameters for nppiSqrDistanceFull functions include:
 * \param pSrc \ref source_image_pointer.
 * \param nSrcStep \ref source_image_line_step.
 * \param oSrcRoiSize \ref roi_specification.
 * \param pTpl Pointer to the template image.
 * \param nTplStep Number of bytes between successive rows in the template image.
 * \param oTplRoiSize \ref roi_specification.
 * \param pDst \ref destination_image_pointer.
 * \param nDstStep \ref destination_image_line_step.
 * \param nScaleFactor \ref integer_result_scaling.
 * \param nppStreamCtx \ref application_managed_stream_context. 
 * \return \ref image_data_error_codes, \ref roi_error_codes
 * @{
 */
 /**
 * One-channel 8-bit unsigned image SqrDistanceFull_Norm, scaled by \f$2^(-nScaleFactor)\f$. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiSqrDistanceFull_Norm_8u_C1RSfs_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                       const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                             Npp8u * pDst, int nDstStep, int nScaleFactor, NppStreamContext nppStreamCtx);
/**
 * Three-channel 8-bit unsigned image SqrDistanceFull_Norm, scaled by \f$2^(-nScaleFactor)\f$. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiSqrDistanceFull_Norm_8u_C3RSfs_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                       const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                             Npp8u * pDst, int nDstStep, int nScaleFactor, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image SqrDistanceFull_Norm, scaled by \f$2^(-nScaleFactor)\f$. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiSqrDistanceFull_Norm_8u_C4RSfs_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                       const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                             Npp8u * pDst, int nDstStep, int nScaleFactor, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image SqrDistanceFull_Norm ignoring alpha channel, scaled by \f$2^(-nScaleFactor)\f$. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiSqrDistanceFull_Norm_8u_AC4RSfs_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                        const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                              Npp8u * pDst, int nDstStep, int nScaleFactor, NppStreamContext nppStreamCtx);
/**
 * One-channel 32-bit floating point image SqrDistanceFull_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiSqrDistanceFull_Norm_32f_C1R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                     const Npp32f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                           Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Three-channel 32-bit floating point image SqrDistanceFull_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiSqrDistanceFull_Norm_32f_C3R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                     const Npp32f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                           Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit floating point image SqrDistanceFull_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiSqrDistanceFull_Norm_32f_C4R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                     const Npp32f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                           Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit floating point image SqrDistanceFull_Norm ignoring alpha channel. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiSqrDistanceFull_Norm_32f_AC4R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                      const Npp32f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                            Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * One-channel 8-bit unsigned image SqrDistanceFull_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiSqrDistanceFull_Norm_8u32f_C1R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                       const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                             Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Three-channel 8-bit unsigned image SqrDistanceFull_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiSqrDistanceFull_Norm_8u32f_C3R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                       const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                             Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image SqrDistanceFull_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiSqrDistanceFull_Norm_8u32f_C4R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                       const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                             Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image SqrDistanceFull_Norm ignoring alpha channel. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiSqrDistanceFull_Norm_8u32f_AC4R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                        const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                              Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * One-channel 8-bit signed image SqrDistanceFull_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiSqrDistanceFull_Norm_8s32f_C1R_Ctx(const Npp8s * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                       const Npp8s * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                             Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Three-channel 8-bit signed image SqrDistanceFull_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiSqrDistanceFull_Norm_8s32f_C3R_Ctx(const Npp8s * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                       const Npp8s * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                             Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit signed image SqrDistanceFull_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiSqrDistanceFull_Norm_8s32f_C4R_Ctx(const Npp8s * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                       const Npp8s * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                             Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit signed image SqrDistanceFull_Norm ignoring alpha channel. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiSqrDistanceFull_Norm_8s32f_AC4R_Ctx(const Npp8s * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                        const Npp8s * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                              Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * One-channel 16-bit unsigned image SqrDistanceFull_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiSqrDistanceFull_Norm_16u32f_C1R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                        const Npp16u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                              Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Three-channel 16-bit unsigned image SqrDistanceFull_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiSqrDistanceFull_Norm_16u32f_C3R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                        const Npp16u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                              Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit unsigned image SqrDistanceFull_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiSqrDistanceFull_Norm_16u32f_C4R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                        const Npp16u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                              Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit unsigned image SqrDistanceFull_Norm ignoring alpha channel. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiSqrDistanceFull_Norm_16u32f_AC4R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                         const Npp16u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                               Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);

/** @} */
/** @} SqrDistanceFull_Norm */

/**
 * \section sqrdistancesamenorm SqrDistanceSame_Norm
 * @defgroup sqrdistancesamenorm SqrDistanceSame_Norm
 * Primitives for computing the normalized Euclidean distance between two images with same mode.
 * @{
 */
/**
 * @name SqrDistanceSame_Norm
 * The functions compute the \f$\sigma_{st}(c,r)\f$ in \ref general_introduction 
 * with same mode (see \ref category).
 * @{
 */
/**
 * One-channel 8-bit unsigned image SqrDistanceSame_Norm, scaled by \f$2^(-nScaleFactor)\f$. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiSqrDistanceSame_Norm_8u_C1RSfs_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                       const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                             Npp8u * pDst, int nDstStep, int nScaleFactor, NppStreamContext nppStreamCtx);
/**
 * Three-channel 8-bit unsigned image SqrDistanceSame_Norm, scaled by \f$2^(-nScaleFactor)\f$. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiSqrDistanceSame_Norm_8u_C3RSfs_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                       const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                             Npp8u * pDst, int nDstStep, int nScaleFactor, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image SqrDistanceSame_Norm, scaled by \f$2^(-nScaleFactor)\f$. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiSqrDistanceSame_Norm_8u_C4RSfs_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                       const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                             Npp8u * pDst, int nDstStep, int nScaleFactor, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image SqrDistanceSame_Norm ignoring alpha channel, scaled by \f$2^(-nScaleFactor)\f$. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiSqrDistanceSame_Norm_8u_AC4RSfs_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                        const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                              Npp8u * pDst, int nDstStep, int nScaleFactor, NppStreamContext nppStreamCtx);
/**
 * One-channel 32-bit floating point image SqrDistanceSame_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiSqrDistanceSame_Norm_32f_C1R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                     const Npp32f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                           Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Three-channel 32-bit floating point image SqrDistanceSame_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiSqrDistanceSame_Norm_32f_C3R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                     const Npp32f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                           Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit floating point image SqrDistanceSame_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiSqrDistanceSame_Norm_32f_C4R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                     const Npp32f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                           Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit floating point image SqrDistanceSame_Norm ignoring alpha channel. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiSqrDistanceSame_Norm_32f_AC4R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                      const Npp32f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                            Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * One-channel 8-bit unsigned image SqrDistanceSame_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiSqrDistanceSame_Norm_8u32f_C1R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                       const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                             Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Three-channel 8-bit unsigned image SqrDistanceSame_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiSqrDistanceSame_Norm_8u32f_C3R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                       const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                             Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image SqrDistanceSame_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiSqrDistanceSame_Norm_8u32f_C4R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                       const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                             Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image SqrDistanceSame_Norm ignoring alpha channel. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiSqrDistanceSame_Norm_8u32f_AC4R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                        const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                              Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * One-channel 8-bit signed image SqrDistanceSame_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiSqrDistanceSame_Norm_8s32f_C1R_Ctx(const Npp8s * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                       const Npp8s * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                             Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Three-channel 8-bit signed image SqrDistanceSame_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiSqrDistanceSame_Norm_8s32f_C3R_Ctx(const Npp8s * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                       const Npp8s * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                             Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit signed image SqrDistanceSame_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiSqrDistanceSame_Norm_8s32f_C4R_Ctx(const Npp8s * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                       const Npp8s * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                             Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit signed image SqrDistanceSame_Norm ignoring alpha channel. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiSqrDistanceSame_Norm_8s32f_AC4R_Ctx(const Npp8s * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                        const Npp8s * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                              Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * One-channel 16-bit unsigned image SqrDistanceSame_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiSqrDistanceSame_Norm_16u32f_C1R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                        const Npp16u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                              Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Three-channel 16-bit unsigned image SqrDistanceSame_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiSqrDistanceSame_Norm_16u32f_C3R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                        const Npp16u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                              Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit unsigned image SqrDistanceSame_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiSqrDistanceSame_Norm_16u32f_C4R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                        const Npp16u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                              Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit unsigned image SqrDistanceSame_Norm ignoring alpha channel. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiSqrDistanceSame_Norm_16u32f_AC4R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                         const Npp16u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                               Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit unsigned image SqrDistanceSame_Norm ignoring alpha channel. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiSqrDistanceSame_Norm_16u32f_AC4R(const Npp16u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                     const Npp16u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                           Npp32f * pDst, int nDstStep);

/** @} */
/** @} SqrDistanceFull_Norm */

/**
 * \section sqrdistancevalidnorm SqrDistanceValid_Norm
 * @defgroup sqrdistancevalidnorm SqrDistanceValid_Norm
 * Primitives for computing the normalized Euclidean distance between two images with valid mode.
 * @{
 */
/**
 * @name SqrDistanceValid_Norm
 * The functions compute the \f$\sigma_{st}(c,r)\f$ in \ref general_introduction 
 * with valid mode (see \ref category).
 * @{
 */
/**
 * One-channel 8-bit unsigned image SqrDistanceValid_Norm, scaled by \f$2^(-nScaleFactor)\f$. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiSqrDistanceValid_Norm_8u_C1RSfs_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                        const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                              Npp8u * pDst, int nDstStep, int nScaleFactor, NppStreamContext nppStreamCtx);
/**
 * Three-channel 8-bit unsigned image SqrDistanceValid_Norm, scaled by \f$2^(-nScaleFactor)\f$. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiSqrDistanceValid_Norm_8u_C3RSfs_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                        const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                              Npp8u * pDst, int nDstStep, int nScaleFactor, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image SqrDistanceValid_Norm, scaled by \f$2^(-nScaleFactor)\f$. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiSqrDistanceValid_Norm_8u_C4RSfs_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                        const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                              Npp8u * pDst, int nDstStep, int nScaleFactor, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image SqrDistanceValid_Norm ignoring alpha channel, scaled by \f$2^(-nScaleFactor)\f$. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiSqrDistanceValid_Norm_8u_AC4RSfs_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                         const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                               Npp8u * pDst, int nDstStep, int nScaleFactor, NppStreamContext nppStreamCtx);
/**
 * One-channel 32-bit floating point image SqrDistanceValid_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiSqrDistanceValid_Norm_32f_C1R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                      const Npp32f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                            Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Three-channel 32-bit floating point image SqrDistanceValid_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiSqrDistanceValid_Norm_32f_C3R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                      const Npp32f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                            Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit floating point image SqrDistanceValid_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiSqrDistanceValid_Norm_32f_C4R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                      const Npp32f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                            Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit floating point image SqrDistanceValid_Norm ignoring alpha channel. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiSqrDistanceValid_Norm_32f_AC4R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                       const Npp32f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                             Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * One-channel 8-bit unsigned image SqrDistanceValid_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiSqrDistanceValid_Norm_8u32f_C1R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                        const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                              Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Three-channel 8-bit unsigned image SqrDistanceValid_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiSqrDistanceValid_Norm_8u32f_C3R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                        const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                              Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image SqrDistanceValid_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */ 
NppStatus
nppiSqrDistanceValid_Norm_8u32f_C4R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                        const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                              Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image SqrDistanceValid_Norm ignoring alpha channel. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiSqrDistanceValid_Norm_8u32f_AC4R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                         const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                               Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * One-channel 8-bit signed image SqrDistanceValid_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiSqrDistanceValid_Norm_8s32f_C1R_Ctx(const Npp8s * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                        const Npp8s * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                              Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Three-channel 8-bit signed image SqrDistanceValid_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiSqrDistanceValid_Norm_8s32f_C3R_Ctx(const Npp8s * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                        const Npp8s * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                              Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit signed image SqrDistanceValid_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiSqrDistanceValid_Norm_8s32f_C4R_Ctx(const Npp8s * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                        const Npp8s * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                              Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit signed image SqrDistanceValid_Norm ignoring alpha channel. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiSqrDistanceValid_Norm_8s32f_AC4R_Ctx(const Npp8s * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                         const Npp8s * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                               Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * One-channel 16-bit unsigned image SqrDistanceValid_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiSqrDistanceValid_Norm_16u32f_C1R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                         const Npp16u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                               Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Three-channel 16-bit unsigned image SqrDistanceValid_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiSqrDistanceValid_Norm_16u32f_C3R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                         const Npp16u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                               Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit unsigned image SqrDistanceValid_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiSqrDistanceValid_Norm_16u32f_C4R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                         const Npp16u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                               Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit unsigned image SqrDistanceValid_Norm ignoring alpha channel. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiSqrDistanceValid_Norm_16u32f_AC4R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                          const Npp16u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);

/** @} */
/** @} SqrDistanceValid_Norm*/

/**
 * \section crosscorrfullnorm CrossCorrFull_Norm
 * @defgroup crosscorrfullnorm CrossCorrFull_Norm
 * Primitives for computing the normalized cross correlation between two images with full mode.
 *
 * @{
 */
/**
 * @name CrossCorrFull_Norm
 * The functions compute the \f$\rho_{st}(c,r)\f$ in \ref general_introduction 
 * with full mode (see \ref category).
 * @{
 */
/**
 * One-channel 8-bit unsigned image CrossCorrFull_Norm, scaled by \f$2^(-nScaleFactor)\f$. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrFull_Norm_8u_C1RSfs_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                     const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                           Npp8u * pDst, int nDstStep, int nScaleFactor, NppStreamContext nppStreamCtx);
/**
 * Three-channel 8-bit unsigned image CrossCorrFull_Norm, scaled by \f$2^(-nScaleFactor)\f$. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrFull_Norm_8u_C3RSfs_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                     const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                           Npp8u * pDst, int nDstStep, int nScaleFactor, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image CrossCorrFull_Norm, scaled by \f$2^(-nScaleFactor)\f$. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrFull_Norm_8u_C4RSfs_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                     const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                           Npp8u * pDst, int nDstStep, int nScaleFactor, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image CrossCorrFull_Norm ignoring alpha channel, scaled by \f$2^(-nScaleFactor)\f$. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrFull_Norm_8u_AC4RSfs_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                      const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                            Npp8u * pDst, int nDstStep, int nScaleFactor, NppStreamContext nppStreamCtx);
/**
 * One-channel 32-bit floating point image CrossCorrFull_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrFull_Norm_32f_C1R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                   const Npp32f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                         Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Three-channel 32-bit floating point image CrossCorrFull_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrFull_Norm_32f_C3R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                   const Npp32f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                         Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit floating point image CrossCorrFull_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrFull_Norm_32f_C4R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                   const Npp32f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                         Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit floating point image CrossCorrFull_Norm ignoring alpha channel. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrFull_Norm_32f_AC4R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                    const Npp32f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                          Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * One-channel 64-bit floating point image CrossCorrFull_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrFull_Norm_64f_C1R_Ctx(const Npp64f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                   const Npp64f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                   Npp64f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Three-channel 64-bit floating point image CrossCorrFull_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrFull_Norm_64f_C3R_Ctx(const Npp64f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                   const Npp64f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                         Npp64f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Four-channel 64-bit floating point image CrossCorrFull_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrFull_Norm_64f_C4R_Ctx(const Npp64f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                   const Npp64f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                         Npp64f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Four-channel 64-bit floating point image CrossCorrFull_Norm ignoring alpha channel. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrFull_Norm_64f_AC4R_Ctx(const Npp64f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                    const Npp64f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                          Npp64f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * One-channel 8-bit unsigned image CrossCorrFull_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrFull_Norm_8u32f_C1R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                     const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                           Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Three-channel 8-bit unsigned image CrossCorrFull_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrFull_Norm_8u32f_C3R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                     const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                           Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image CrossCorrFull_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrFull_Norm_8u32f_C4R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                     const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                           Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image CrossCorrFull_Norm ignoring alpha channel. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrFull_Norm_8u32f_AC4R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                      const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                            Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * One-channel 8-bit signed image CrossCorrFull_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrFull_Norm_8s32f_C1R_Ctx(const Npp8s * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                     const Npp8s * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                           Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Three-channel 8-bit signed image CrossCorrFull_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrFull_Norm_8s32f_C3R_Ctx(const Npp8s * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                     const Npp8s * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                           Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit signed image CrossCorrFull_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrFull_Norm_8s32f_C4R_Ctx(const Npp8s * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                     const Npp8s * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                           Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit signed image CrossCorrFull_Norm ignoring alpha channel. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrFull_Norm_8s32f_AC4R_Ctx(const Npp8s * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                      const Npp8s * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                            Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * One-channel 16-bit unsigned image CrossCorrFull_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrFull_Norm_16u32f_C1R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                      const Npp16u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                            Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Three-channel 16-bit unsigned image CrossCorrFull_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrFull_Norm_16u32f_C3R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                      const Npp16u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                            Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit unsigned image CrossCorrFull_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrFull_Norm_16u32f_C4R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                      const Npp16u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                            Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit unsigned image CrossCorrFull_Norm ignoring alpha channel. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrFull_Norm_16u32f_AC4R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                       const Npp16u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                             Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);

/** @} */
/** @} CrossCorrFull_Norm */

/**
 * \section crosscorrsamenorm CrossCorrSame_Norm
 * @defgroup crosscorrsamenorm CrossCorrSame_Norm
 * Primitives for computing the normalized cross correlation between two images with same mode.
 *
 * @{
 */
/**
 * @name CrossCorrSame_Norm
 * The functions compute the \f$\rho_{st}(c,r)\f$ in \ref general_introduction 
 * with same mode (see \ref category).
 * @{
 */
/**
 * One-channel 8-bit unsigned image CrossCorrSame_Norm, scaled by \f$2^(-nScaleFactor)\f$. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrSame_Norm_8u_C1RSfs_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                     const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                           Npp8u * pDst, int nDstStep, int nScaleFactor, NppStreamContext nppStreamCtx);
/**
 * Three-channel 8-bit unsigned image CrossCorrSame_Norm, scaled by \f$2^(-nScaleFactor)\f$. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrSame_Norm_8u_C3RSfs_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                     const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                           Npp8u * pDst, int nDstStep, int nScaleFactor, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image CrossCorrSame_Norm, scaled by \f$2^(-nScaleFactor)\f$. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrSame_Norm_8u_C4RSfs_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                     const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                           Npp8u * pDst, int nDstStep, int nScaleFactor, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image CrossCorrSame_Norm ignoring alpha channel, scaled by \f$2^(-nScaleFactor)\f$. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrSame_Norm_8u_AC4RSfs_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                      const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                            Npp8u * pDst, int nDstStep, int nScaleFactor, NppStreamContext nppStreamCtx);
/**
 * One-channel 32-bit floating point image CrossCorrSame_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrSame_Norm_32f_C1R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                   const Npp32f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                         Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Three-channel 32-bit floating point image CrossCorrSame_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrSame_Norm_32f_C3R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                   const Npp32f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                         Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit floating point image CrossCorrSame_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrSame_Norm_32f_C4R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                   const Npp32f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                         Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit floating point image CrossCorrSame_Norm ignoring alpha channel. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrSame_Norm_32f_AC4R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                    const Npp32f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                          Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * One-channel 64-bit floating point image CrossCorrSame_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrSame_Norm_64f_C1R_Ctx(const Npp64f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                   const Npp64f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                         Npp64f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Three-channel 64-bit floating point image CrossCorrSame_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrSame_Norm_64f_C3R_Ctx(const Npp64f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                   const Npp64f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                         Npp64f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Four-channel 64-bit floating point image CrossCorrSame_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrSame_Norm_64f_C4R_Ctx(const Npp64f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                   const Npp64f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                         Npp64f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Four-channel 64-bit floating point image CrossCorrSame_Norm ignoring alpha channel. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrSame_Norm_64f_AC4R_Ctx(const Npp64f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                    const Npp64f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                          Npp64f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * One-channel 8-bit unsigned image CrossCorrSame_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrSame_Norm_8u32f_C1R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                     const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                           Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Three-channel 8-bit unsigned image CrossCorrSame_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrSame_Norm_8u32f_C3R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                     const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                           Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image CrossCorrSame_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrSame_Norm_8u32f_C4R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                     const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                           Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image CrossCorrSame_Norm ignoring alpha channel. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrSame_Norm_8u32f_AC4R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                      const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                            Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * One-channel 8-bit signed image CrossCorrSame_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrSame_Norm_8s32f_C1R_Ctx(const Npp8s * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                     const Npp8s * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                           Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Three-channel 8-bit signed image CrossCorrSame_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrSame_Norm_8s32f_C3R_Ctx(const Npp8s * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                     const Npp8s * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                           Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit signed image CrossCorrSame_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrSame_Norm_8s32f_C4R_Ctx(const Npp8s * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                     const Npp8s * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                           Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit signed image CrossCorrSame_Norm ignoring alpha channel. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrSame_Norm_8s32f_AC4R_Ctx(const Npp8s * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                      const Npp8s * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                            Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * One-channel 16-bit unsigned image CrossCorrSame_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrSame_Norm_16u32f_C1R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                      const Npp16u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                            Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Three-channel 16-bit unsigned image CrossCorrSame_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrSame_Norm_16u32f_C3R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                      const Npp16u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                            Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit unsigned image CrossCorrSame_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrSame_Norm_16u32f_C4R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                      const Npp16u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                            Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit unsigned image CrossCorrSame_Norm ignoring alpha channel. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrSame_Norm_16u32f_AC4R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                       const Npp16u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                             Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);

/** @} */
/** @} CrossCorrSame_Norm*/

/**
 * \section crosscorrvalidnorm CrossCorrValid_Norm
 * @defgroup crosscorrvalidnorm CrossCorrValid_Norm
 * Primitives for computing the normalized cross correlation between two images with valid mode.
 *
 * @{
 */
/**
 * @name CrossCorrValid_Norm
 * The functions compute the \f$\rho_{st}(c,r)\f$ in \ref general_introduction 
 * with valid mode (see \ref category).
 * @{
 */
/**
 * One-channel 8-bit unsigned image CrossCorrValid_Norm, scaled by \f$2^(-nScaleFactor)\f$. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrValid_Norm_8u_C1RSfs_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                      const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                            Npp8u * pDst, int nDstStep, int nScaleFactor, NppStreamContext nppStreamCtx);
/**
 * Three-channel 8-bit unsigned image CrossCorrValid_Norm, scaled by \f$2^(-nScaleFactor)\f$. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrValid_Norm_8u_C3RSfs_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                      const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                            Npp8u * pDst, int nDstStep, int nScaleFactor, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image CrossCorrValid_Norm, scaled by \f$2^(-nScaleFactor)\f$. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrValid_Norm_8u_C4RSfs_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                      const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                            Npp8u * pDst, int nDstStep, int nScaleFactor, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image CrossCorrValid_Norm ignoring alpha channel, scaled by \f$2^(-nScaleFactor)\f$. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrValid_Norm_8u_AC4RSfs_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                       const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                             Npp8u * pDst, int nDstStep, int nScaleFactor, NppStreamContext nppStreamCtx);
/**
 * One-channel 32-bit floating point image CrossCorrValid_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrValid_Norm_32f_C1R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                    const Npp32f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                          Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Three-channel 32-bit floating point image CrossCorrValid_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrValid_Norm_32f_C3R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                    const Npp32f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                          Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit floating point image CrossCorrValid_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrValid_Norm_32f_C4R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                    const Npp32f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                          Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit floating point image CrossCorrValid_Norm ignoring alpha channel. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrValid_Norm_32f_AC4R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                     const Npp32f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                           Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * One-channel 64-bit floating point image CrossCorrValid_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrValid_Norm_64f_C1R_Ctx(const Npp64f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                    const Npp64f * pTpl, int nTplStep, NppiSize oTplRoiSize,
                                    Npp64f * pDst, int nDstStep, NppStreamContext nppStreamCtx); 
/**
 * Three-channel 64-bit floating point image CrossCorrValid_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrValid_Norm_64f_C3R_Ctx(const Npp64f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                    const Npp64f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                          Npp64f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Four-channel 64-bit floating point image CrossCorrValid_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrValid_Norm_64f_C4R_Ctx(const Npp64f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                    const Npp64f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                          Npp64f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Four-channel 64-bit floating point image CrossCorrValid_Norm ignoring alpha channel. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrValid_Norm_64f_AC4R_Ctx(const Npp64f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                     const Npp64f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                           Npp64f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * One-channel 8-bit unsigned image CrossCorrValid_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrValid_Norm_8u32f_C1R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                      const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                            Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Three-channel 8-bit unsigned image CrossCorrValid_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrValid_Norm_8u32f_C3R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                      const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                            Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image CrossCorrValid_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrValid_Norm_8u32f_C4R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                      const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                            Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image CrossCorrValid_Norm ignoring alpha channel. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrValid_Norm_8u32f_AC4R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                       const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                             Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * One-channel 8-bit signed image CrossCorrValid_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrValid_Norm_8s32f_C1R_Ctx(const Npp8s * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                      const Npp8s * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                            Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Three-channel 8-bit signed image CrossCorrValid_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrValid_Norm_8s32f_C3R_Ctx(const Npp8s * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                      const Npp8s * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                            Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit signed image CrossCorrValid_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrValid_Norm_8s32f_C4R_Ctx(const Npp8s * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                      const Npp8s * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                            Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit signed image CrossCorrValid_Norm ignoring alpha channel. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrValid_Norm_8s32f_AC4R_Ctx(const Npp8s * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                       const Npp8s * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                             Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * One-channel 16-bit unsigned image CrossCorrValid_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrValid_Norm_16u32f_C1R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                       const Npp16u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                             Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Three-channel 16-bit unsigned image CrossCorrValid_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrValid_Norm_16u32f_C3R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                       const Npp16u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                             Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit unsigned image CrossCorrValid_Norm. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrValid_Norm_16u32f_C4R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                       const Npp16u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                             Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit unsigned image CrossCorrValid_Norm ignoring alpha channel. 
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrValid_Norm_16u32f_AC4R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                        const Npp16u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                              Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);

/** @} */
/** @} CrossCorrValid_Norm*/

/**
 * \section crosscorrvalid CrossCorrValid
 * @defgroup crosscorrvalid CrossCorrValid
 * Primitives for computing the cross correlation between two images with valid mode.
 *
 * @{
 */
/**
 * @name CrossCorrValid
 * The functions compute the \f$R_{st}(c,r)\f$ in \ref general_introduction 
 * with valid mode (see \ref category).
 * @{
 */
/**
 * One-channel 32-bit floating point images CrossCorrValid.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrValid_32f_C1R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                               const Npp32f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                     Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * One-channel 64-bit floating point images CrossCorrValid.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrValid_64f_C1R_Ctx(const Npp64f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                               const Npp64f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                     Npp64f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * One-channel 8-bit unsigned images CrossCorrValid.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrValid_8u32f_C1R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                 const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                       Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * One-channel 8-bit signed images CrossCorrValid.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrValid_8s32f_C1R_Ctx(const Npp8s * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                 const Npp8s * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                       Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);
/**
 * One-channel 16-bit unsigned images CrossCorrValid.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrValid_16u32f_C1R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                  const Npp16u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                        Npp32f * pDst, int nDstStep, NppStreamContext nppStreamCtx);                                          

/** @} */
/** @} CrossCorrValid*/

/**
 * \section crosscorrfullnormlevel CrossCorrFull_NormLevel
 * @defgroup crosscorrfullnormlevel CrossCorrFull_NormLevel
 * Primitives for computing the normalized cross correlation coefficient between two images with full mode. 
 *
 * @{
 */
/**
 * @name CrossCorrFull_NormLevel
 * The functions compute the \f$\gamma_{st}(c,r)\f$ in \ref general_introduction 
 * with full mode (see \ref category). The functions require additional scratch buffer for computations. 
 *  
 * Note: For maximum performance oSrcRoiSize.width + oTplRoiSize.width - 1 MUST be an integer multiple of 4. 
 *  
 * @{
 */
/**
 * One-channel 8-bit unsigned image CrossCorrFull_NormLevel.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrFull_NormLevel_8u_C1RSfs_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                          const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                Npp8u * pDst, int nDstStep, int nScaleFactor, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 8-bit unsigned image CrossCorrFull_NormLevel.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrFull_NormLevel_8u_C3RSfs_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                          const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                Npp8u * pDst, int nDstStep, int nScaleFactor, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image CrossCorrFull_NormLevel.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrFull_NormLevel_8u_C4RSfs_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                          const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                Npp8u * pDst, int nDstStep, int nScaleFactor, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image CrossCorrFull_NormLevel ignoring alpha channel.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrFull_NormLevel_8u_AC4RSfs_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                           const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                 Npp8u * pDst, int nDstStep, int nScaleFactor, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 32-bit floating point image CrossCorrFull_NormLevel.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrFull_NormLevel_32f_C1R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                        const Npp32f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                              Npp32f * pDst, int nDstStep, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 32-bit floating point image CrossCorrFull_NormLevel.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrFull_NormLevel_32f_C3R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                        const Npp32f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                              Npp32f * pDst, int nDstStep, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit floating point image CrossCorrFull_NormLevel.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrFull_NormLevel_32f_C4R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                        const Npp32f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                              Npp32f * pDst, int nDstStep, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit floating point image CrossCorrFull_NormLevel ignoring alpha channel.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrFull_NormLevel_32f_AC4R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                         const Npp32f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                               Npp32f * pDst, int nDstStep, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 64-bit floating point image CrossCorrFull_NormLevel.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrFull_NormLevel_64f_C1R_Ctx(const Npp64f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                        const Npp64f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                              Npp64f * pDst, int nDstStep, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 64-bit floating point image CrossCorrFull_NormLevel.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrFull_NormLevel_64f_C3R_Ctx(const Npp64f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                        const Npp64f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                              Npp64f * pDst, int nDstStep, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 64-bit floating point image CrossCorrFull_NormLevel.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrFull_NormLevel_64f_C4R_Ctx(const Npp64f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                        const Npp64f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                              Npp64f * pDst, int nDstStep, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 64-bit floating point image CrossCorrFull_NormLevel ignoring alpha channel.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrFull_NormLevel_64f_AC4R_Ctx(const Npp64f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                         const Npp64f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                               Npp64f * pDst, int nDstStep, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 8-bit unsigned image CrossCorrFull_NormLevel.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrFull_NormLevel_8u32f_C1R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                          const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                Npp32f * pDst, int nDstStep, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 8-bit unsigned image CrossCorrFull_NormLevel.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrFull_NormLevel_8u32f_C3R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                          const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                Npp32f * pDst, int nDstStep, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image CrossCorrFull_NormLevel.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrFull_NormLevel_8u32f_C4R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                          const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                Npp32f * pDst, int nDstStep, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image CrossCorrFull_NormLevel ignoring alpha channel.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrFull_NormLevel_8u32f_AC4R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                           const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                 Npp32f * pDst, int nDstStep, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 8-bit signed image CrossCorrFull_NormLevel.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrFull_NormLevel_8s32f_C1R_Ctx(const Npp8s * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                          const Npp8s * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                          Npp32f * pDst, int nDstStep, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 8-bit signed image CrossCorrFull_NormLevel.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrFull_NormLevel_8s32f_C3R_Ctx(const Npp8s * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                          const Npp8s * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                Npp32f * pDst, int nDstStep, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit signed image CrossCorrFull_NormLevel.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrFull_NormLevel_8s32f_C4R_Ctx(const Npp8s * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                          const Npp8s * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                Npp32f * pDst, int nDstStep, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit signed image CrossCorrFull_NormLevel ignoring alpha channel.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrFull_NormLevel_8s32f_AC4R_Ctx(const Npp8s * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                           const Npp8s * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                 Npp32f * pDst, int nDstStep, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 16-bit unsigned image CrossCorrFull_NormLevel.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrFull_NormLevel_16u32f_C1R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                           const Npp16u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                 Npp32f * pDst, int nDstStep, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 16-bit unsigned image CrossCorrFull_NormLevel.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrFull_NormLevel_16u32f_C3R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                           const Npp16u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                 Npp32f * pDst, int nDstStep, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit unsigned image CrossCorrFull_NormLevel.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrFull_NormLevel_16u32f_C4R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                           const Npp16u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                 Npp32f * pDst, int nDstStep, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit unsigned image CrossCorrFull_NormLevel ignoring alpha channel.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrFull_NormLevel_16u32f_AC4R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                            const Npp16u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                  Npp32f * pDst, int nDstStep, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);

/** @} */

/** @name FullNormLevelGetBufferHostSize 
 * Companion primitives for computing the device buffer size (in bytes) required by the CrossCorrFull_NormLevel primitives. 
 *  
 * \section CommonFullNormLevelGetBufferHostSizeParameters
 * Common parameters for nppiFullNormLevelGetBufferHostSize functions include:
 * \param oSizeROI \ref roi_specification.
 * \param hpBufferSize Required buffer size. Important: hpBufferSize is a 
 *        <em>host pointer.</em> \ref general_scratch_buffer.
 * \param nppStreamCtx \ref application_managed_stream_context. 
 * \return NPP_NULL_POINTER_ERROR if hpBufferSize is 0 (NULL),  \ref roi_error_codes.  
 * @{
 */
/** 
 * Buffer size (in bytes) for nppiCrossCorrFull_NormLevel_8u_C1RSfs_Ctx.
 * For common parameter descriptions, see \ref CommonFullNormLevelGetBufferHostSizeParameters.
 */
NppStatus 
nppiFullNormLevelGetBufferHostSize_8u_C1RSfs_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size (in bytes) for \ref nppiCrossCorrFull_NormLevel_8u_C3RSfs_Ctx.
 * For common parameter descriptions, see \ref CommonFullNormLevelGetBufferHostSizeParameters.
 */
NppStatus 
nppiFullNormLevelGetBufferHostSize_8u_C3RSfs_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size (in bytes) for \ref nppiCrossCorrFull_NormLevel_8u_C4RSfs_Ctx.
 * For common parameter descriptions, see \ref CommonFullNormLevelGetBufferHostSizeParameters.
 */
NppStatus 
nppiFullNormLevelGetBufferHostSize_8u_C4RSfs_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size (in bytes) for \ref nppiCrossCorrFull_NormLevel_8u_AC4RSfs_Ctx.
 * For common parameter descriptions, see \ref CommonFullNormLevelGetBufferHostSizeParameters.
 */
NppStatus 
nppiFullNormLevelGetBufferHostSize_8u_AC4RSfs_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size (in bytes) for \ref nppiCrossCorrFull_NormLevel_32f_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonFullNormLevelGetBufferHostSizeParameters.
 */
NppStatus 
nppiFullNormLevelGetBufferHostSize_32f_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size (in bytes) for \ref nppiCrossCorrFull_NormLevel_32f_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonFullNormLevelGetBufferHostSizeParameters.
 */
NppStatus 
nppiFullNormLevelGetBufferHostSize_32f_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size (in bytes) for \ref nppiCrossCorrFull_NormLevel_32f_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonFullNormLevelGetBufferHostSizeParameters.
 */
NppStatus 
nppiFullNormLevelGetBufferHostSize_32f_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size (in bytes) for \ref nppiCrossCorrFull_NormLevel_32f_AC4R_Ctx.
 * For common parameter descriptions, see \ref CommonFullNormLevelGetBufferHostSizeParameters.
 */
NppStatus 
nppiFullNormLevelGetBufferHostSize_32f_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size (in bytes) for \ref nppiCrossCorrFull_NormLevel_64f_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonFullNormLevelGetBufferHostSizeParameters.
 */
NppStatus 
nppiFullNormLevelGetBufferHostSize_64f_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size (in bytes) for \ref nppiCrossCorrFull_NormLevel_64f_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonFullNormLevelGetBufferHostSizeParameters.
 */
NppStatus 
nppiFullNormLevelGetBufferHostSize_64f_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size (in bytes) for \ref nppiCrossCorrFull_NormLevel_64f_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonFullNormLevelGetBufferHostSizeParameters.
 */
NppStatus 
nppiFullNormLevelGetBufferHostSize_64f_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size (in bytes) for \ref nppiCrossCorrFull_NormLevel_64f_AC4R_Ctx.
 * For common parameter descriptions, see \ref CommonFullNormLevelGetBufferHostSizeParameters.
 */
NppStatus 
nppiFullNormLevelGetBufferHostSize_64f_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size (in bytes) for \ref nppiCrossCorrFull_NormLevel_8u32f_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonFullNormLevelGetBufferHostSizeParameters.
 */
NppStatus 
nppiFullNormLevelGetBufferHostSize_8u32f_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size (in bytes) for \ref nppiCrossCorrFull_NormLevel_8u32f_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonFullNormLevelGetBufferHostSizeParameters.
 */
NppStatus 
nppiFullNormLevelGetBufferHostSize_8u32f_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size (in bytes) for \ref nppiCrossCorrFull_NormLevel_8u32f_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonFullNormLevelGetBufferHostSizeParameters.
 */
NppStatus 
nppiFullNormLevelGetBufferHostSize_8u32f_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size (in bytes) for \ref nppiCrossCorrFull_NormLevel_8u32f_AC4R_Ctx.
 * For common parameter descriptions, see \ref CommonFullNormLevelGetBufferHostSizeParameters.
 */
NppStatus 
nppiFullNormLevelGetBufferHostSize_8u32f_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size (in bytes) for \ref nppiCrossCorrFull_NormLevel_8s32f_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonFullNormLevelGetBufferHostSizeParameters.
 */
NppStatus 
nppiFullNormLevelGetBufferHostSize_8s32f_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size (in bytes) for \ref nppiCrossCorrFull_NormLevel_8s32f_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonFullNormLevelGetBufferHostSizeParameters.
 */
NppStatus 
nppiFullNormLevelGetBufferHostSize_8s32f_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size (in bytes) for \ref nppiCrossCorrFull_NormLevel_8s32f_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonFullNormLevelGetBufferHostSizeParameters.
 */
NppStatus 
nppiFullNormLevelGetBufferHostSize_8s32f_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size (in bytes) for \ref nppiCrossCorrFull_NormLevel_8s32f_AC4R_Ctx.
 * For common parameter descriptions, see \ref CommonFullNormLevelGetBufferHostSizeParameters.
 */
NppStatus 
nppiFullNormLevelGetBufferHostSize_8s32f_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size (in bytes) for \ref nppiCrossCorrFull_NormLevel_16u32f_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonFullNormLevelGetBufferHostSizeParameters.
 */
NppStatus 
nppiFullNormLevelGetBufferHostSize_16u32f_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size (in bytes) for \ref nppiCrossCorrFull_NormLevel_16u32f_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonFullNormLevelGetBufferHostSizeParameters.
 */
NppStatus 
nppiFullNormLevelGetBufferHostSize_16u32f_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size (in bytes) for \ref nppiCrossCorrFull_NormLevel_16u32f_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonFullNormLevelGetBufferHostSizeParameters.
 */
NppStatus 
nppiFullNormLevelGetBufferHostSize_16u32f_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size (in bytes) for \ref nppiCrossCorrFull_NormLevel_16u32f_AC4R_Ctx.
 * For common parameter descriptions, see \ref CommonFullNormLevelGetBufferHostSizeParameters.
 */
NppStatus 
nppiFullNormLevelGetBufferHostSize_16u32f_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);

/** @} */
/** @} CrossCorrFull_NormLevel*/

/**
 * \section crosscorrsamenormlevel CrossCorrSame_NormLevel
 * @defgroup crosscorrsamenormlevel CrossCorrSame_NormLevel
 * Primitives for computing the normalized cross correlation coefficient between two images with same mode.
 *
 * @{
 */
/**
 * @name CrossCorrSame_NormLevel
 * The functions compute the \f$\gamma_{st}(c,r)\f$ in \ref general_introduction 
 * with same mode (see \ref category). The functions require additional scratch buffer for computations. 
 *  
 * Note: For maximum performance oSrcRoiSize.width MUST be an integer multiple of 4. 
 * 
 * @{
 */
/**
 * One-channel 8-bit unsigned image CrossCorrSame_NormLevel.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrSame_NormLevel_8u_C1RSfs_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                          const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                Npp8u * pDst, int nDstStep, int nScaleFactor, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 8-bit unsigned image CrossCorrSame_NormLevel.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrSame_NormLevel_8u_C3RSfs_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                          const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                Npp8u * pDst, int nDstStep, int nScaleFactor, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image CrossCorrSame_NormLevel.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrSame_NormLevel_8u_C4RSfs_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                          const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                Npp8u * pDst, int nDstStep, int nScaleFactor, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image CrossCorrSame_NormLevel ignoring alpha channel.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrSame_NormLevel_8u_AC4RSfs_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                           const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                 Npp8u * pDst, int nDstStep, int nScaleFactor, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 32-bit floating point image CrossCorrSame_NormLevel.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrSame_NormLevel_32f_C1R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                        const Npp32f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                              Npp32f * pDst, int nDstStep, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 32-bit floating point image CrossCorrSame_NormLevel.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrSame_NormLevel_32f_C3R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                        const Npp32f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                              Npp32f * pDst, int nDstStep, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit floating point image CrossCorrSame_NormLevel.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrSame_NormLevel_32f_C4R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                        const Npp32f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                              Npp32f * pDst, int nDstStep, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit floating point image CrossCorrSame_NormLevel ignoring alpha channel.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrSame_NormLevel_32f_AC4R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                         const Npp32f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                               Npp32f * pDst, int nDstStep, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 64-bit floating point image CrossCorrSame_NormLevel.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrSame_NormLevel_64f_C1R_Ctx(const Npp64f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                        const Npp64f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                              Npp64f * pDst, int nDstStep, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 64-bit floating point image CrossCorrSame_NormLevel.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrSame_NormLevel_64f_C3R_Ctx(const Npp64f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                        const Npp64f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                              Npp64f * pDst, int nDstStep, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 64-bit floating point image CrossCorrSame_NormLevel.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrSame_NormLevel_64f_C4R_Ctx(const Npp64f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                        const Npp64f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                              Npp64f * pDst, int nDstStep, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 64-bit floating point image CrossCorrSame_NormLevel ignoring alpha channel.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrSame_NormLevel_64f_AC4R_Ctx(const Npp64f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                         const Npp64f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                               Npp64f * pDst, int nDstStep, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 8-bit unsigned image CrossCorrSame_NormLevel.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrSame_NormLevel_8u32f_C1R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                          const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                Npp32f * pDst, int nDstStep, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 8-bit unsigned image CrossCorrSame_NormLevel.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrSame_NormLevel_8u32f_C3R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                          const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                Npp32f * pDst, int nDstStep, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image CrossCorrSame_NormLevel.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrSame_NormLevel_8u32f_C4R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                          const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                Npp32f * pDst, int nDstStep, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image CrossCorrSame_NormLevel ignoring alpha channel.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrSame_NormLevel_8u32f_AC4R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                           const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                 Npp32f * pDst, int nDstStep, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 8-bit signed image CrossCorrSame_NormLevel.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrSame_NormLevel_8s32f_C1R_Ctx(const Npp8s * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                          const Npp8s * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                Npp32f * pDst, int nDstStep, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 8-bit signed image CrossCorrSame_NormLevel.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrSame_NormLevel_8s32f_C3R_Ctx(const Npp8s * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                          const Npp8s * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                Npp32f * pDst, int nDstStep, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit signed image CrossCorrSame_NormLevel.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrSame_NormLevel_8s32f_C4R_Ctx(const Npp8s * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                          const Npp8s * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                Npp32f * pDst, int nDstStep, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit signed image CrossCorrSame_NormLevel ignoring alpha channel.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrSame_NormLevel_8s32f_AC4R_Ctx(const Npp8s * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                           const Npp8s * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                 Npp32f * pDst, int nDstStep, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 16-bit unsigned image CrossCorrSame_NormLevel.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrSame_NormLevel_16u32f_C1R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                           const Npp16u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                 Npp32f * pDst, int nDstStep, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 16-bit unsigned image CrossCorrSame_NormLevel.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrSame_NormLevel_16u32f_C3R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                           const Npp16u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                 Npp32f * pDst, int nDstStep, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit unsigned image CrossCorrSame_NormLevel.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrSame_NormLevel_16u32f_C4R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                           const Npp16u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                 Npp32f * pDst, int nDstStep, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit unsigned image CrossCorrSame_NormLevel ignoring alpha channel.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrSame_NormLevel_16u32f_AC4R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                            const Npp16u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                  Npp32f * pDst, int nDstStep, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/** @} */

/** @name SameNormLevelGetBufferHostSize 
 * Companion primitives for computing the device buffer size (in bytes) required by the CrossCorrSame_NormLevel primitives.
 * \section CommonSameNormLevelGetBufferHostSizeParameters
 * Common parameters for nppiSameNormLevelGetBufferHostSize functions include:
 * \param oSizeROI \ref roi_specification.
 * \param hpBufferSize Required buffer size. Important: hpBufferSize is a 
 *        <em>host pointer.</em> \ref general_scratch_buffer.
 * \param nppStreamCtx \ref application_managed_stream_context. 
 * \return NPP_NULL_POINTER_ERROR if hpBufferSize is 0 (NULL),  \ref roi_error_codes.  
 * @{
 */
/** 
 * Buffer size (in bytes) for nppiCrossCorrSame_NormLevel_8u_C1RSfs.
 * For common parameter descriptions, see \ref CommonSameNormLevelGetBufferHostSizeParameters.
 */
NppStatus 
nppiSameNormLevelGetBufferHostSize_8u_C1RSfs_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size (in bytes) for \ref nppiCrossCorrSame_NormLevel_8u_C3RSfs_Ctx.
 * For common parameter descriptions, see \ref CommonSameNormLevelGetBufferHostSizeParameters.
 */
NppStatus 
nppiSameNormLevelGetBufferHostSize_8u_C3RSfs_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size (in bytes) for \ref nppiCrossCorrSame_NormLevel_8u_C4RSfs_Ctx.
 * For common parameter descriptions, see \ref CommonSameNormLevelGetBufferHostSizeParameters.
 */
NppStatus 
nppiSameNormLevelGetBufferHostSize_8u_C4RSfs_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size (in bytes) for \ref nppiCrossCorrSame_NormLevel_8u_AC4RSfs_Ctx.
 * For common parameter descriptions, see \ref CommonSameNormLevelGetBufferHostSizeParameters.
 */
NppStatus 
nppiSameNormLevelGetBufferHostSize_8u_AC4RSfs_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size (in bytes) for \ref nppiCrossCorrSame_NormLevel_32f_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonSameNormLevelGetBufferHostSizeParameters.
 */
NppStatus 
nppiSameNormLevelGetBufferHostSize_32f_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size (in bytes) for \ref nppiCrossCorrSame_NormLevel_32f_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonSameNormLevelGetBufferHostSizeParameters.
 */
NppStatus 
nppiSameNormLevelGetBufferHostSize_32f_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size (in bytes) for \ref nppiCrossCorrSame_NormLevel_32f_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonSameNormLevelGetBufferHostSizeParameters.
 */
NppStatus 
nppiSameNormLevelGetBufferHostSize_32f_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size (in bytes) for \ref nppiCrossCorrSame_NormLevel_32f_AC4R_Ctx.
 * For common parameter descriptions, see \ref CommonSameNormLevelGetBufferHostSizeParameters.
 */
NppStatus 
nppiSameNormLevelGetBufferHostSize_32f_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size (in bytes) for \ref nppiCrossCorrSame_NormLevel_64f_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonSameNormLevelGetBufferHostSizeParameters.
 */
NppStatus 
nppiSameNormLevelGetBufferHostSize_64f_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size (in bytes) for \ref nppiCrossCorrSame_NormLevel_64f_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonSameNormLevelGetBufferHostSizeParameters.
 */
NppStatus 
nppiSameNormLevelGetBufferHostSize_64f_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size (in bytes) for \ref nppiCrossCorrSame_NormLevel_64f_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonSameNormLevelGetBufferHostSizeParameters.
 */
NppStatus 
nppiSameNormLevelGetBufferHostSize_64f_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size (in bytes) for \ref nppiCrossCorrSame_NormLevel_64f_AC4R_Ctx.
 * For common parameter descriptions, see \ref CommonSameNormLevelGetBufferHostSizeParameters.
 */
NppStatus 
nppiSameNormLevelGetBufferHostSize_64f_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size (in bytes) for \ref nppiCrossCorrSame_NormLevel_8u32f_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonSameNormLevelGetBufferHostSizeParameters.
 */
NppStatus 
nppiSameNormLevelGetBufferHostSize_8u32f_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size (in bytes) for \ref nppiCrossCorrSame_NormLevel_8u32f_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonSameNormLevelGetBufferHostSizeParameters.
 */
NppStatus 
nppiSameNormLevelGetBufferHostSize_8u32f_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size (in bytes) for \ref nppiCrossCorrSame_NormLevel_8u32f_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonSameNormLevelGetBufferHostSizeParameters.
 */
NppStatus 
nppiSameNormLevelGetBufferHostSize_8u32f_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size (in bytes) for \ref nppiCrossCorrSame_NormLevel_8u32f_AC4R_Ctx.
 * For common parameter descriptions, see \ref CommonSameNormLevelGetBufferHostSizeParameters.
 */
NppStatus 
nppiSameNormLevelGetBufferHostSize_8u32f_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size (in bytes) for \ref nppiCrossCorrSame_NormLevel_8s32f_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonSameNormLevelGetBufferHostSizeParameters.
 */
NppStatus 
nppiSameNormLevelGetBufferHostSize_8s32f_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size (in bytes) for \ref nppiCrossCorrSame_NormLevel_8s32f_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonSameNormLevelGetBufferHostSizeParameters.
 */
NppStatus 
nppiSameNormLevelGetBufferHostSize_8s32f_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size (in bytes) for \ref nppiCrossCorrSame_NormLevel_8s32f_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonSameNormLevelGetBufferHostSizeParameters.
 */
NppStatus 
nppiSameNormLevelGetBufferHostSize_8s32f_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size (in bytes) for \ref nppiCrossCorrSame_NormLevel_8s32f_AC4R_Ctx.
 * For common parameter descriptions, see \ref CommonSameNormLevelGetBufferHostSizeParameters.
 */
NppStatus 
nppiSameNormLevelGetBufferHostSize_8s32f_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size (in bytes) for \ref nppiCrossCorrSame_NormLevel_16u32f_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonSameNormLevelGetBufferHostSizeParameters.
 */
NppStatus 
nppiSameNormLevelGetBufferHostSize_16u32f_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size (in bytes) for \ref nppiCrossCorrSame_NormLevel_16u32f_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonSameNormLevelGetBufferHostSizeParameters.
 */
NppStatus 
nppiSameNormLevelGetBufferHostSize_16u32f_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size (in bytes) for \ref nppiCrossCorrSame_NormLevel_16u32f_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonSameNormLevelGetBufferHostSizeParameters.
 */
NppStatus 
nppiSameNormLevelGetBufferHostSize_16u32f_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size (in bytes) for \ref nppiCrossCorrSame_NormLevel_16u32f_AC4R_Ctx.
 * For common parameter descriptions, see \ref CommonSameNormLevelGetBufferHostSizeParameters.
 */
NppStatus 
nppiSameNormLevelGetBufferHostSize_16u32f_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);

/** @} */
/** @} CrossCorrSame_NormLevel*/

/**
 * \section crosscorrvalidnormlevel CrossCorrValid_NormLevel
 * @defgroup crosscorrvalidnormlevel CrossCorrValid_NormLevel
 * Primitives for computing the normalized cross correlation coefficient between two images with valid mode.
 *
 * @{
 */
/**
 * @name CrossCorrValid_NormLevel
 * The functions compute the \f$\gamma_{st}(c,r)\f$ in \ref general_introduction 
 * with valid mode (see \ref category). The functions require additional scratch buffer for computations. 
 *  
 * Note: For maximum performance oSrcRoiSize.width - oTplRoiSize + 1 MUST be an integer multiple of 4. 
 *
 * @{
 */
/**
 * One-channel 8-bit unsigned image CrossCorrValid_NormLevel.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrValid_NormLevel_8u_C1RSfs_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                           const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                 Npp8u * pDst, int nDstStep, int nScaleFactor, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 8-bit unsigned image CrossCorrValid_NormLevel.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrValid_NormLevel_8u_C3RSfs_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                           const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                 Npp8u * pDst, int nDstStep, int nScaleFactor, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image CrossCorrValid_NormLevel.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrValid_NormLevel_8u_C4RSfs_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                           const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                 Npp8u * pDst, int nDstStep, int nScaleFactor, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image CrossCorrValid_NormLevel ignoring alpha channel.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrValid_NormLevel_8u_AC4RSfs_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                            const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                  Npp8u * pDst, int nDstStep, int nScaleFactor, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 32-bit floating point image CrossCorrValid_NormLevel.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrValid_NormLevel_32f_C1R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                         const Npp32f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                               Npp32f * pDst, int nDstStep, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 32-bit floating point image CrossCorrValid_NormLevel.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrValid_NormLevel_32f_C3R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                         const Npp32f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                               Npp32f * pDst, int nDstStep, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit floating point image CrossCorrValid_NormLevel.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrValid_NormLevel_32f_C4R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                         const Npp32f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                               Npp32f * pDst, int nDstStep, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit floating point image CrossCorrValid_NormLevel ignoring alpha channel.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrValid_NormLevel_32f_AC4R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                          const Npp32f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                Npp32f * pDst, int nDstStep, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 64-bit floating point image CrossCorrValid_NormLevel.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrValid_NormLevel_64f_C1R_Ctx(const Npp64f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                         const Npp64f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                               Npp64f * pDst, int nDstStep, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 64-bit floating point image CrossCorrValid_NormLevel.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrValid_NormLevel_64f_C3R_Ctx(const Npp64f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                         const Npp64f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                               Npp64f * pDst, int nDstStep, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 64-bit floating point image CrossCorrValid_NormLevel.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrValid_NormLevel_64f_C4R_Ctx(const Npp64f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                         const Npp64f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                               Npp64f * pDst, int nDstStep, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 64-bit floating point image CrossCorrValid_NormLevel ignoring alpha channel.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrValid_NormLevel_64f_AC4R_Ctx(const Npp64f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                          const Npp64f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                Npp64f * pDst, int nDstStep, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 8-bit unsigned image CrossCorrValid_NormLevel.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrValid_NormLevel_8u32f_C1R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                           const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                 Npp32f * pDst, int nDstStep, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 8-bit unsigned image CrossCorrValid_NormLevel.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrValid_NormLevel_8u32f_C3R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                           const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                 Npp32f * pDst, int nDstStep, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image CrossCorrValid_NormLevel.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrValid_NormLevel_8u32f_C4R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                           const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                 Npp32f * pDst, int nDstStep, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image CrossCorrValid_NormLevel ignoring alpha channel.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrValid_NormLevel_8u32f_AC4R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                            const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                  Npp32f * pDst, int nDstStep, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 8-bit signed image CrossCorrValid_NormLevel.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrValid_NormLevel_8s32f_C1R_Ctx(const Npp8s * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                           const Npp8s * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                 Npp32f * pDst, int nDstStep, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 8-bit signed image CrossCorrValid_NormLevel.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrValid_NormLevel_8s32f_C3R_Ctx(const Npp8s * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                           const Npp8s * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                 Npp32f * pDst, int nDstStep, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit signed image CrossCorrValid_NormLevel.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrValid_NormLevel_8s32f_C4R_Ctx(const Npp8s * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                           const Npp8s * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                 Npp32f * pDst, int nDstStep, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit signed image CrossCorrValid_NormLevel ignoring alpha channel.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrValid_NormLevel_8s32f_AC4R_Ctx(const Npp8s * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                            const Npp8s * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                  Npp32f * pDst, int nDstStep, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 16-bit unsigned image CrossCorrValid_NormLevel.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrValid_NormLevel_16u32f_C1R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                            const Npp16u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                  Npp32f * pDst, int nDstStep, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 16-bit unsigned image CrossCorrValid_NormLevel.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrValid_NormLevel_16u32f_C3R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                            const Npp16u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                  Npp32f * pDst, int nDstStep, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit unsigned image CrossCorrValid_NormLevel.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrValid_NormLevel_16u32f_C4R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                            const Npp16u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                  Npp32f * pDst, int nDstStep, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit unsigned image CrossCorrValid_NormLevel ignoring alpha channel.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrValid_NormLevel_16u32f_AC4R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                             const Npp16u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                   Npp32f * pDst, int nDstStep, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);

/** @} */

/** @name ValidNormLevelGetBufferHostSize 
 * Companion primitives for computing the device buffer size (in bytes) required by the CrossCorrValid_NormLevel primitives.
 * \section CommonValidNormLevelGetBufferHostSizeParameters
 * Common parameters for nppiValidNormLevelGetBufferHostSize functions include:
 * \param oSizeROI \ref roi_specification.
 * \param hpBufferSize Required buffer size. Important: hpBufferSize is a 
 *        <em>host pointer.</em> \ref general_scratch_buffer.
 * \param nppStreamCtx \ref application_managed_stream_context. 
 * \return NPP_NULL_POINTER_ERROR if hpBufferSize is 0 (NULL),  \ref roi_error_codes.  
 * @{
 */
/** 
 * Buffer size (in bytes) for nppiCrossCorrValid_NormLevel_8u_C1RSfs_Ctx.
 * For common parameter descriptions, see \ref CommonValidNormLevelGetBufferHostSizeParameters.
 */
NppStatus 
nppiValidNormLevelGetBufferHostSize_8u_C1RSfs_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size (in bytes) for nppiCrossCorrValid_NormLevel_8u_C1RSfs_Ctx.
 * For common parameter descriptions, see \ref CommonValidNormLevelGetBufferHostSizeParameters.
 */
NppStatus 
nppiValidNormLevelGetBufferHostSize_8u_C3RSfs_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size (in bytes) for \ref nppiCrossCorrValid_NormLevel_8u_C4RSfs_Ctx.
 * For common parameter descriptions, see \ref CommonValidNormLevelGetBufferHostSizeParameters.
 */
NppStatus 
nppiValidNormLevelGetBufferHostSize_8u_C4RSfs_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size (in bytes) for \ref nppiCrossCorrValid_NormLevel_8u_AC4RSfs_Ctx.
 * For common parameter descriptions, see \ref CommonValidNormLevelGetBufferHostSizeParameters.
 */
NppStatus 
nppiValidNormLevelGetBufferHostSize_8u_AC4RSfs_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size (in bytes) for \ref nppiCrossCorrValid_NormLevel_32f_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonValidNormLevelGetBufferHostSizeParameters.
 */
NppStatus 
nppiValidNormLevelGetBufferHostSize_32f_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size (in bytes) for \ref nppiCrossCorrValid_NormLevel_32f_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonValidNormLevelGetBufferHostSizeParameters.
 */
NppStatus 
nppiValidNormLevelGetBufferHostSize_32f_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size (in bytes) for \ref nppiCrossCorrValid_NormLevel_32f_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonValidNormLevelGetBufferHostSizeParameters.
 */
NppStatus 
nppiValidNormLevelGetBufferHostSize_32f_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size (in bytes) for \ref nppiCrossCorrValid_NormLevel_32f_AC4R_Ctx.
 * For common parameter descriptions, see \ref CommonValidNormLevelGetBufferHostSizeParameters.
 */
NppStatus 
nppiValidNormLevelGetBufferHostSize_32f_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size (in bytes) for \ref nppiCrossCorrValid_NormLevel_64f_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonValidNormLevelGetBufferHostSizeParameters.
 */
NppStatus 
nppiValidNormLevelGetBufferHostSize_64f_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size (in bytes) for \ref nppiCrossCorrValid_NormLevel_64f_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonValidNormLevelGetBufferHostSizeParameters.
 */
NppStatus 
nppiValidNormLevelGetBufferHostSize_64f_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size (in bytes) for \ref nppiCrossCorrValid_NormLevel_64f_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonValidNormLevelGetBufferHostSizeParameters.
 */
NppStatus 
nppiValidNormLevelGetBufferHostSize_64f_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size (in bytes) for \ref nppiCrossCorrValid_NormLevel_64f_AC4R_Ctx.
 * For common parameter descriptions, see \ref CommonValidNormLevelGetBufferHostSizeParameters.
 */
NppStatus 
nppiValidNormLevelGetBufferHostSize_64f_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size (in bytes) for \ref nppiCrossCorrValid_NormLevel_8u32f_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonValidNormLevelGetBufferHostSizeParameters.
 */
NppStatus 
nppiValidNormLevelGetBufferHostSize_8u32f_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size (in bytes) for \ref nppiCrossCorrValid_NormLevel_8u32f_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonValidNormLevelGetBufferHostSizeParameters.
 */
NppStatus 
nppiValidNormLevelGetBufferHostSize_8u32f_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size (in bytes) for \ref nppiCrossCorrValid_NormLevel_8u32f_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonValidNormLevelGetBufferHostSizeParameters.
 */
NppStatus 
nppiValidNormLevelGetBufferHostSize_8u32f_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size (in bytes) for \ref nppiCrossCorrValid_NormLevel_8u32f_AC4R_Ctx.
 * For common parameter descriptions, see \ref CommonValidNormLevelGetBufferHostSizeParameters.
 */
NppStatus 
nppiValidNormLevelGetBufferHostSize_8u32f_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size (in bytes) for \ref nppiCrossCorrValid_NormLevel_8s32f_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonValidNormLevelGetBufferHostSizeParameters.
 */
NppStatus 
nppiValidNormLevelGetBufferHostSize_8s32f_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size (in bytes) for \ref nppiCrossCorrValid_NormLevel_8s32f_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonValidNormLevelGetBufferHostSizeParameters.
 */
NppStatus 
nppiValidNormLevelGetBufferHostSize_8s32f_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size (in bytes) for \ref nppiCrossCorrValid_NormLevel_8s32f_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonValidNormLevelGetBufferHostSizeParameters.
 */
NppStatus 
nppiValidNormLevelGetBufferHostSize_8s32f_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size (in bytes) for \ref nppiCrossCorrValid_NormLevel_8s32f_AC4R_Ctx.
 * For common parameter descriptions, see \ref CommonValidNormLevelGetBufferHostSizeParameters.
 */
NppStatus 
nppiValidNormLevelGetBufferHostSize_8s32f_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size (in bytes) for \ref nppiCrossCorrValid_NormLevel_16u32f_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonValidNormLevelGetBufferHostSizeParameters.
 */
NppStatus 
nppiValidNormLevelGetBufferHostSize_16u32f_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size (in bytes) for \ref nppiCrossCorrValid_NormLevel_16u32f_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonValidNormLevelGetBufferHostSizeParameters.
 */
NppStatus 
nppiValidNormLevelGetBufferHostSize_16u32f_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size (in bytes) for \ref nppiCrossCorrValid_NormLevel_16u32f_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonValidNormLevelGetBufferHostSizeParameters.
 */
NppStatus 
nppiValidNormLevelGetBufferHostSize_16u32f_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size (in bytes) for \ref nppiCrossCorrValid_NormLevel_16u32f_AC4R_Ctx.
 * For common parameter descriptions, see \ref CommonValidNormLevelGetBufferHostSizeParameters.
 */
NppStatus 
nppiValidNormLevelGetBufferHostSize_16u32f_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);

/** @} */
/** @} CrossCorrValid_NormLevel*/

/**
 * \section crosscorrfullnormlevelAdvanced CrossCorrFull_NormLevelAdvanced
 * @defgroup crosscorrfullnormlevelAdvanced CrossCorrFull_NormLevelAdvanced
 * Primitives for computing the normalized cross correlation coefficient between two images with full mode with large image template sizes. 
 *
 * @{
 */
/**
 * @name CrossCorrFull_NormLevelAdvanced
 * The functions compute the \f$\gamma_{st}(c,r)\f$ in \ref general_introduction 
 * with full mode (see \ref category). The functions require an additional scratch buffer and advanced scratch buffer for computations. 
 *  
 * Note: For maximum performance oSrcRoiSize.width + oTplRoiSize.width - 1 MUST be an integer multiple of 4. 
 *  
 * @{
 */

/**
 * One-channel 32-bit floating point image CrossCorrFull_NormLevelAdvanced.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrFull_NormLevelAdvanced_32f_C1R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                                const Npp32f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                      Npp32f * pDst, int nDstStep, 
                                                      Npp8u * pDeviceBuffer, Npp8u * pAdvancedScratchBuffer, NppStreamContext nppStreamCtx);

/**
 * Three-channel 32-bit floating point image CrossCorrFull_NormLevelAdvanced.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrFull_NormLevelAdvanced_32f_C3R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                                const Npp32f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                      Npp32f * pDst, int nDstStep, 
                                                      Npp8u * pDeviceBuffer, Npp8u * pAdvancedScratchBuffer, NppStreamContext nppStreamCtx);

/**
 * Four-channel 32-bit floating point image CrossCorrFull_NormLevelAdvanced.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrFull_NormLevelAdvanced_32f_C4R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                                const Npp32f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                      Npp32f * pDst, int nDstStep, 
                                                      Npp8u * pDeviceBuffer, Npp8u * pAdvancedScratchBuffer, NppStreamContext nppStreamCtx);

/**
 * One-channel 64-bit floating point image CrossCorrFull_NormLevelAdvanced.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrFull_NormLevelAdvanced_64f_C1R_Ctx(const Npp64f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                                const Npp64f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                      Npp64f * pDst, int nDstStep, 
                                                      Npp8u * pDeviceBuffer, Npp8u * pAdvancedScratchBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 64-bit floating point image CrossCorrFull_NormLevelAdvanced.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrFull_NormLevelAdvanced_64f_C3R_Ctx(const Npp64f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                                const Npp64f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                      Npp64f * pDst, int nDstStep, 
                                                      Npp8u * pDeviceBuffer, Npp8u * pAdvancedScratchBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 64-bit floating point image CrossCorrFull_NormLevelAdvanced.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrFull_NormLevelAdvanced_64f_C4R_Ctx(const Npp64f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                                const Npp64f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                      Npp64f * pDst, int nDstStep, 
                                                      Npp8u * pDeviceBuffer, Npp8u * pAdvancedScratchBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 8-bit unsigned image CrossCorrFull_NormLevelAdvanced.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrFull_NormLevelAdvanced_8u32f_C1R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                                  const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                        Npp32f * pDst, int nDstStep, 
                                                        Npp8u * pDeviceBuffer, Npp8u * pAdvancedScratchBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 8-bit unsigned image CrossCorrFull_NormLevelAdvanced.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrFull_NormLevelAdvanced_8u32f_C3R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                                  const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                        Npp32f * pDst, int nDstStep, 
                                                        Npp8u * pDeviceBuffer, Npp8u * pAdvancedScratchBuffer, NppStreamContext nppStreamCtx);

/**
 * Four-channel 8-bit unsigned image CrossCorrFull_NormLevelAdvanced.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrFull_NormLevelAdvanced_8u32f_C4R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                                  const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                        Npp32f * pDst, int nDstStep, 
                                                        Npp8u * pDeviceBuffer, Npp8u * pAdvancedScratchBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 8-bit signed image CrossCorrFull_NormLevelAdvanced.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrFull_NormLevelAdvanced_8s32f_C1R_Ctx(const Npp8s * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                                  const Npp8s * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                        Npp32f * pDst, int nDstStep, 
                                                        Npp8u * pDeviceBuffer, Npp8u * pAdvancedScratchBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 8-bit signed image CrossCorrFull_NormLevelAdvanced.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrFull_NormLevelAdvanced_8s32f_C3R_Ctx(const Npp8s * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                                  const Npp8s * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                        Npp32f * pDst, int nDstStep, 
                                                        Npp8u * pDeviceBuffer, Npp8u * pAdvancedScratchBuffer, NppStreamContext nppStreamCtx);

/**
 * Four-channel 8-bit signed image CrossCorrFull_NormLevelAdvanced.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrFull_NormLevelAdvanced_8s32f_C4R_Ctx(const Npp8s * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                                  const Npp8s * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                        Npp32f * pDst, int nDstStep, 
                                                        Npp8u * pDeviceBuffer, Npp8u * pAdvancedScratchBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 16-bit unsigned image CrossCorrFull_NormLevelAdvanced.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrFull_NormLevelAdvanced_16u32f_C1R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                                   const Npp16u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                         Npp32f * pDst, int nDstStep, 
                                                         Npp8u * pDeviceBuffer, Npp8u * pAdvancedScratchBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 16-bit unsigned image CrossCorrFull_NormLevelAdvanced.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrFull_NormLevelAdvanced_16u32f_C3R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                                   const Npp16u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                         Npp32f * pDst, int nDstStep, 
                                                         Npp8u * pDeviceBuffer, Npp8u * pAdvancedScratchBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit unsigned image CrossCorrFull_NormLevelAdvanced.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrFull_NormLevelAdvanced_16u32f_C4R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                                   const Npp16u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                         Npp32f * pDst, int nDstStep, 
                                                         Npp8u * pDeviceBuffer, Npp8u * pAdvancedScratchBuffer, NppStreamContext nppStreamCtx);
/** @} */

/** @name FullNormLevelGetAdvancedScratchBufferHostSize 
 * Companion primitives for computing the device buffer size (in bytes) required by the CrossCorrFull_NormLevelAdvanced primitives. 
 *  
 * @{
 */
/** 
 * Buffer size (in bytes) for nppiCrossCorrFull_NormLevelAdvanced functions. 
 *  
 * \param oSrcRoiSize \ref roi_specification.
 * \param oTplRoiSize \ref roi_specification.
 * \param nSizeofDstData sizeof(destination data type (usually Npp32f)).
 * \param nSrcChannels number of source image color channels. 
 * \param hpBufferSize Required buffer size. Important: hpBufferSize is a 
 *        <em>host pointer.</em> \ref general_scratch_buffer.
 * \return \ref image_data_error_codes, \ref roi_error_codes
 */
NppStatus 
nppiCrossCorrFull_NormLevel_GetAdvancedScratchBufferSize(NppiSize oSrcRoiSize, NppiSize oTplRoiSize, 
                                                         int nSizeofDstData, int nSrcChannels, 
                                                         size_t * hpBufferSize /* host pointer */);

/** @} */
/** @} CrossCorrFull_NormLevelAdvanced*/

/**
 * \section crosscorrsamenormlevelAdvanced CrossCorrSame_NormLevelAdvanced
 * @defgroup crosscorrsamenormlevelAdvanced CrossCorrSame_NormLevelAdvanced
 * 
 * Primitives for computing the normalized cross correlation coefficient between two images with same mode with large image template sizes.
 * 
 * @{
 */
/**
 * @name CrossCorrSame_NormLevelAdvanced
 * The functions compute the \f$\gamma_{st}(c,r)\f$ in \ref general_introduction 
 * with same mode (see \ref category). The functions require and additional scratch buffer and advanced scratch buffer for computations. 
 *  
 * Note: For maximum performance oSrcRoiSize.width MUST be an integer multiple of 4. 
 * 
 * @{
 */

/**
 * One-channel 32-bit floating point image CrossCorrSame_NormLevelAdvanced.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrSame_NormLevelAdvanced_32f_C1R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                                const Npp32f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                      Npp32f * pDst, int nDstStep, 
                                                      Npp8u * pDeviceBuffer, Npp8u * pAdvancedScratchBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 32-bit floating point image CrossCorrSame_NormLevelAdvanced.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrSame_NormLevelAdvanced_32f_C3R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                                const Npp32f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                      Npp32f * pDst, int nDstStep, 
                                                      Npp8u * pDeviceBuffer, Npp8u * pAdvancedScratchBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit floating point image CrossCorrSame_NormLevelAdvanced.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrSame_NormLevelAdvanced_32f_C4R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                                const Npp32f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                      Npp32f * pDst, int nDstStep, 
                                                      Npp8u * pDeviceBuffer, Npp8u * pAdvancedScratchBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 64-bit floating point image CrossCorrSame_NormLevelAdvanced.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrSame_NormLevelAdvanced_64f_C1R_Ctx(const Npp64f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                                const Npp64f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                      Npp64f * pDst, int nDstStep, 
                                                      Npp8u * pDeviceBuffer, Npp8u * pAdvancedScratchBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 64-bit floating point image CrossCorrSame_NormLevelAdvanced.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrSame_NormLevelAdvanced_64f_C3R_Ctx(const Npp64f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                                const Npp64f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                      Npp64f * pDst, int nDstStep, 
                                                      Npp8u * pDeviceBuffer, Npp8u * pAdvancedScratchBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 64-bit floating point image CrossCorrSame_NormLevelAdvanced.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrSame_NormLevelAdvanced_64f_C4R_Ctx(const Npp64f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                                const Npp64f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                      Npp64f * pDst, int nDstStep, 
                                                      Npp8u * pDeviceBuffer, Npp8u * pAdvancedScratchBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 8-bit unsigned image CrossCorrSame_NormLevelAdvanced.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrSame_NormLevelAdvanced_8u32f_C1R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                                  const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                        Npp32f * pDst, int nDstStep, 
                                                        Npp8u * pDeviceBuffer, Npp8u * pAdvancedScratchBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 8-bit unsigned image CrossCorrSame_NormLevelAdvanced.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrSame_NormLevelAdvanced_8u32f_C3R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                                  const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                        Npp32f * pDst, int nDstStep, 
                                                        Npp8u * pDeviceBuffer, Npp8u * pAdvancedScratchBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image CrossCorrSame_NormLevelAdvanced.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrSame_NormLevelAdvanced_8u32f_C4R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                                  const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                        Npp32f * pDst, int nDstStep, 
                                                        Npp8u * pDeviceBuffer, Npp8u * pAdvancedScratchBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 8-bit signed image CrossCorrSame_NormLevelAdvanced.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrSame_NormLevelAdvanced_8s32f_C1R_Ctx(const Npp8s * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                                  const Npp8s * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                        Npp32f * pDst, int nDstStep, 
                                                        Npp8u * pDeviceBuffer, Npp8u * pAdvancedScratchBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 8-bit signed image CrossCorrSame_NormLevelAdvanced.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrSame_NormLevelAdvanced_8s32f_C3R_Ctx(const Npp8s * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                                  const Npp8s * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                        Npp32f * pDst, int nDstStep, 
                                                        Npp8u * pDeviceBuffer, Npp8u * pAdvancedScratchBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit signed image CrossCorrSame_NormLevelAdvanced.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrSame_NormLevelAdvanced_8s32f_C4R_Ctx(const Npp8s * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                                  const Npp8s * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                        Npp32f * pDst, int nDstStep, 
                                                        Npp8u * pDeviceBuffer, Npp8u * pAdvancedScratchBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 16-bit unsigned image CrossCorrSame_NormLevelAdvanced.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrSame_NormLevelAdvanced_16u32f_C1R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                                   const Npp16u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                         Npp32f * pDst, int nDstStep, 
                                                         Npp8u * pDeviceBuffer, Npp8u * pAdvancedScratchBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 16-bit unsigned image CrossCorrSame_NormLevelAdvanced.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrSame_NormLevelAdvanced_16u32f_C3R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                                   const Npp16u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                         Npp32f * pDst, int nDstStep, 
                                                         Npp8u * pDeviceBuffer, Npp8u * pAdvancedScratchBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit unsigned image CrossCorrSame_NormLevelAdvanced.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrSame_NormLevelAdvanced_16u32f_C4R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                                   const Npp16u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                         Npp32f * pDst, int nDstStep, 
                                                         Npp8u * pDeviceBuffer, Npp8u * pAdvancedScratchBuffer, NppStreamContext nppStreamCtx);
/** @} */

/** @name SameNormLevelGetAdvancedScratchBufferHostSize 
 * Companion primitives for computing the device buffer size (in bytes) required by the CrossCorrSame_NormLevelAdvanced primitives.
 *  
 * @{
 */
/** 
 * Buffer size (in bytes) for nppiCrossCorrSame_NormLevelAdvanced functions.
 *  
 * \param oSrcRoiSize \ref roi_specification.
 * \param oTplRoiSize \ref roi_specification.
 * \param nSizeofDstData sizeof(destination data type (usually Npp32f)).
 * \param nSrcChannels number of source image color channels. 
 * \param hpBufferSize Required buffer size. Important: hpBufferSize is a 
 *        <em>host pointer.</em> \ref general_scratch_buffer.
 * \return \ref image_data_error_codes, \ref roi_error_codes
 */
NppStatus 
nppiCrossCorrSame_NormLevel_GetAdvancedScratchBufferSize(NppiSize oSrcRoiSize, NppiSize oTplRoiSize, 
                                                         int nSizeofDstData, int nSrcChannels, 
                                                         size_t * hpBufferSize /* host pointer */);

/** @} */
/** @} CrossCorrSame_NormLevelAdvanced*/

/**
 * \section crosscorrvalidnormlevelAdvanced CrossCorrValid_NormLevelAdvanced
 * @defgroup crosscorrvalidnormlevelAdvanced CrossCorrValid_NormLevelAdvanced
 * Primitives for computing the normalized cross correlation coefficient between two images with valid mode with large template sizes.
 *
 * @{
 */
/**
 * @name CrossCorrValid_NormLevelAdvanced
 * The functions compute the \f$\gamma_{st}(c,r)\f$ in \ref general_introduction 
 * with valid mode (see \ref category). The functions require an additional scratch buffer and advanced scratch buffer for computations. 
 *  
 * Note: For maximum performance oSrcRoiSize.width - oTplRoiSize + 1 MUST be an integer multiple of 4. 
 *
 * @{
 */

/**
 * One-channel 32-bit floating point image CrossCorrValid_NormLevelAdvanced.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrValid_NormLevelAdvanced_32f_C1R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                                 const Npp32f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                       Npp32f * pDst, int nDstStep, 
                                                       Npp8u * pDeviceBuffer, Npp8u * pAdvancedScratchBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 32-bit floating point image CrossCorrValid_NormLevelAdvanced.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrValid_NormLevelAdvanced_32f_C3R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                                 const Npp32f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                       Npp32f * pDst, int nDstStep, 
                                                       Npp8u * pDeviceBuffer, Npp8u * pAdvancedScratchBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit floating point image CrossCorrValid_NormLevelAdvanced.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrValid_NormLevelAdvanced_32f_C4R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                                 const Npp32f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                       Npp32f * pDst, int nDstStep, 
                                                       Npp8u * pDeviceBuffer, Npp8u * pAdvancedScratchBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 64-bit floating point image CrossCorrValid_NormLevelAdvanced.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrValid_NormLevelAdvanced_64f_C1R_Ctx(const Npp64f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                                 const Npp64f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                       Npp64f * pDst, int nDstStep, 
                                                       Npp8u * pDeviceBuffer, Npp8u * pAdvancedScratchBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 64-bit floating point image CrossCorrValid_NormLevelAdvanced.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrValid_NormLevelAdvanced_64f_C3R_Ctx(const Npp64f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                                 const Npp64f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                       Npp64f * pDst, int nDstStep, 
                                                       Npp8u * pDeviceBuffer, Npp8u * pAdvancedScratchBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 64-bit floating point image CrossCorrValid_NormLevelAdvanced.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrValid_NormLevelAdvanced_64f_C4R_Ctx(const Npp64f * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                                 const Npp64f * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                       Npp64f * pDst, int nDstStep, 
                                                       Npp8u * pDeviceBuffer, Npp8u * pAdvancedScratchBuffer, NppStreamContext nppStreamCtx);

/**
 * One-channel 8-bit unsigned image CrossCorrValid_NormLevelAdvanced.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrValid_NormLevelAdvanced_8u32f_C1R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                                   const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                         Npp32f * pDst, int nDstStep, 
                                                         Npp8u * pDeviceBuffer, Npp8u * pAdvancedScratchBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 8-bit unsigned image CrossCorrValid_NormLevelAdvanced.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrValid_NormLevelAdvanced_8u32f_C3R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                                   const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                         Npp32f * pDst, int nDstStep, 
                                                         Npp8u * pDeviceBuffer, Npp8u * pAdvancedScratchBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image CrossCorrValid_NormLevelAdvanced.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrValid_NormLevelAdvanced_8u32f_C4R_Ctx(const Npp8u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                                   const Npp8u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                         Npp32f * pDst, int nDstStep, 
                                                         Npp8u * pDeviceBuffer, Npp8u * pAdvancedScratchBuffer, NppStreamContext nppStreamCtx);

/**
 * One-channel 8-bit signed image CrossCorrValid_NormLevelAdvanced.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrValid_NormLevelAdvanced_8s32f_C1R_Ctx(const Npp8s * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                                   const Npp8s * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                         Npp32f * pDst, int nDstStep, 
                                                         Npp8u * pDeviceBuffer, Npp8u * pAdvancedScratchBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 8-bit signed image CrossCorrValid_NormLevelAdvanced.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrValid_NormLevelAdvanced_8s32f_C3R_Ctx(const Npp8s * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                                   const Npp8s * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                         Npp32f * pDst, int nDstStep, 
                                                         Npp8u * pDeviceBuffer, Npp8u * pAdvancedScratchBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit signed image CrossCorrValid_NormLevelAdvanced.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrValid_NormLevelAdvanced_8s32f_C4R_Ctx(const Npp8s * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                                   const Npp8s * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                         Npp32f * pDst, int nDstStep, 
                                                         Npp8u * pDeviceBuffer, Npp8u * pAdvancedScratchBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 16-bit unsigned image CrossCorrValid_NormLevelAdvanced.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrValid_NormLevelAdvanced_16u32f_C1R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                                    const Npp16u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                          Npp32f * pDst, int nDstStep, 
                                                          Npp8u * pDeviceBuffer, Npp8u * pAdvancedScratchBuffer, NppStreamContext nppStreamCtx);

/**
 * Three-channel 16-bit unsigned image CrossCorrValid_NormLevelAdvanced.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrValid_NormLevelAdvanced_16u32f_C3R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                                    const Npp16u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                          Npp32f * pDst, int nDstStep, 
                                                          Npp8u * pDeviceBuffer, Npp8u * pAdvancedScratchBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit unsigned image CrossCorrValid_NormLevelAdvanced.
 * For common parameter descriptions, see \ref CommonSqrDistanceFullNormParameters.
 */
NppStatus
nppiCrossCorrValid_NormLevelAdvanced_16u32f_C4R_Ctx(const Npp16u * pSrc, int nSrcStep, NppiSize oSrcRoiSize, 
                                                    const Npp16u * pTpl, int nTplStep, NppiSize oTplRoiSize, 
                                                          Npp32f * pDst, int nDstStep, 
                                                          Npp8u * pDeviceBuffer, Npp8u * pAdvancedScratchBuffer, NppStreamContext nppStreamCtx);
/** @} */

/** @name ValidNormLevelGetAdvancedScratchBufferHostSize 
 * Companion primitives for computing the device buffer size (in bytes) required by the CrossCorrValid_NormLevelAdvanced primitives.
 * @{
 */
/** 
 * Buffer size (in bytes) for nppiCrossCorrValid_NormLevelAdvanced functions.
*  
 * \param oSrcRoiSize \ref roi_specification.
 * \param oTplRoiSize \ref roi_specification.
 * \param nSizeofDstData sizeof(destination data type (usually Npp32f)).
 * \param nSrcChannels number of source image color channels. 
 * \param hpBufferSize Required buffer size. Important: hpBufferSize is a 
 *        <em>host pointer.</em> \ref general_scratch_buffer.
 * \return \ref image_data_error_codes, \ref roi_error_codes
 */
NppStatus 
nppiCrossCorrValid_NormLevel_GetAdvancedScratchBufferSize(NppiSize oSrcRoiSize, NppiSize oTplRoiSize, 
                                                          int nSizeofDstData, int nSrcChannels, 
                                                          size_t * hpBufferSize /* host pointer */);
/** @} */
/** @} CrossCorrValid_NormLevelAdvanced*/




/** @} image_proximity */

/** 
 * \section image_quality_index Image Quality Index
 * @defgroup image_quality_index Image Quality Index
 * Primitives for computing the image quality index of two images.
 * @{
 */
/**
 * @name QualityIndex
  * Given two images \f$M\f$ and \f$N\f$ (both \f$W \times H\f$), the
 * mathematical formula to calculate the image quality index \f$Q\f$ between them is expressed as:
 * \f[Q = \frac{4\sigma_{MN}\tilde{M}\tilde{N}}{[(\tilde{M}^2)+(\tilde{N}^2)][(\sigma_M)^2+(\sigma_N)^2]} \f]
 * where
 * \f[\tilde{M} = \frac{1}{W\cdot H}\sum_{j=0}^{H-1}\sum_{i=0}^{W-1}M(j,i)\f]
 * \f[\tilde{N} = \frac{1}{W\cdot H}\sum_{j=0}^{H-1}\sum_{i=0}^{W-1}N(j,i)\f]
 * \f[\sigma_{M} = \sqrt{\frac{1}{W\cdot H-1}\sum_{j=0}^{H-1}\sum_{i=0}^{W-1}[M(j,i)-\tilde{M}]^2}\f]
 * \f[\sigma_{N} = \sqrt{\frac{1}{W\cdot H-1}\sum_{j=0}^{H-1}\sum_{i=0}^{W-1}[N(j,i)-\tilde{N}]^2}\f]
 * \f[\sigma_{MN} = \frac{1}{W\cdot H-1}\sum_{j=0}^{H-1}\sum_{i=0}^{W-1}[M(j,i)-\tilde{M}][N(j,i)-\tilde{N}]\f]
 * The functions require additional scratch buffer for computations.
 * \section CommonQualityIndexParameters Common parameters for nppiQualityIndex functions include:
 * \param pSrc1 \ref source_image_pointer.
 * \param nSrc1Step \ref source_image_line_step.
 * \param pSrc2 \ref source_image_pointer.
 * \param nSrc2Step \ref source_image_line_step.
 * \param oRoiSize \ref roi_specification.
 * \param pDst Pointer to the quality index.
 * \param pDeviceBuffer Pointer to the required device memory allocation, \ref general_scratch_buffer. 
 * \param nppStreamCtx \ref application_managed_stream_context. 
 * \return \ref image_data_error_codes, \ref roi_error_codes, or NPP_QUALITY_INDEX_ERROR if pixels of 
 * either image are constant numberse.
 * @{
 */
/**
 * One-channel 8-bit unsigned image QualityIndex.
 * For common parameter descriptions, see \ref CommonQualityIndexParameters.
 */
NppStatus
nppiQualityIndex_8u32f_C1R_Ctx(const Npp8u * pSrc1, int nSrc1Step, 
                               const Npp8u * pSrc2, int nSrc2Step, 
                               NppiSize oRoiSize, Npp32f * pDst, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 16-bit unsigned image QualityIndex.
 * For common parameter descriptions, see \ref CommonQualityIndexParameters.
 */
NppStatus
nppiQualityIndex_16u32f_C1R_Ctx(const Npp16u * pSrc1, int nSrc1Step, 
                                const Npp16u * pSrc2, int nSrc2Step, 
                                NppiSize oRoiSize, Npp32f * pDst, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 32-bit floating point image QualityIndex.
 * For common parameter descriptions, see \ref CommonQualityIndexParameters.
 */
NppStatus
nppiQualityIndex_32f_C1R_Ctx(const Npp32f * pSrc1, int nSrc1Step, 
                             const Npp32f * pSrc2, int nSrc2Step, 
                             NppiSize oRoiSize, Npp32f * pDst, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 8-bit unsigned image QualityIndex.
 * For common parameter descriptions, see \ref CommonQualityIndexParameters.
 */
NppStatus
nppiQualityIndex_8u32f_C3R_Ctx(const Npp8u * pSrc1, int nSrc1Step, 
                               const Npp8u * pSrc2, int nSrc2Step, 
                               NppiSize oRoiSize, Npp32f * pDst, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 16-bit unsigned image QualityIndex.
 * For common parameter descriptions, see \ref CommonQualityIndexParameters.
 */
NppStatus
nppiQualityIndex_16u32f_C3R_Ctx(const Npp16u * pSrc1, int nSrc1Step, 
                                const Npp16u * pSrc2, int nSrc2Step, 
                                NppiSize oRoiSize, Npp32f * pDst, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 32-bit floating point image QualityIndex.
 * For common parameter descriptions, see \ref CommonQualityIndexParameters.
 */
NppStatus
nppiQualityIndex_32f_C3R_Ctx(const Npp32f * pSrc1, int nSrc1Step, 
                             const Npp32f * pSrc2, int nSrc2Step, 
                             NppiSize oRoiSize, Npp32f * pDst, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image QualityIndex.
 * For common parameter descriptions, see \ref CommonQualityIndexParameters.
 */
NppStatus
nppiQualityIndex_8u32f_AC4R_Ctx(const Npp8u * pSrc1, int nSrc1Step, 
                                const Npp8u * pSrc2, int nSrc2Step, 
                                NppiSize oRoiSize, Npp32f * pDst, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit unsigned image QualityIndex.
 * For common parameter descriptions, see \ref CommonQualityIndexParameters.
 */
NppStatus
nppiQualityIndex_16u32f_AC4R_Ctx(const Npp16u * pSrc1, int nSrc1Step, 
                                 const Npp16u * pSrc2, int nSrc2Step, 
                                 NppiSize oRoiSize, Npp32f * pDst, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit floating point image QualityIndex.
 * For common parameter descriptions, see \ref CommonQualityIndexParameters.
 */
NppStatus
nppiQualityIndex_32f_AC4R_Ctx(const Npp32f * pSrc1, int nSrc1Step, 
                              const Npp32f * pSrc2, int nSrc2Step, 
                              NppiSize oRoiSize, Npp32f * pDst, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);

/** @} */

/** @name QualityIndexGetBufferHostSize 
 * Companion primitives for computing the device buffer size (in bytes) required by the QualityIndex primitives.
 * \section CommonQualityIndexGetBufferHostSizeParameters
 * Common parameters for nppiQualityIndexGetBufferHostSize functions include:
 * \param oSizeROI \ref roi_specification.
 * \param hpBufferSize Required buffer size. Important: hpBufferSize is a 
 *        <em>host pointer.</em> \ref general_scratch_buffer.
 * \param nppStreamCtx \ref application_managed_stream_context. 
 * \return NPP_NULL_POINTER_ERROR if hpBufferSize is 0 (NULL),  \ref roi_error_codes.  
 * @{
 */
/** 
 * Buffer size (in bytes) for \ref nppiQualityIndex_8u32f_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonQualityIndexGetBufferHostSizeParameters.
 */
NppStatus 
nppiQualityIndexGetBufferHostSize_8u32f_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size (in bytes) for \ref nppiQualityIndex_16u32f_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonQualityIndexGetBufferHostSizeParameters.
 */
NppStatus 
nppiQualityIndexGetBufferHostSize_16u32f_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size (in bytes) for \ref nppiQualityIndex_32f_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonQualityIndexGetBufferHostSizeParameters.
 */
NppStatus 
nppiQualityIndexGetBufferHostSize_32f_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size (in bytes) for \ref nppiQualityIndex_8u32f_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonQualityIndexGetBufferHostSizeParameters.
 */
NppStatus 
nppiQualityIndexGetBufferHostSize_8u32f_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size (in bytes) for \ref nppiQualityIndex_16u32f_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonQualityIndexGetBufferHostSizeParameters.
 */
NppStatus 
nppiQualityIndexGetBufferHostSize_16u32f_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size (in bytes) for \ref nppiQualityIndex_32f_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonQualityIndexGetBufferHostSizeParameters.
 */
NppStatus 
nppiQualityIndexGetBufferHostSize_32f_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size (in bytes) for \ref nppiQualityIndex_8u32f_AC4R_Ctx.
 * For common parameter descriptions, see \ref CommonQualityIndexGetBufferHostSizeParameters.
 */
NppStatus 
nppiQualityIndexGetBufferHostSize_8u32f_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size (in bytes) for \ref nppiQualityIndex_16u32f_AC4R_Ctx.
 * For common parameter descriptions, see \ref CommonQualityIndexGetBufferHostSizeParameters.
 */
NppStatus 
nppiQualityIndexGetBufferHostSize_16u32f_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size (in bytes) for \ref nppiQualityIndex_32f_AC4R_Ctx.
 * For common parameter descriptions, see \ref CommonQualityIndexGetBufferHostSizeParameters.
 */
NppStatus 
nppiQualityIndexGetBufferHostSize_32f_AC4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);

/** @}  */
/** @} image_quality_index */

/** 
 * \section image_maximum_error MaximumError
 * @defgroup image_maximum_error MaximumError
 * Primitives for computing the maximum error between two images.
 * Given two images \f$pSrc1\f$ and \f$pSrc2\f$ both with width \f$W\f$ and height \f$H\f$, 
 * the maximum error is defined as the largest absolute difference between pixels of two images.
 * If the image is in complex format, the absolute value of the complex number is provided.
 * \section CommonMaximumErrorParameters Common parameters for nppiMaximumError functions include:
 * \param pSrc1 \ref source_image_pointer.
 * \param nSrc1Step \ref source_image_line_step.
 * \param pSrc2 \ref source_image_pointer.
 * \param nSrc2Step \ref source_image_line_step.
 * \param oSizeROI \ref roi_specification.
 * \param pError Pointer to the computed error. 
 * \param pDeviceBuffer Pointer to the required device memory allocation, \ref general_scratch_buffer. 
 * \param nppStreamCtx \ref application_managed_stream_context. 
 * \return \ref image_data_error_codes, \ref roi_error_codes.
 * @{
 */

/** @name MaximumError
* @{
 */
/**
 * One-channel 8-bit unsigned image Maximum_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumError_8u_C1R_Ctx(const Npp8u * pSrc1, int nSrc1Step, 
                                      const Npp8u * pSrc2, int nSrc2Step, 
                                      NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 8-bit signed image Maximum_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumError_8s_C1R_Ctx(const Npp8s * pSrc1, int nSrc1Step, 
                                      const Npp8s * pSrc2, int nSrc2Step, 
                                      NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 16-bit unsigned image Maximum_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumError_16u_C1R_Ctx(const Npp16u * pSrc1, int nSrc1Step, 
                                       const Npp16u * pSrc2, int nSrc2Step, 
                                       NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 16-bit signed image Maximum_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumError_16s_C1R_Ctx(const Npp16s * pSrc1, int nSrc1Step, 
                                       const Npp16s * pSrc2, int nSrc2Step, 
                                       NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 16-bit signed complex image Maximum_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumError_16sc_C1R_Ctx(const Npp16sc * pSrc1, int nSrc1Step, 
                                        const Npp16sc * pSrc2, int nSrc2Step, 
                                        NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 32-bit unsigned image Maximum_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumError_32u_C1R_Ctx(const Npp32u * pSrc1, int nSrc1Step, 
                                       const Npp32u * pSrc2, int nSrc2Step, 
                                       NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 32-bit signed image Maximum_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumError_32s_C1R_Ctx(const Npp32s * pSrc1, int nSrc1Step, 
                                       const Npp32s * pSrc2, int nSrc2Step, 
                                       NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 32-bit signed complex image Maximum_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumError_32sc_C1R_Ctx(const Npp32sc * pSrc1, int nSrc1Step, 
                                        const Npp32sc * pSrc2, int nSrc2Step, 
                                        NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 32-bit floating point image Maximum_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumError_32f_C1R_Ctx(const Npp32f * pSrc1, int nSrc1Step, 
                                       const Npp32f * pSrc2, int nSrc2Step, 
                                       NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 32-bit floating point complex image Maximum_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumError_32fc_C1R_Ctx(const Npp32fc * pSrc1, int nSrc1Step, 
                                        const Npp32fc * pSrc2, int nSrc2Step, 
                                        NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 64-bit floating point image Maximum_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumError_64f_C1R_Ctx(const Npp64f * pSrc1, int nSrc1Step, 
                                       const Npp64f * pSrc2, int nSrc2Step, 
                                       NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Two-channel 8-bit unsigned image Maximum_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumError_8u_C2R_Ctx(const Npp8u * pSrc1, int nSrc1Step, 
                                      const Npp8u * pSrc2, int nSrc2Step, 
                                      NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Two-channel 8-bit signed image Maximum_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumError_8s_C2R_Ctx(const Npp8s * pSrc1, int nSrc1Step, 
                                      const Npp8s * pSrc2, int nSrc2Step, 
                                      NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Two-channel 16-bit unsigned image Maximum_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumError_16u_C2R_Ctx(const Npp16u * pSrc1, int nSrc1Step, 
                                       const Npp16u * pSrc2, int nSrc2Step, 
                                       NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Two-channel 16-bit signed image Maximum_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumError_16s_C2R_Ctx(const Npp16s * pSrc1, int nSrc1Step, 
                                       const Npp16s * pSrc2, int nSrc2Step, 
                                       NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Two-channel 16-bit signed complex image Maximum_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumError_16sc_C2R_Ctx(const Npp16sc * pSrc1, int nSrc1Step, 
                                        const Npp16sc * pSrc2, int nSrc2Step, 
                                        NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Two-channel 32-bit unsigned image Maximum_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumError_32u_C2R_Ctx(const Npp32u * pSrc1, int nSrc1Step, 
                                       const Npp32u * pSrc2, int nSrc2Step, 
                                       NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Two-channel 32-bit signed image Maximum_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumError_32s_C2R_Ctx(const Npp32s * pSrc1, int nSrc1Step, 
                                       const Npp32s * pSrc2, int nSrc2Step, 
                                       NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Two-channel 32-bit signed complex image Maximum_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumError_32sc_C2R_Ctx(const Npp32sc * pSrc1, int nSrc1Step, 
                                        const Npp32sc * pSrc2, int nSrc2Step, 
                                        NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Two-channel 32-bit floating point image Maximum_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumError_32f_C2R_Ctx(const Npp32f * pSrc1, int nSrc1Step, 
                                       const Npp32f * pSrc2, int nSrc2Step, 
                                       NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Two-channel 32-bit floating point complex image Maximum_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumError_32fc_C2R_Ctx(const Npp32fc * pSrc1, int nSrc1Step, 
                                        const Npp32fc * pSrc2, int nSrc2Step, 
                                        NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Two-channel 64-bit floating point image Maximum_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumError_64f_C2R_Ctx(const Npp64f * pSrc1, int nSrc1Step, 
                                       const Npp64f * pSrc2, int nSrc2Step, 
                                       NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 8-bit unsigned image Maximum_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumError_8u_C3R_Ctx(const Npp8u * pSrc1, int nSrc1Step, 
                                      const Npp8u * pSrc2, int nSrc2Step, 
                                      NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 8-bit signed image Maximum_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumError_8s_C3R_Ctx(const Npp8s * pSrc1, int nSrc1Step, 
                                      const Npp8s * pSrc2, int nSrc2Step, 
                                      NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 16-bit unsigned image Maximum_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumError_16u_C3R_Ctx(const Npp16u * pSrc1, int nSrc1Step, 
                                       const Npp16u * pSrc2, int nSrc2Step, 
                                       NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 16-bit signed image Maximum_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumError_16s_C3R_Ctx(const Npp16s * pSrc1, int nSrc1Step, 
                                       const Npp16s * pSrc2, int nSrc2Step, 
                                       NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 16-bit signed complex image Maximum_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumError_16sc_C3R_Ctx(const Npp16sc * pSrc1, int nSrc1Step, 
                                        const Npp16sc * pSrc2, int nSrc2Step, 
                                        NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 32-bit unsigned image Maximum_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumError_32u_C3R_Ctx(const Npp32u * pSrc1, int nSrc1Step, 
                                       const Npp32u * pSrc2, int nSrc2Step, 
                                       NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 32-bit signed image Maximum_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumError_32s_C3R_Ctx(const Npp32s * pSrc1, int nSrc1Step, 
                                       const Npp32s * pSrc2, int nSrc2Step, 
                                       NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 32-bit signed complex image Maximum_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumError_32sc_C3R_Ctx(const Npp32sc * pSrc1, int nSrc1Step, 
                                        const Npp32sc * pSrc2, int nSrc2Step, 
                                        NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 32-bit floating point image Maximum_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumError_32f_C3R_Ctx(const Npp32f * pSrc1, int nSrc1Step, 
                                       const Npp32f * pSrc2, int nSrc2Step, 
                                       NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 32-bit floating point complex image Maximum_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumError_32fc_C3R_Ctx(const Npp32fc * pSrc1, int nSrc1Step, 
                                        const Npp32fc * pSrc2, int nSrc2Step, 
                                        NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 64-bit floating point image Maximum_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumError_64f_C3R_Ctx(const Npp64f * pSrc1, int nSrc1Step, 
                                       const Npp64f * pSrc2, int nSrc2Step, 
                                       NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image Maximum_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumError_8u_C4R_Ctx(const Npp8u * pSrc1, int nSrc1Step, 
                                      const Npp8u * pSrc2, int nSrc2Step, 
                                      NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit signed image Maximum_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumError_8s_C4R_Ctx(const Npp8s * pSrc1, int nSrc1Step, 
                                      const Npp8s * pSrc2, int nSrc2Step, 
                                      NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit unsigned image Maximum_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumError_16u_C4R_Ctx(const Npp16u * pSrc1, int nSrc1Step, 
                                       const Npp16u * pSrc2, int nSrc2Step, 
                                       NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit signed image Maximum_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumError_16s_C4R_Ctx(const Npp16s * pSrc1, int nSrc1Step, 
                                       const Npp16s * pSrc2, int nSrc2Step, 
                                       NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit signed complex image Maximum_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumError_16sc_C4R_Ctx(const Npp16sc * pSrc1, int nSrc1Step, 
                                        const Npp16sc * pSrc2, int nSrc2Step, 
                                        NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit unsigned image Maximum_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumError_32u_C4R_Ctx(const Npp32u * pSrc1, int nSrc1Step, 
                                       const Npp32u * pSrc2, int nSrc2Step, 
                                       NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit signed image Maximum_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumError_32s_C4R_Ctx(const Npp32s * pSrc1, int nSrc1Step, 
                                       const Npp32s * pSrc2, int nSrc2Step, 
                                       NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit signed complex image Maximum_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumError_32sc_C4R_Ctx(const Npp32sc * pSrc1, int nSrc1Step, 
                                        const Npp32sc * pSrc2, int nSrc2Step, 
                                        NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit floating point image Maximum_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumError_32f_C4R_Ctx(const Npp32f * pSrc1, int nSrc1Step, 
                                       const Npp32f * pSrc2, int nSrc2Step, 
                                       NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit floating point complex image Maximum_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumError_32fc_C4R_Ctx(const Npp32fc * pSrc1, int nSrc1Step, 
                                        const Npp32fc * pSrc2, int nSrc2Step, 
                                        NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 64-bit floating point image Maximum_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumError_64f_C4R_Ctx(const Npp64f * pSrc1, int nSrc1Step, 
                                       const Npp64f * pSrc2, int nSrc2Step, 
                                       NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);

/** @} */

/** @name MaximumErrorGetBufferHostSize
 * Companion primitives for computing the device buffer size (in bytes) required by the MaximumError primitives.
 * \section CommonMaximumErrorGetBufferHostSizeParameters
 * Common parameters for nppiMaximumErrorGetBufferHostSize functions include:
 * \param oSizeROI \ref roi_specification.
 * \param hpBufferSize Required buffer size. Important: hpBufferSize is a 
 *        <em>host pointer.</em> \ref general_scratch_buffer.
 * \param nppStreamCtx \ref application_managed_stream_context. 
 * \return NPP_NULL_POINTER_ERROR if hpBufferSize is 0 (NULL),  \ref roi_error_codes.  
 * @{
 */
/** 
 * Buffer size for \ref nppiMaximumError_8u_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumErrorGetBufferHostSize_8u_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumError_8s_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumErrorGetBufferHostSize_8s_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumError_16u_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumErrorGetBufferHostSize_16u_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumError_16s_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumErrorGetBufferHostSize_16s_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumError_16sc_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumErrorGetBufferHostSize_16sc_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumError_32u_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumErrorGetBufferHostSize_32u_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumError_32s_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumErrorGetBufferHostSize_32s_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumError_32sc_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumErrorGetBufferHostSize_32sc_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumError_32f_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumErrorGetBufferHostSize_32f_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumError_32fc_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumErrorGetBufferHostSize_32fc_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumError_64f_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumErrorGetBufferHostSize_64f_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumError_8u_C2R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumErrorGetBufferHostSize_8u_C2R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumError_8s_C2R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumErrorGetBufferHostSize_8s_C2R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumError_16u_C2R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumErrorGetBufferHostSize_16u_C2R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumError_16s_C2R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumErrorGetBufferHostSize_16s_C2R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumError_16sc_C2R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumErrorGetBufferHostSize_16sc_C2R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumError_32u_C2R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumErrorGetBufferHostSize_32u_C2R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumError_32s_C2R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumErrorGetBufferHostSize_32s_C2R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumError_32sc_C2R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumErrorGetBufferHostSize_32sc_C2R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumError_32f_C2R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumErrorGetBufferHostSize_32f_C2R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumError_32fc_C2R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumErrorGetBufferHostSize_32fc_C2R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumError_64f_C2R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumErrorGetBufferHostSize_64f_C2R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumError_8u_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumErrorGetBufferHostSize_8u_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumError_8s_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumErrorGetBufferHostSize_8s_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumError_16u_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumErrorGetBufferHostSize_16u_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumError_16s_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumErrorGetBufferHostSize_16s_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumError_16sc_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumErrorGetBufferHostSize_16sc_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumError_32u_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumErrorGetBufferHostSize_32u_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumError_32s_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumErrorGetBufferHostSize_32s_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumError_32sc_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumErrorGetBufferHostSize_32sc_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumError_32f_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumErrorGetBufferHostSize_32f_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumError_32fc_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumErrorGetBufferHostSize_32fc_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumError_64f_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumErrorGetBufferHostSize_64f_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumError_8u_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumErrorGetBufferHostSize_8u_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumError_8s_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumErrorGetBufferHostSize_8s_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumError_16u_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumErrorGetBufferHostSize_16u_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumError_16s_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumErrorGetBufferHostSize_16s_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumError_16sc_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumErrorGetBufferHostSize_16sc_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumError_32u_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumErrorGetBufferHostSize_32u_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumError_32s_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumErrorGetBufferHostSize_32s_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumError_32sc_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumErrorGetBufferHostSize_32sc_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumError_32f_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumErrorGetBufferHostSize_32f_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumError_32fc_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumErrorGetBufferHostSize_32fc_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumError_64f_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumErrorGetBufferHostSize_64f_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);

/** @} */ 
/** @} image_maximum_error */

/** 
 * \section image_average_error AverageError
 * @defgroup image_average_error AverageError
 * Primitives for computing the average error between two images.
 * Given two images \f$pSrc1\f$ and \f$pSrc2\f$ both with width \f$W\f$ and height \f$H\f$, 
 * the average error is defined as:
 * \f[Average Error = \frac{1}{W\cdot H\cdot N}\sum_{n=0}^{N-1}\sum_{j=0}^{H-1}\sum_{i=0}^{W-1}\left|pSrc1(j,i) - pSrc2(j,i)\right|\f]
 * where N stands for the number of channels. If the image is in complex format, the absolute value is used for computation.
 * @{
 */

/** @name AverageError
 * @{
 */
/**
 * One-channel 8-bit unsigned image Average_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageError_8u_C1R_Ctx(const Npp8u * pSrc1, int nSrc1Step, 
                                      const Npp8u * pSrc2, int nSrc2Step, 
                                      NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 8-bit signed image Average_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageError_8s_C1R_Ctx(const Npp8s * pSrc1, int nSrc1Step, 
                                      const Npp8s * pSrc2, int nSrc2Step, 
                                      NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 16-bit unsigned image Average_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageError_16u_C1R_Ctx(const Npp16u * pSrc1, int nSrc1Step, 
                                       const Npp16u * pSrc2, int nSrc2Step, 
                                       NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 16-bit signed image Average_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageError_16s_C1R_Ctx(const Npp16s * pSrc1, int nSrc1Step, 
                                       const Npp16s * pSrc2, int nSrc2Step, 
                                       NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 16-bit signed complex image Average_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageError_16sc_C1R_Ctx(const Npp16sc * pSrc1, int nSrc1Step, 
                                        const Npp16sc * pSrc2, int nSrc2Step, 
                                        NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 32-bit unsigned image Average_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageError_32u_C1R_Ctx(const Npp32u * pSrc1, int nSrc1Step, 
                                       const Npp32u * pSrc2, int nSrc2Step, 
                                       NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 32-bit signed image Average_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageError_32s_C1R_Ctx(const Npp32s * pSrc1, int nSrc1Step, 
                                       const Npp32s * pSrc2, int nSrc2Step, 
                                       NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 32-bit signed complex image Average_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageError_32sc_C1R_Ctx(const Npp32sc * pSrc1, int nSrc1Step, 
                                        const Npp32sc * pSrc2, int nSrc2Step, 
                                        NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 32-bit floating point image Average_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageError_32f_C1R_Ctx(const Npp32f * pSrc1, int nSrc1Step, 
                                       const Npp32f * pSrc2, int nSrc2Step, 
                                       NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 32-bit floating point complex image Average_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageError_32fc_C1R_Ctx(const Npp32fc * pSrc1, int nSrc1Step, 
                                        const Npp32fc * pSrc2, int nSrc2Step, 
                                        NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 64-bit floating point image Average_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageError_64f_C1R_Ctx(const Npp64f * pSrc1, int nSrc1Step, 
                                       const Npp64f * pSrc2, int nSrc2Step, 
                                       NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Two-channel 8-bit unsigned image Average_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageError_8u_C2R_Ctx(const Npp8u * pSrc1, int nSrc1Step, 
                                      const Npp8u * pSrc2, int nSrc2Step, 
                                      NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Two-channel 8-bit signed image Average_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageError_8s_C2R_Ctx(const Npp8s * pSrc1, int nSrc1Step, 
                                      const Npp8s * pSrc2, int nSrc2Step, 
                                      NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Two-channel 16-bit unsigned image Average_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageError_16u_C2R_Ctx(const Npp16u * pSrc1, int nSrc1Step, 
                                       const Npp16u * pSrc2, int nSrc2Step, 
                                       NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Two-channel 16-bit signed image Average_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageError_16s_C2R_Ctx(const Npp16s * pSrc1, int nSrc1Step, 
                                       const Npp16s * pSrc2, int nSrc2Step, 
                                       NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Two-channel 16-bit signed complex image Average_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageError_16sc_C2R_Ctx(const Npp16sc * pSrc1, int nSrc1Step, 
                                        const Npp16sc * pSrc2, int nSrc2Step, 
                                        NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Two-channel 32-bit unsigned image Average_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageError_32u_C2R_Ctx(const Npp32u * pSrc1, int nSrc1Step, 
                                       const Npp32u * pSrc2, int nSrc2Step, 
                                       NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Two-channel 32-bit signed image Average_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageError_32s_C2R_Ctx(const Npp32s * pSrc1, int nSrc1Step, 
                                       const Npp32s * pSrc2, int nSrc2Step, 
                                       NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Two-channel 32-bit signed complex image Average_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageError_32sc_C2R_Ctx(const Npp32sc * pSrc1, int nSrc1Step, 
                                        const Npp32sc * pSrc2, int nSrc2Step, 
                                        NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Two-channel 32-bit floating point image Average_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageError_32f_C2R_Ctx(const Npp32f * pSrc1, int nSrc1Step, 
                                       const Npp32f * pSrc2, int nSrc2Step, 
                                       NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Two-channel 32-bit floating point complex image Average_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageError_32fc_C2R_Ctx(const Npp32fc * pSrc1, int nSrc1Step, 
                                        const Npp32fc * pSrc2, int nSrc2Step, 
                                        NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Two-channel 64-bit floating point image Average_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageError_64f_C2R_Ctx(const Npp64f * pSrc1, int nSrc1Step, 
                                       const Npp64f * pSrc2, int nSrc2Step, 
                                       NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 8-bit unsigned image Average_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageError_8u_C3R_Ctx(const Npp8u * pSrc1, int nSrc1Step, 
                                      const Npp8u * pSrc2, int nSrc2Step, 
                                      NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 8-bit signed image Average_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageError_8s_C3R_Ctx(const Npp8s * pSrc1, int nSrc1Step, 
                                      const Npp8s * pSrc2, int nSrc2Step, 
                                      NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 16-bit unsigned image Average_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageError_16u_C3R_Ctx(const Npp16u * pSrc1, int nSrc1Step, 
                                       const Npp16u * pSrc2, int nSrc2Step, 
                                       NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 16-bit signed image Average_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageError_16s_C3R_Ctx(const Npp16s * pSrc1, int nSrc1Step, 
                                       const Npp16s * pSrc2, int nSrc2Step, 
                                       NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 16-bit signed complex image Average_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageError_16sc_C3R_Ctx(const Npp16sc * pSrc1, int nSrc1Step, 
                                        const Npp16sc * pSrc2, int nSrc2Step, 
                                        NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 32-bit unsigned image Average_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageError_32u_C3R_Ctx(const Npp32u * pSrc1, int nSrc1Step, 
                                       const Npp32u * pSrc2, int nSrc2Step, 
                                       NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 32-bit signed image Average_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageError_32s_C3R_Ctx(const Npp32s * pSrc1, int nSrc1Step, 
                                       const Npp32s * pSrc2, int nSrc2Step, 
                                       NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 32-bit signed complex image Average_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageError_32sc_C3R_Ctx(const Npp32sc * pSrc1, int nSrc1Step, 
                                        const Npp32sc * pSrc2, int nSrc2Step, 
                                        NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 32-bit floating point image Average_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageError_32f_C3R_Ctx(const Npp32f * pSrc1, int nSrc1Step, 
                                       const Npp32f * pSrc2, int nSrc2Step, 
                                       NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 32-bit floating point complex image Average_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageError_32fc_C3R_Ctx(const Npp32fc * pSrc1, int nSrc1Step, 
                                        const Npp32fc * pSrc2, int nSrc2Step, 
                                        NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 64-bit floating point image Average_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageError_64f_C3R_Ctx(const Npp64f * pSrc1, int nSrc1Step, 
                                       const Npp64f * pSrc2, int nSrc2Step, 
                                       NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image Average_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageError_8u_C4R_Ctx(const Npp8u * pSrc1, int nSrc1Step, 
                                      const Npp8u * pSrc2, int nSrc2Step, 
                                      NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit signed image Average_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageError_8s_C4R_Ctx(const Npp8s * pSrc1, int nSrc1Step, 
                                      const Npp8s * pSrc2, int nSrc2Step, 
                                      NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit unsigned image Average_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageError_16u_C4R_Ctx(const Npp16u * pSrc1, int nSrc1Step, 
                                       const Npp16u * pSrc2, int nSrc2Step, 
                                       NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit signed image Average_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageError_16s_C4R_Ctx(const Npp16s * pSrc1, int nSrc1Step, 
                                       const Npp16s * pSrc2, int nSrc2Step, 
                                       NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit signed complex image Average_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageError_16sc_C4R_Ctx(const Npp16sc * pSrc1, int nSrc1Step, 
                                        const Npp16sc * pSrc2, int nSrc2Step, 
                                        NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit unsigned image Average_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageError_32u_C4R_Ctx(const Npp32u * pSrc1, int nSrc1Step, 
                                       const Npp32u * pSrc2, int nSrc2Step, 
                                       NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit signed image Average_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageError_32s_C4R_Ctx(const Npp32s * pSrc1, int nSrc1Step, 
                                       const Npp32s * pSrc2, int nSrc2Step, 
                                       NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit signed complex image Average_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageError_32sc_C4R_Ctx(const Npp32sc * pSrc1, int nSrc1Step, 
                                        const Npp32sc * pSrc2, int nSrc2Step, 
                                        NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit floating point image Average_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageError_32f_C4R_Ctx(const Npp32f * pSrc1, int nSrc1Step, 
                                       const Npp32f * pSrc2, int nSrc2Step, 
                                       NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit floating point complex image Average_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageError_32fc_C4R_Ctx(const Npp32fc * pSrc1, int nSrc1Step, 
                                        const Npp32fc * pSrc2, int nSrc2Step, 
                                        NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 64-bit floating point image Average_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageError_64f_C4R_Ctx(const Npp64f * pSrc1, int nSrc1Step, 
                                       const Npp64f * pSrc2, int nSrc2Step, 
                                       NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);

/** @} */

/** @name AverageErrorGetBufferHostSize
 * Companion primitives for computing the device buffer size (in bytes) required by the NormDiff_Inf primitives.
 * \section CommonAverageErrorGetBufferHostSizeParameters
 * Common parameters for nppiSumAverageErrorBufferHostSize functions include:
 * \param oSizeROI \ref roi_specification.
 * \param hpBufferSize Required buffer size. Important: hpBufferSize is a 
 *        <em>host pointer.</em> \ref general_scratch_buffer.
 * \param nppStreamCtx \ref application_managed_stream_context. 
 * \return NPP_NULL_POINTER_ERROR if hpBufferSize is 0 (NULL),  \ref roi_error_codes.  
 * @{
 */
/** 
 * Buffer size for \ref nppiAverageError_8u_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageErrorGetBufferHostSize_8u_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageError_8s_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageErrorGetBufferHostSize_8s_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageError_16u_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageErrorGetBufferHostSize_16u_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageError_16s_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageErrorGetBufferHostSize_16s_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageError_16sc_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageErrorGetBufferHostSize_16sc_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageError_32u_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageErrorGetBufferHostSize_32u_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageError_32s_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageErrorGetBufferHostSize_32s_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageError_32sc_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageErrorGetBufferHostSize_32sc_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageError_32f_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageErrorGetBufferHostSize_32f_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageError_32fc_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageErrorGetBufferHostSize_32fc_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageError_64f_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageErrorGetBufferHostSize_64f_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageError_8u_C2R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageErrorGetBufferHostSize_8u_C2R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageError_8s_C2R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageErrorGetBufferHostSize_8s_C2R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageError_16u_C2R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageErrorGetBufferHostSize_16u_C2R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageError_16s_C2R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageErrorGetBufferHostSize_16s_C2R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageError_16sc_C2R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageErrorGetBufferHostSize_16sc_C2R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageError_32u_C2R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageErrorGetBufferHostSize_32u_C2R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageError_32s_C2R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageErrorGetBufferHostSize_32s_C2R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageError_32sc_C2R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageErrorGetBufferHostSize_32sc_C2R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageError_32f_C2R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageErrorGetBufferHostSize_32f_C2R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageError_32fc_C2R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageErrorGetBufferHostSize_32fc_C2R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageError_64f_C2R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageErrorGetBufferHostSize_64f_C2R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageError_8u_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageErrorGetBufferHostSize_8u_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageError_8s_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageErrorGetBufferHostSize_8s_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageError_16u_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageErrorGetBufferHostSize_16u_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageError_16s_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageErrorGetBufferHostSize_16s_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageError_16sc_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageErrorGetBufferHostSize_16sc_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageError_32u_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageErrorGetBufferHostSize_32u_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageError_32s_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageErrorGetBufferHostSize_32s_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageError_32sc_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageErrorGetBufferHostSize_32sc_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageError_32f_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageErrorGetBufferHostSize_32f_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageError_32fc_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageErrorGetBufferHostSize_32fc_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageError_64f_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageErrorGetBufferHostSize_64f_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageError_8u_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageErrorGetBufferHostSize_8u_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageError_8s_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageErrorGetBufferHostSize_8s_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageError_16u_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageErrorGetBufferHostSize_16u_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageError_16s_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageErrorGetBufferHostSize_16s_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageError_16sc_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageErrorGetBufferHostSize_16sc_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageError_32u_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageErrorGetBufferHostSize_32u_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageError_32s_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageErrorGetBufferHostSize_32s_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageError_32sc_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageErrorGetBufferHostSize_32sc_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageError_32f_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageErrorGetBufferHostSize_32f_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageError_32fc_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageErrorGetBufferHostSize_32fc_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageError_64f_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageErrorGetBufferHostSize_64f_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);

/** @} */
/** @} image_average_error */

/** 
 * \section image_maximum_relative_error MaximumRelativeError
 * @defgroup image_maximum_relative_error MaximumRelativeError
 * Primitives for computing the maximum relative error between two images.
 * Given two images \f$pSrc1\f$ and \f$pSrc2\f$ both with width \f$W\f$ and height \f$H\f$, 
 * the maximum relative error is defined as:
 * \f[MaximumRelativeError = max{\frac{\left|pSrc1(j,i) - pSrc2(j,i)\right|}{max(\left|pSrc1(j,i)\right|, \left|pSrc2(j,i)\right|)}}\f]
 * If the image is in complex format, the absolute value is used for computation.
 * For multiple channles, the maximum relative error of all the channles is returned.
 * @{
 */

/** @name MaximumRelativeError
 * @{
 */
/**
 * One-channel 8-bit unsigned image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumRelativeError_8u_C1R_Ctx(const Npp8u * pSrc1, int nSrc1Step, 
                                              const Npp8u * pSrc2, int nSrc2Step, 
                                              NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 8-bit signed image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumRelativeError_8s_C1R_Ctx(const Npp8s * pSrc1, int nSrc1Step, 
                                              const Npp8s * pSrc2, int nSrc2Step, 
                                              NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 16-bit unsigned image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumRelativeError_16u_C1R_Ctx(const Npp16u * pSrc1, int nSrc1Step, 
                                               const Npp16u * pSrc2, int nSrc2Step, 
                                               NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 16-bit signed image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumRelativeError_16s_C1R_Ctx(const Npp16s * pSrc1, int nSrc1Step, 
                                               const Npp16s * pSrc2, int nSrc2Step, 
                                               NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 16-bit signed complex image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumRelativeError_16sc_C1R_Ctx(const Npp16sc * pSrc1, int nSrc1Step, 
                                                const Npp16sc * pSrc2, int nSrc2Step, 
                                                NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 32-bit unsigned image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumRelativeError_32u_C1R_Ctx(const Npp32u * pSrc1, int nSrc1Step, 
                                               const Npp32u * pSrc2, int nSrc2Step, 
                                               NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 32-bit signed image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumRelativeError_32s_C1R_Ctx(const Npp32s * pSrc1, int nSrc1Step, 
                                               const Npp32s * pSrc2, int nSrc2Step, 
                                               NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 32-bit signed complex image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumRelativeError_32sc_C1R_Ctx(const Npp32sc * pSrc1, int nSrc1Step, 
                                                const Npp32sc * pSrc2, int nSrc2Step, 
                                                NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 32-bit floating point image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumRelativeError_32f_C1R_Ctx(const Npp32f * pSrc1, int nSrc1Step, 
                                               const Npp32f * pSrc2, int nSrc2Step, 
                                               NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 32-bit floating point complex image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumRelativeError_32fc_C1R_Ctx(const Npp32fc * pSrc1, int nSrc1Step, 
                                                const Npp32fc * pSrc2, int nSrc2Step, 
                                                NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 64-bit floating point image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumRelativeError_64f_C1R_Ctx(const Npp64f * pSrc1, int nSrc1Step, 
                                               const Npp64f * pSrc2, int nSrc2Step, 
                                               NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Two-channel 8-bit unsigned image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumRelativeError_8u_C2R_Ctx(const Npp8u * pSrc1, int nSrc1Step, 
                                              const Npp8u * pSrc2, int nSrc2Step, 
                                              NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Two-channel 8-bit signed image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumRelativeError_8s_C2R_Ctx(const Npp8s * pSrc1, int nSrc1Step, 
                                              const Npp8s * pSrc2, int nSrc2Step, 
                                              NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Two-channel 16-bit unsigned image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumRelativeError_16u_C2R_Ctx(const Npp16u * pSrc1, int nSrc1Step, 
                                               const Npp16u * pSrc2, int nSrc2Step, 
                                               NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Two-channel 16-bit signed image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumRelativeError_16s_C2R_Ctx(const Npp16s * pSrc1, int nSrc1Step, 
                                               const Npp16s * pSrc2, int nSrc2Step, 
                                               NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Two-channel 16-bit signed complex image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumRelativeError_16sc_C2R_Ctx(const Npp16sc * pSrc1, int nSrc1Step, 
                                                const Npp16sc * pSrc2, int nSrc2Step, 
                                                NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Two-channel 32-bit unsigned image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumRelativeError_32u_C2R_Ctx(const Npp32u * pSrc1, int nSrc1Step, 
                                               const Npp32u * pSrc2, int nSrc2Step, 
                                               NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Two-channel 32-bit signed image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumRelativeError_32s_C2R_Ctx(const Npp32s * pSrc1, int nSrc1Step, 
                                               const Npp32s * pSrc2, int nSrc2Step, 
                                               NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Two-channel 32-bit signed complex image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumRelativeError_32sc_C2R_Ctx(const Npp32sc * pSrc1, int nSrc1Step, 
                                                const Npp32sc * pSrc2, int nSrc2Step, 
                                                NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Two-channel 32-bit floating point image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumRelativeError_32f_C2R_Ctx(const Npp32f * pSrc1, int nSrc1Step, 
                                               const Npp32f * pSrc2, int nSrc2Step, 
                                               NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Two-channel 32-bit floating point complex image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumRelativeError_32fc_C2R_Ctx(const Npp32fc * pSrc1, int nSrc1Step, 
                                                const Npp32fc * pSrc2, int nSrc2Step, 
                                                NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Two-channel 64-bit floating point image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumRelativeError_64f_C2R_Ctx(const Npp64f * pSrc1, int nSrc1Step, 
                                               const Npp64f * pSrc2, int nSrc2Step, 
                                               NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 8-bit unsigned image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumRelativeError_8u_C3R_Ctx(const Npp8u * pSrc1, int nSrc1Step, 
                                              const Npp8u * pSrc2, int nSrc2Step, 
                                              NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 8-bit signed image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumRelativeError_8s_C3R_Ctx(const Npp8s * pSrc1, int nSrc1Step, 
                                              const Npp8s * pSrc2, int nSrc2Step, 
                                              NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 16-bit unsigned image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumRelativeError_16u_C3R_Ctx(const Npp16u * pSrc1, int nSrc1Step, 
                                               const Npp16u * pSrc2, int nSrc2Step, 
                                               NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 16-bit signed image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumRelativeError_16s_C3R_Ctx(const Npp16s * pSrc1, int nSrc1Step, 
                                               const Npp16s * pSrc2, int nSrc2Step, 
                                               NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 16-bit signed complex image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumRelativeError_16sc_C3R_Ctx(const Npp16sc * pSrc1, int nSrc1Step, 
                                                const Npp16sc * pSrc2, int nSrc2Step, 
                                                NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 32-bit unsigned image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumRelativeError_32u_C3R_Ctx(const Npp32u * pSrc1, int nSrc1Step, 
                                               const Npp32u * pSrc2, int nSrc2Step, 
                                               NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 32-bit signed image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumRelativeError_32s_C3R_Ctx(const Npp32s * pSrc1, int nSrc1Step, 
                                               const Npp32s * pSrc2, int nSrc2Step, 
                                               NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 32-bit signed complex image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumRelativeError_32sc_C3R_Ctx(const Npp32sc * pSrc1, int nSrc1Step, 
                                                const Npp32sc * pSrc2, int nSrc2Step, 
                                                NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 32-bit floating point image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumRelativeError_32f_C3R_Ctx(const Npp32f * pSrc1, int nSrc1Step, 
                                               const Npp32f * pSrc2, int nSrc2Step, 
                                               NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 32-bit floating point complex image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumRelativeError_32fc_C3R_Ctx(const Npp32fc * pSrc1, int nSrc1Step, 
                                                const Npp32fc * pSrc2, int nSrc2Step, 
                                                NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 64-bit floating point image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumRelativeError_64f_C3R_Ctx(const Npp64f * pSrc1, int nSrc1Step, 
                                               const Npp64f * pSrc2, int nSrc2Step, 
                                               NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumRelativeError_8u_C4R_Ctx(const Npp8u * pSrc1, int nSrc1Step, 
                                              const Npp8u * pSrc2, int nSrc2Step, 
                                              NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit signed image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumRelativeError_8s_C4R_Ctx(const Npp8s * pSrc1, int nSrc1Step, 
                                              const Npp8s * pSrc2, int nSrc2Step, 
                                              NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit unsigned image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters. 
 */ 
NppStatus nppiMaximumRelativeError_16u_C4R_Ctx(const Npp16u * pSrc1, int nSrc1Step, 
                                               const Npp16u * pSrc2, int nSrc2Step, 
                                               NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit signed image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumRelativeError_16s_C4R_Ctx(const Npp16s * pSrc1, int nSrc1Step, 
                                               const Npp16s * pSrc2, int nSrc2Step, 
                                               NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit signed complex image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumRelativeError_16sc_C4R_Ctx(const Npp16sc * pSrc1, int nSrc1Step, 
                                                const Npp16sc * pSrc2, int nSrc2Step, 
                                                NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit unsigned image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumRelativeError_32u_C4R_Ctx(const Npp32u * pSrc1, int nSrc1Step, 
                                               const Npp32u * pSrc2, int nSrc2Step, 
                                               NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit signed image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumRelativeError_32s_C4R_Ctx(const Npp32s * pSrc1, int nSrc1Step, 
                                               const Npp32s * pSrc2, int nSrc2Step, 
                                               NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit signed complex image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumRelativeError_32sc_C4R_Ctx(const Npp32sc * pSrc1, int nSrc1Step, 
                                                const Npp32sc * pSrc2, int nSrc2Step, 
                                                NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit floating point image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumRelativeError_32f_C4R_Ctx(const Npp32f * pSrc1, int nSrc1Step, 
                                               const Npp32f * pSrc2, int nSrc2Step, 
                                               NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit floating point complex image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumRelativeError_32fc_C4R_Ctx(const Npp32fc * pSrc1, int nSrc1Step, 
                                                const Npp32fc * pSrc2, int nSrc2Step, 
                                                NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 64-bit floating point image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiMaximumRelativeError_64f_C4R_Ctx(const Npp64f * pSrc1, int nSrc1Step, 
                                               const Npp64f * pSrc2, int nSrc2Step, 
                                               NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);

/** @} */

/** @name MaximumRelativeErrorGetBufferHostSize
 * Companion primitives for computing the device buffer size (in bytes) required by the NormDiff_Inf primitives.
 * \section CommonMaximumRelativeErrorGetBufferHostSizeParameters
 * Common parameters for nppiMaximumRelativeErrorGetBufferHostSize functions include:
 * \param oSizeROI \ref roi_specification.
 * \param hpBufferSize Required buffer size. Important: hpBufferSize is a 
 *        <em>host pointer.</em> \ref general_scratch_buffer.
 * \param nppStreamCtx \ref application_managed_stream_context. 
 * \return NPP_NULL_POINTER_ERROR if hpBufferSize is 0 (NULL),  \ref roi_error_codes.  
 * @{
 */
/** 
 * Buffer size for \ref nppiMaximumRelativeError_8u_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumRelativeErrorGetBufferHostSize_8u_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumRelativeError_8s_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumRelativeErrorGetBufferHostSize_8s_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumRelativeError_16u_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumRelativeErrorGetBufferHostSize_16u_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumRelativeError_16s_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumRelativeErrorGetBufferHostSize_16s_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumRelativeError_16sc_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumRelativeErrorGetBufferHostSize_16sc_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumRelativeError_32u_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumRelativeErrorGetBufferHostSize_32u_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumRelativeError_32s_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumRelativeErrorGetBufferHostSize_32s_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumRelativeError_32sc_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumRelativeErrorGetBufferHostSize_32sc_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumRelativeError_32f_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumRelativeErrorGetBufferHostSize_32f_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumRelativeError_32fc_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumRelativeErrorGetBufferHostSize_32fc_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumRelativeError_64f_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumRelativeErrorGetBufferHostSize_64f_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumRelativeError_8u_C2R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumRelativeErrorGetBufferHostSize_8u_C2R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumRelativeError_8s_C2R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumRelativeErrorGetBufferHostSize_8s_C2R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumRelativeError_16u_C2R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumRelativeErrorGetBufferHostSize_16u_C2R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumRelativeError_16s_C2R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumRelativeErrorGetBufferHostSize_16s_C2R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumRelativeError_16sc_C2R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumRelativeErrorGetBufferHostSize_16sc_C2R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumRelativeError_32u_C2R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumRelativeErrorGetBufferHostSize_32u_C2R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumRelativeError_32s_C2R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumRelativeErrorGetBufferHostSize_32s_C2R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumRelativeError_32sc_C2R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumRelativeErrorGetBufferHostSize_32sc_C2R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumRelativeError_32f_C2R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumRelativeErrorGetBufferHostSize_32f_C2R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumRelativeError_32fc_C2R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumRelativeErrorGetBufferHostSize_32fc_C2R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumRelativeError_64f_C2R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumRelativeErrorGetBufferHostSize_64f_C2R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumRelativeError_8u_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumRelativeErrorGetBufferHostSize_8u_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumRelativeError_8s_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumRelativeErrorGetBufferHostSize_8s_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumRelativeError_16u_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumRelativeErrorGetBufferHostSize_16u_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumRelativeError_16s_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumRelativeErrorGetBufferHostSize_16s_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumRelativeError_16sc_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumRelativeErrorGetBufferHostSize_16sc_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumRelativeError_32u_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumRelativeErrorGetBufferHostSize_32u_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumRelativeError_32s_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumRelativeErrorGetBufferHostSize_32s_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumRelativeError_32sc_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumRelativeErrorGetBufferHostSize_32sc_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumRelativeError_32f_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumRelativeErrorGetBufferHostSize_32f_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumRelativeError_32fc_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumRelativeErrorGetBufferHostSize_32fc_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumRelativeError_64f_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumRelativeErrorGetBufferHostSize_64f_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumRelativeError_8u_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumRelativeErrorGetBufferHostSize_8u_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumRelativeError_8s_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumRelativeErrorGetBufferHostSize_8s_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumRelativeError_16u_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumRelativeErrorGetBufferHostSize_16u_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumRelativeError_16s_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumRelativeErrorGetBufferHostSize_16s_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumRelativeError_16sc_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumRelativeErrorGetBufferHostSize_16sc_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumRelativeError_32u_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumRelativeErrorGetBufferHostSize_32u_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumRelativeError_32s_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumRelativeErrorGetBufferHostSize_32s_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumRelativeError_32sc_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumRelativeErrorGetBufferHostSize_32sc_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumRelativeError_32f_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumRelativeErrorGetBufferHostSize_32f_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumRelativeError_32fc_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumRelativeErrorGetBufferHostSize_32fc_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMaximumRelativeError_64f_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonMaximumRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiMaximumRelativeErrorGetBufferHostSize_64f_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);

/** @} */
/** @} image_maximum_relative_error */

/** 
 * \section image_average_relative_error AverageRelativeError
 * @defgroup image_average_relative_error AverageRelativeError
 * Primitives for computing the average relative error between two images.
 * Given two images \f$pSrc1\f$ and \f$pSrc2\f$ both with width \f$W\f$ and height \f$H\f$, 
 * the maximum relative error is defined as:
 * \f[AverageRelativeError = \frac{1}{W\cdot H\cdot N}\sum_{n=0}^{N-1}\sum_{j=0}^{H-1}\sum_{i=0}^{W-1}\frac{\left|pSrc1(j,i) - pSrc2(j,i)\right|}{max(\left|pSrc1(j,i)\right|, \left|pSrc2(j,i)\right|)}\f]
 * where N is the number of channels.
 * If the image is in complex format, the absolute value is used for computation.
 * @{
 */

/** @name AverageRelativeError
 * @{
 */

/**
 * One-channel 8-bit unsigned image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageRelativeError_8u_C1R_Ctx(const Npp8u * pSrc1, int nSrc1Step, 
                                              const Npp8u * pSrc2, int nSrc2Step, 
                                              NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 8-bit signed image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageRelativeError_8s_C1R_Ctx(const Npp8s * pSrc1, int nSrc1Step, 
                                              const Npp8s * pSrc2, int nSrc2Step, 
                                              NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 16-bit unsigned image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageRelativeError_16u_C1R_Ctx(const Npp16u * pSrc1, int nSrc1Step, 
                                               const Npp16u * pSrc2, int nSrc2Step, 
                                               NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 16-bit signed image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageRelativeError_16s_C1R_Ctx(const Npp16s * pSrc1, int nSrc1Step, 
                                               const Npp16s * pSrc2, int nSrc2Step, 
                                               NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 16-bit signed complex image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageRelativeError_16sc_C1R_Ctx(const Npp16sc * pSrc1, int nSrc1Step, 
                                                const Npp16sc * pSrc2, int nSrc2Step, 
                                                NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 32-bit unsigned image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageRelativeError_32u_C1R_Ctx(const Npp32u * pSrc1, int nSrc1Step, 
                                               const Npp32u * pSrc2, int nSrc2Step, 
                                               NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 32-bit signed image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageRelativeError_32s_C1R_Ctx(const Npp32s * pSrc1, int nSrc1Step, 
                                               const Npp32s * pSrc2, int nSrc2Step, 
                                               NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 32-bit signed complex image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageRelativeError_32sc_C1R_Ctx(const Npp32sc * pSrc1, int nSrc1Step, 
                                                const Npp32sc * pSrc2, int nSrc2Step, 
                                                NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 32-bit floating point image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageRelativeError_32f_C1R_Ctx(const Npp32f * pSrc1, int nSrc1Step, 
                                               const Npp32f * pSrc2, int nSrc2Step, 
                                               NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 32-bit floating point complex image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageRelativeError_32fc_C1R_Ctx(const Npp32fc * pSrc1, int nSrc1Step, 
                                                const Npp32fc * pSrc2, int nSrc2Step, 
                                                NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * One-channel 64-bit floating point image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageRelativeError_64f_C1R_Ctx(const Npp64f * pSrc1, int nSrc1Step, 
                                               const Npp64f * pSrc2, int nSrc2Step, 
                                               NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Two-channel 8-bit unsigned image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageRelativeError_8u_C2R_Ctx(const Npp8u * pSrc1, int nSrc1Step, 
                                              const Npp8u * pSrc2, int nSrc2Step, 
                                              NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Two-channel 8-bit signed image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageRelativeError_8s_C2R_Ctx(const Npp8s * pSrc1, int nSrc1Step, 
                                              const Npp8s * pSrc2, int nSrc2Step, 
                                              NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Two-channel 16-bit unsigned image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageRelativeError_16u_C2R_Ctx(const Npp16u * pSrc1, int nSrc1Step, 
                                               const Npp16u * pSrc2, int nSrc2Step, 
                                               NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Two-channel 16-bit signed image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageRelativeError_16s_C2R_Ctx(const Npp16s * pSrc1, int nSrc1Step, 
                                               const Npp16s * pSrc2, int nSrc2Step, 
                                               NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Two-channel 16-bit signed complex image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageRelativeError_16sc_C2R_Ctx(const Npp16sc * pSrc1, int nSrc1Step, 
                                                const Npp16sc * pSrc2, int nSrc2Step, 
                                                NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Two-channel 32-bit unsigned image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageRelativeError_32u_C2R_Ctx(const Npp32u * pSrc1, int nSrc1Step, 
                                               const Npp32u * pSrc2, int nSrc2Step, 
                                               NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Two-channel 32-bit signed image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageRelativeError_32s_C2R_Ctx(const Npp32s * pSrc1, int nSrc1Step, 
                                               const Npp32s * pSrc2, int nSrc2Step, 
                                               NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Two-channel 32-bit signed complex image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageRelativeError_32sc_C2R_Ctx(const Npp32sc * pSrc1, int nSrc1Step, 
                                                const Npp32sc * pSrc2, int nSrc2Step, 
                                                NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Two-channel 32-bit floating point image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageRelativeError_32f_C2R_Ctx(const Npp32f * pSrc1, int nSrc1Step, 
                                               const Npp32f * pSrc2, int nSrc2Step, 
                                               NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Two-channel 32-bit floating point complex image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageRelativeError_32fc_C2R_Ctx(const Npp32fc * pSrc1, int nSrc1Step, 
                                                const Npp32fc * pSrc2, int nSrc2Step, 
                                                NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Two-channel 64-bit floating point image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageRelativeError_64f_C2R_Ctx(const Npp64f * pSrc1, int nSrc1Step, 
                                               const Npp64f * pSrc2, int nSrc2Step, 
                                               NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 8-bit unsigned image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageRelativeError_8u_C3R_Ctx(const Npp8u * pSrc1, int nSrc1Step, 
                                              const Npp8u * pSrc2, int nSrc2Step, 
                                              NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 8-bit signed image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageRelativeError_8s_C3R_Ctx(const Npp8s * pSrc1, int nSrc1Step, 
                                              const Npp8s * pSrc2, int nSrc2Step, 
                                              NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 16-bit unsigned image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageRelativeError_16u_C3R_Ctx(const Npp16u * pSrc1, int nSrc1Step, 
                                               const Npp16u * pSrc2, int nSrc2Step, 
                                               NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 16-bit signed image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageRelativeError_16s_C3R_Ctx(const Npp16s * pSrc1, int nSrc1Step, 
                                               const Npp16s * pSrc2, int nSrc2Step, 
                                               NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 16-bit signed complex image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageRelativeError_16sc_C3R_Ctx(const Npp16sc * pSrc1, int nSrc1Step, 
                                                const Npp16sc * pSrc2, int nSrc2Step, 
                                                NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 32-bit unsigned image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageRelativeError_32u_C3R_Ctx(const Npp32u * pSrc1, int nSrc1Step, 
                                               const Npp32u * pSrc2, int nSrc2Step, 
                                               NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 32-bit signed image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
*/
NppStatus nppiAverageRelativeError_32s_C3R_Ctx(const Npp32s * pSrc1, int nSrc1Step, 
                                               const Npp32s * pSrc2, int nSrc2Step, 
                                               NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 32-bit signed complex image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageRelativeError_32sc_C3R_Ctx(const Npp32sc * pSrc1, int nSrc1Step, 
                                                const Npp32sc * pSrc2, int nSrc2Step, 
                                                NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 32-bit floating point image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageRelativeError_32f_C3R_Ctx(const Npp32f * pSrc1, int nSrc1Step, 
                                               const Npp32f * pSrc2, int nSrc2Step, 
                                               NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);

/**
 * Three-channel 32-bit floating point complex image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageRelativeError_32fc_C3R_Ctx(const Npp32fc * pSrc1, int nSrc1Step, 
                                                const Npp32fc * pSrc2, int nSrc2Step, 
                                                NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 64-bit floating point image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageRelativeError_64f_C3R_Ctx(const Npp64f * pSrc1, int nSrc1Step, 
                                               const Npp64f * pSrc2, int nSrc2Step, 
                                               NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit unsigned image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageRelativeError_8u_C4R_Ctx(const Npp8u * pSrc1, int nSrc1Step, 
                                              const Npp8u * pSrc2, int nSrc2Step, 
                                              NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 8-bit signed image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageRelativeError_8s_C4R_Ctx(const Npp8s * pSrc1, int nSrc1Step, 
                                              const Npp8s * pSrc2, int nSrc2Step, 
                                              NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit unsigned image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageRelativeError_16u_C4R_Ctx(const Npp16u * pSrc1, int nSrc1Step, 
                                               const Npp16u * pSrc2, int nSrc2Step, 
                                               NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit signed image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageRelativeError_16s_C4R_Ctx(const Npp16s * pSrc1, int nSrc1Step, 
                                               const Npp16s * pSrc2, int nSrc2Step, 
                                               NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 16-bit signed complex image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageRelativeError_16sc_C4R_Ctx(const Npp16sc * pSrc1, int nSrc1Step, 
                                                const Npp16sc * pSrc2, int nSrc2Step, 
                                                NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit unsigned image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageRelativeError_32u_C4R_Ctx(const Npp32u * pSrc1, int nSrc1Step, 
                                               const Npp32u * pSrc2, int nSrc2Step, 
                                               NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit signed image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageRelativeError_32s_C4R_Ctx(const Npp32s * pSrc1, int nSrc1Step, 
                                               const Npp32s * pSrc2, int nSrc2Step, 
                                               NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit signed complex image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageRelativeError_32sc_C4R_Ctx(const Npp32sc * pSrc1, int nSrc1Step, 
                                                const Npp32sc * pSrc2, int nSrc2Step, 
                                                NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit floating point image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageRelativeError_32f_C4R_Ctx(const Npp32f * pSrc1, int nSrc1Step, 
                                               const Npp32f * pSrc2, int nSrc2Step, 
                                               NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 32-bit floating point complex image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageRelativeError_32fc_C4R_Ctx(const Npp32fc * pSrc1, int nSrc1Step, 
                                                const Npp32fc * pSrc2, int nSrc2Step, 
                                                NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Four-channel 64-bit floating point image MaximumRelative_Error.
 * For common parameter descriptions, see \ref CommonMaximumErrorParameters.
 */
NppStatus nppiAverageRelativeError_64f_C4R_Ctx(const Npp64f * pSrc1, int nSrc1Step, 
                                               const Npp64f * pSrc2, int nSrc2Step, 
                                               NppiSize oSizeROI, Npp64f * pError, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);

/** @} */

/** @name AverageRelativeErrorGetBufferHostSize
 * Companion primitives for computing the device buffer size (in bytes) required by the NormDiff_Inf primitives.
 * \section CommonAverageRelativeErrorGetBufferHostSizeParameters
 * Common parameters for nppiAverageRelativeErrorGetBufferHostSize functions include:
 * \param oSizeROI \ref roi_specification.
 * \param hpBufferSize Required buffer size. Important: hpBufferSize is a 
 *        <em>host pointer.</em> \ref general_scratch_buffer.
 * \param nppStreamCtx \ref application_managed_stream_context. 
 * \return NPP_NULL_POINTER_ERROR if hpBufferSize is 0 (NULL),  \ref roi_error_codes.  
 * @{
 */
/** 
 * Buffer size for \ref nppiAverageRelativeError_8u_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageRelativeErrorGetBufferHostSize_8u_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageRelativeError_8s_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageRelativeErrorGetBufferHostSize_8s_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageRelativeError_16u_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageRelativeErrorGetBufferHostSize_16u_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageRelativeError_16s_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageRelativeErrorGetBufferHostSize_16s_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageRelativeError_16sc_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageRelativeErrorGetBufferHostSize_16sc_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageRelativeError_32u_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageRelativeErrorGetBufferHostSize_32u_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageRelativeError_32s_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageRelativeErrorGetBufferHostSize_32s_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageRelativeError_32sc_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageRelativeErrorGetBufferHostSize_32sc_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageRelativeError_32f_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageRelativeErrorGetBufferHostSize_32f_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageRelativeError_32fc_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageRelativeErrorGetBufferHostSize_32fc_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageRelativeError_64f_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageRelativeErrorGetBufferHostSize_64f_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageRelativeError_8u_C2R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageRelativeErrorGetBufferHostSize_8u_C2R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageRelativeError_8s_C2R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageRelativeErrorGetBufferHostSize_8s_C2R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageRelativeError_16u_C2R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageRelativeErrorGetBufferHostSize_16u_C2R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageRelativeError_16s_C2R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageRelativeErrorGetBufferHostSize_16s_C2R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageRelativeError_16sc_C2R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageRelativeErrorGetBufferHostSize_16sc_C2R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageRelativeError_32u_C2R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageRelativeErrorGetBufferHostSize_32u_C2R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageRelativeError_32s_C2R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageRelativeErrorGetBufferHostSize_32s_C2R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageRelativeError_32sc_C2R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageRelativeErrorGetBufferHostSize_32sc_C2R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageRelativeError_32f_C2R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageRelativeErrorGetBufferHostSize_32f_C2R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageRelativeError_32fc_C2R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageRelativeErrorGetBufferHostSize_32fc_C2R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageRelativeError_64f_C2R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageRelativeErrorGetBufferHostSize_64f_C2R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageRelativeError_8u_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageRelativeErrorGetBufferHostSize_8u_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageRelativeError_8s_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageRelativeErrorGetBufferHostSize_8s_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageRelativeError_16u_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageRelativeErrorGetBufferHostSize_16u_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageRelativeError_16s_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageRelativeErrorGetBufferHostSize_16s_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageRelativeError_16sc_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageRelativeErrorGetBufferHostSize_16sc_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageRelativeError_32u_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageRelativeErrorGetBufferHostSize_32u_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageRelativeError_32s_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageRelativeErrorGetBufferHostSize_32s_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageRelativeError_32sc_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageRelativeErrorGetBufferHostSize_32sc_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageRelativeError_32f_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageRelativeErrorGetBufferHostSize_32f_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageRelativeError_32fc_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageRelativeErrorGetBufferHostSize_32fc_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageRelativeError_64f_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageRelativeErrorGetBufferHostSize_64f_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageRelativeError_8u_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageRelativeErrorGetBufferHostSize_8u_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageRelativeError_8s_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageRelativeErrorGetBufferHostSize_8s_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageRelativeError_16u_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageRelativeErrorGetBufferHostSize_16u_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageRelativeError_16s_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageRelativeErrorGetBufferHostSize_16s_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageRelativeError_16sc_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageRelativeErrorGetBufferHostSize_16sc_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageRelativeError_32u_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageRelativeErrorGetBufferHostSize_32u_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageRelativeError_32s_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageRelativeErrorGetBufferHostSize_32s_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageRelativeError_32sc_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageRelativeErrorGetBufferHostSize_32sc_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageRelativeError_32f_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageRelativeErrorGetBufferHostSize_32f_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageRelativeError_32fc_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageRelativeErrorGetBufferHostSize_32fc_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiAverageRelativeError_64f_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonAverageRelativeErrorGetBufferHostSizeParameters.
 */
NppStatus 
nppiAverageRelativeErrorGetBufferHostSize_64f_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);

/** @} */

/** @} AverageRelativeErrorGetBufferHostSize */

/** @} image_average_relative_error */

/** 
 * \section image_quality_assessment IQA
 * @defgroup image_quality_assessment IQA
 * Primitives for computing the image quality between two images, such as MSE, PSNR, SSIM, and MS-SSIM.
 * @{
 */

/** @name MSE
 * @{
 */

/**
 * One-channel 8-bit unsigned image MSE.
 *
 * \param pSrc1 \ref source_image_pointer.
 * \param nSrc1Step \ref source_image_line_step.
 * \param pSrc2 \ref source_image_pointer.
 * \param nSrc2Step \ref source_image_line_step.
 * \param oSizeROI \ref roi_specification.
 * \param pMSE Device memory pointer to the computed MSE of two images. 
 * \param pDeviceBuffer Pointer to the required device memory allocation, \ref general_scratch_buffer. 
 * \param nppStreamCtx \ref application_managed_stream_context. 
 * \return \ref image_data_error_codes, \ref roi_error_codes.
 */
NppStatus nppiMSE_8u_C1R_Ctx(const Npp8u * pSrc1, int nSrc1Step, 
                             const Npp8u * pSrc2, int nSrc2Step, 
                             NppiSize oSizeROI, Npp32f * pMSE, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 8-bit unsigned image MSE.
 *
 * \param pSrc1 \ref source_image_pointer.
 * \param nSrc1Step \ref source_image_line_step.
 * \param pSrc2 \ref source_image_pointer.
 * \param nSrc2Step \ref source_image_line_step.
 * \param oSizeROI \ref roi_specification.
 * \param pMSE Device memory pointer to the computed MSE of two images. 
 * \param pDeviceBuffer Pointer to the required device memory allocation, \ref general_scratch_buffer. 
 * \param nppStreamCtx \ref application_managed_stream_context. 
 * \return \ref image_data_error_codes, \ref roi_error_codes.
 */
NppStatus nppiMSE_8u_C3R_Ctx(const Npp8u * pSrc1, int nSrc1Step, 
                             const Npp8u * pSrc2, int nSrc2Step, 
                             NppiSize oSizeROI, Npp32f * pMSE, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);

/** @} MSE */

/** @name PSNR
 * @{
 */

/**
 * One-channel 8-bit unsigned image PSNR_Ctx.
 *
 * \param pSrc1 \ref source_image_pointer.
 * \param nSrc1Step \ref source_image_line_step.
 * \param pSrc2 \ref source_image_pointer.
 * \param nSrc2Step \ref source_image_line_step.
 * \param oSizeROI \ref roi_specification.
 * \param pPSNR Device memory pointer to the computed PSNR of two images. 
 * \param pDeviceBuffer Pointer to the required device memory allocation, \ref general_scratch_buffer. 
 * \param nppStreamCtx \ref application_managed_stream_context. 
 * \return \ref image_data_error_codes, \ref roi_error_codes.
 */
NppStatus nppiPSNR_8u_C1R_Ctx(const Npp8u * pSrc1, int nSrc1Step, 
                              const Npp8u * pSrc2, int nSrc2Step, 
                              NppiSize oSizeROI, Npp32f * pPSNR, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 8-bit unsigned image PSNR_Ctx.
 *
 * \param pSrc1 \ref source_image_pointer.
 * \param nSrc1Step \ref source_image_line_step.
 * \param pSrc2 \ref source_image_pointer.
 * \param nSrc2Step \ref source_image_line_step.
 * \param oSizeROI \ref roi_specification.
 * \param pPSNR Device memory pointer to the computed PSNR of two images. 
 * \param pDeviceBuffer Pointer to the required device memory allocation, \ref general_scratch_buffer. 
 * \param nppStreamCtx \ref application_managed_stream_context. 
 * \return \ref image_data_error_codes, \ref roi_error_codes.
 */
NppStatus nppiPSNR_8u_C3R_Ctx(const Npp8u * pSrc1, int nSrc1Step, 
                              const Npp8u * pSrc2, int nSrc2Step, 
                              NppiSize oSizeROI, Npp32f * pPSNR, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);

/** @} PSNR */

/** @name SSIM
 * @{
 */

/**
 * One-channel 8-bit unsigned image SSIM.
 *
 * \param pSrc1 \ref source_image_pointer.
 * \param nSrc1Step \ref source_image_line_step.
 * \param pSrc2 \ref source_image_pointer.
 * \param nSrc2Step \ref source_image_line_step.
 * \param oSizeROI \ref roi_specification.
 * \param pSSIM Device memory pointer to the computed SSIM of two images. 
 * \param pDeviceBuffer Pointer to the required device memory allocation, \ref general_scratch_buffer. 
 * \param nppStreamCtx \ref application_managed_stream_context. 
 * \return \ref image_data_error_codes, \ref roi_error_codes.
 */
NppStatus nppiSSIM_8u_C1R_Ctx(const Npp8u * pSrc1, int nSrc1Step, 
                              const Npp8u * pSrc2, int nSrc2Step, 
                              NppiSize oSizeROI, Npp32f * pSSIM, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 8-bit unsigned image SSIM.
 *
 * \param pSrc1 \ref source_image_pointer.
 * \param nSrc1Step \ref source_image_line_step.
 * \param pSrc2 \ref source_image_pointer.
 * \param nSrc2Step \ref source_image_line_step.
 * \param oSizeROI \ref roi_specification.
 * \param pSSIM Device memory pointer to the computed SSIM of two images. 
 * \param pDeviceBuffer Pointer to the required device memory allocation, \ref general_scratch_buffer. 
 * \param nppStreamCtx \ref application_managed_stream_context. 
 * \return \ref image_data_error_codes, \ref roi_error_codes.
 */
NppStatus nppiSSIM_8u_C3R_Ctx(const Npp8u * pSrc1, int nSrc1Step, 
                              const Npp8u * pSrc2, int nSrc2Step, 
                              NppiSize oSizeROI, Npp32f * pSSIM, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);

/** @} SSIM */

/** @name MSSSIM
 * @{
 */

/**
 * One-channel 8-bit unsigned image MS-SSIM*.
 * This function will be deprecated in a future release use the nppiWMSSSIM functions instead.
 *
 * \param pSrc1 \ref source_image_pointer.
 * \param nSrc1Step \ref source_image_line_step.
 * \param pSrc2 \ref source_image_pointer.
 * \param nSrc2Step \ref source_image_line_step.
 * \param oSizeROI \ref roi_specification.
 * \param pMSSSIM Device memory pointer to the computed MS-SSIM of two images. 
 * \param pDeviceBuffer Pointer to the required device memory allocation, \ref general_scratch_buffer. 
 * \param nppStreamCtx \ref application_managed_stream_context. 
 * \return \ref image_data_error_codes, \ref roi_error_codes.
 */
NppStatus nppiMSSSIM_8u_C1R_Ctx(const Npp8u * pSrc1, int nSrc1Step, 
                                const Npp8u * pSrc2, int nSrc2Step, 
                                NppiSize oSizeROI, Npp32f * pMSSSIM, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);

/** @} MSSSIM */

/** @name WMSSSIM
 * @{
 */

/**
 * One-channel 8-bit unsigned image MS-SSIM*.
 * This function uses the algorithm described in the paper by Wang et. al.
 * Wang, Z., Simoncelli, E.P., Bovik, A.C. Multiscale Structural Similarity for Image Quality Assessment. In: The
 * Thirty-Seventh Asilomar Conference on Signals, Systems & Computers, 2003, 13981402. Pacific Grove, CA, USA: IEEE,
 * 2003. https://doi.org/10.1109/ACSSC.2003.1292216.
 * NOTE: this API call can only process oSizeROI dimensions 16px by 16px and above. Any oSizeROI dimensions less than 16px by 16px
 * will result in undefined behaviour.
 *
 * \param pSrc1 \ref source_image_pointer.
 * \param nSrc1Step \ref source_image_line_step.
 * \param pSrc2 \ref source_image_pointer.
 * \param nSrc2Step \ref source_image_line_step.
 * \param oSizeROI \ref roi_specification.
 * \param pMSSSIM Device memory pointer to the computed MS-SSIM of two images. 
 * \param pDeviceBuffer Pointer to the required device memory allocation, \ref general_scratch_buffer. 
 * \param nppStreamCtx \ref application_managed_stream_context. 
 * \return \ref image_data_error_codes, \ref roi_error_codes.
 */
NppStatus nppiWMSSSIM_8u_C1R_Ctx(const Npp8u * pSrc1, int nSrc1Step, 
                                const Npp8u * pSrc2, int nSrc2Step, 
                                NppiSize oSizeROI, Npp32f * pMSSSIM, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);
/**
 * Three-channel 8-bit unsigned image MS-SSIM*.
 * This function uses the algorithm described in the paper by Wang et. al.
 * Wang, Z., Simoncelli, E.P., Bovik, A.C. Multiscale Structural Similarity for Image Quality Assessment. In: The
 * Thirty-Seventh Asilomar Conference on Signals, Systems & Computers, 2003, 13981402. Pacific Grove, CA, USA: IEEE,
 * 2003. https://doi.org/10.1109/ACSSC.2003.1292216.
 * NOTE: this API call can only process oSizeROI dimensions 16px by 16px and above. Any oSizeROI dimensions less than 16px by 16px
 * will result in undefined behaviour.
 *
 * \param pSrc1 \ref source_image_pointer.
 * \param nSrc1Step \ref source_image_line_step.
 * \param pSrc2 \ref source_image_pointer.
 * \param nSrc2Step \ref source_image_line_step.
 * \param oSizeROI \ref roi_specification.
 * \param pMSSSIM Device memory pointer to the computed MS-SSIM of two images. 
 * \param pDeviceBuffer Pointer to the required device memory allocation, \ref general_scratch_buffer. 
 * \param nppStreamCtx \ref application_managed_stream_context. 
 * \return \ref image_data_error_codes, \ref roi_error_codes.
 */
NppStatus nppiWMSSSIM_8u_C3R_Ctx(const Npp8u * pSrc1, int nSrc1Step, 
                                const Npp8u * pSrc2, int nSrc2Step, 
                                NppiSize oSizeROI, Npp32f * pMSSSIM, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);

/**
 * Four-channel 8-bit unsigned image MS-SSIM*.
 * This function uses the algorithm described in the paper by Wang et. al.
 * Wang, Z., Simoncelli, E.P., Bovik, A.C. Multiscale Structural Similarity for Image Quality Assessment. In: The
 * Thirty-Seventh Asilomar Conference on Signals, Systems & Computers, 2003, 13981402. Pacific Grove, CA, USA: IEEE,
 * 2003. https://doi.org/10.1109/ACSSC.2003.1292216.
 * NOTE: this API call can only process oSizeROI dimensions 16px by 16px and above. Any oSizeROI dimensions less than 16px by 16px
 * will result in undefined behaviour.
 *
 * \param pSrc1 \ref source_image_pointer.
 * \param nSrc1Step \ref source_image_line_step.
 * \param pSrc2 \ref source_image_pointer.
 * \param nSrc2Step \ref source_image_line_step.
 * \param oSizeROI \ref roi_specification.
 * \param pMSSSIM Device memory pointer to the computed MS-SSIM of two images. 
 * \param pDeviceBuffer Pointer to the required device memory allocation, \ref general_scratch_buffer. 
 * \param nppStreamCtx \ref application_managed_stream_context. 
 * \return \ref image_data_error_codes, \ref roi_error_codes.
 */
NppStatus nppiWMSSSIM_8u_C4R_Ctx(const Npp8u * pSrc1, int nSrc1Step, 
                                const Npp8u * pSrc2, int nSrc2Step, 
                                NppiSize oSizeROI, Npp32f * pMSSSIM, Npp8u * pDeviceBuffer, NppStreamContext nppStreamCtx);

/** @} WMSSSIM */

/** @name MSEGetBufferHostSize
 * @{
 */

/** 
 * Buffer size for \ref nppiMSE_8u_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonGetBufferHostSizeParameters.
 */
NppStatus 
nppiMSEGetBufferHostSize_8u_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMSE_8u_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonGetBufferHostSizeParameters.
 */
NppStatus 
nppiMSEGetBufferHostSize_8u_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiPSNR_8u_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonGetBufferHostSizeParameters.
 */
NppStatus 
nppiPSNRGetBufferHostSize_8u_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiPSNR_8u_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonGetBufferHostSizeParameters.
 */
NppStatus 
nppiPSNRGetBufferHostSize_8u_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiSSIM_8u_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonGetBufferHostSizeParameters.
 */
NppStatus 
nppiSSIMGetBufferHostSize_8u_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiSSIM_8u_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonGetBufferHostSizeParameters.
 */
NppStatus 
nppiSSIMGetBufferHostSize_8u_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiMSSSIM_8u_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonGetBufferHostSizeParameters.
 */
NppStatus 
nppiMSSSIMGetBufferHostSize_8u_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiWMSSSIM_8u_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonGetBufferHostSizeParameters.
 */
NppStatus 
nppiWMSSSIMGetBufferHostSize_8u_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiWMSSSIM_8u_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonGetBufferHostSizeParameters.
 */
NppStatus 
nppiWMSSSIMGetBufferHostSize_8u_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for \ref nppiWMSSSIM_8u_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonGetBufferHostSizeParameters.
 */
NppStatus 
nppiWMSSSIMGetBufferHostSize_8u_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);

/** @} MSEGetBufferHostSize */

/** @} image_quality_assessment */

/** 
 * \section batch_image_quality_assessment IQABatch
 * @defgroup batch_image_quality_assessment IQABatch
 * Primitives for computing the image quality for a batch of image pairs, such as MSE, PSNR, SSIM, and MS-SSIM with a single \ref
 * roi_specification for all pairs of input images
 * @{
 */

/** @name MSEBatch
 * @{
 */

/**
 * 1 channel 8-bit unsigned MSE for a batch of image pairs for a single ROI.
 * Provided oSizeROI will be used for all images passed in pSrc1BatchList and pSrc2BatchList
 * arguments. API user must ensure that provided ROI (oSizeROI) does not go beyond the borders of any of provided images.
 *
 * \param pSrc1BatchList \ref source_batch_images_pointer device memory pointer to the list of device memory image descriptors, per image oSize must be initialized.
 * \param pSrc2BatchList \ref source_batch_images_pointer device memory pointer to the list of device memory image descriptors, per image oSize must be initialized.
 * \param nBatchSize Number of \ref NppiImageDescriptor, NppiBufferDescriptor, and new max number structures/values processed in this call (must be > 1).
 * \param oSizeROI \ref roi_specification ROI width and height of ALL images in the batch, MUST match the ROI used when the label markers UF image was generated.
 * \param pMSE Device memory pointer to output array of the computed MSE for nBatchSize * sizeof(Npp32f) * 1 image pairs. 
 * \param pDeviceBufferList Device memory pointer to the list of NppiBufferDescriptor buffer descriptors specifying per image device
 * memory buffer pointers and size as returned by at least one nppiMSEBatchGetBufferHostSize call.
 * \param nppStreamCtx \ref application_managed_stream_context.
 * \return \ref image_data_error_codes, \ref roi_error_codes
 */
NppStatus nppiMSEBatch_8u_C1R_Ctx(const NppiImageDescriptor * pSrc1BatchList, 
                                  const NppiImageDescriptor * pSrc2BatchList, 
                                  int nBatchSize, NppiSize oSizeROI, Npp32f * pMSE, 
                                  NppiBufferDescriptor * pDeviceBufferList, NppStreamContext nppStreamCtx);

/**
 * 3 channel 8-bit unsigned MSE for a batch of image pairs for a single ROI.
 * Provided oSizeROI will be used for all images passed in pSrc1BatchList and pSrc2BatchList
 * arguments. API user must ensure that provided ROI (oSizeROI) does not go beyond the borders of any of provided images.
 *
 * \param pSrc1BatchList \ref source_batch_images_pointer device memory pointer to the list of device memory image descriptors, per image oSize must be initialized.
 * \param pSrc2BatchList \ref source_batch_images_pointer device memory pointer to the list of device memory image descriptors, per image oSize must be initialized.
 * \param nBatchSize Number of \ref NppiImageDescriptor, NppiBufferDescriptor, and new max number structures/values processed in this call (must be > 1).
 * \param oSizeROI \ref roi_specification ROI width and height of ALL images in the batch, MUST match the ROI used when the label markers UF image was generated.
 * \param pMSE Device memory pointer to output array of the computed MSE for nBatchSize * sizeof(Npp32f) * 3 image pairs. 
 * \param pDeviceBufferList Device memory pointer to the list of NppiBufferDescriptor buffer descriptors specifying per image device
 * memory buffer pointers and size as returned by at least one nppiMSEBatchGetBufferHostSize call.
 * \param nppStreamCtx \ref application_managed_stream_context.
 * \return \ref image_data_error_codes, \ref roi_error_codes
 */
NppStatus nppiMSEBatch_8u_C3R_Ctx(const NppiImageDescriptor * pSrc1BatchList, 
                                  const NppiImageDescriptor * pSrc2BatchList, 
                                  int nBatchSize, NppiSize oSizeROI, Npp32f * pMSE, 
                                  NppiBufferDescriptor * pDeviceBufferList, NppStreamContext nppStreamCtx);

/** @} MSEBatch */

/** @name PSNRBatch
 * @{
 */

/**
 * 1 channel 8-bit unsigned PSNR for a batch of image pairs for a single ROI.
 * Provided oSizeROI will be used for all images passed in pSrc1BatchList and pSrc2BatchList
 * arguments. API user must ensure that provided ROI (oSizeROI) does not go beyond the borders of any of provided images.
 *
 * \param pSrc1BatchList \ref source_batch_images_pointer device memory pointer to the list of device memory image descriptors, per image oSize must be initialized.
 * \param pSrc2BatchList \ref source_batch_images_pointer device memory pointer to the list of device memory image descriptors, per image oSize must be initialized.
 * \param nBatchSize Number of \ref NppiImageDescriptor, NppiBufferDescriptor, and new max number structures/values processed in this call (must be > 1).
 * \param oSizeROI \ref roi_specification ROI width and height of ALL images in the batch, MUST match the ROI used when the label markers UF image was generated.
 * \param pPSNR Device memory pointer to output array of the computed PSNR for nBatchSize * sizeof(Npp32f) * 1 image pairs. 
 * \param pDeviceBufferList Device memory pointer to the list of NppiBufferDescriptor buffer descriptors specifying per image device
 * memory buffer pointers and size as returned by at least one nppiPSNRBatchGetBufferHostSize call.
 * \param nppStreamCtx \ref application_managed_stream_context.
 * \return \ref image_data_error_codes, \ref roi_error_codes
 */
NppStatus nppiPSNRBatch_8u_C1R_Ctx(const NppiImageDescriptor * pSrc1BatchList, 
                                   const NppiImageDescriptor * pSrc2BatchList, 
                                   int nBatchSize, NppiSize oSizeROI, Npp32f * pPSNR, 
                                   NppiBufferDescriptor * pDeviceBufferList, NppStreamContext nppStreamCtx);

/**
 * 3 channel 8-bit unsigned PSNR for a batch of image pairs for a single ROI.
 * Provided oSizeROI will be used for all images passed in pSrc1BatchList and pSrc2BatchList
 * arguments. API user must ensure that provided ROI (oSizeROI) does not go beyond the borders of any of provided images.
 *
 * \param pSrc1BatchList \ref source_batch_images_pointer device memory pointer to the list of device memory image descriptors, per image oSize must be initialized.
 * \param pSrc2BatchList \ref source_batch_images_pointer device memory pointer to the list of device memory image descriptors, per image oSize must be initialized.
 * \param nBatchSize Number of \ref NppiImageDescriptor, NppiBufferDescriptor, and new max number structures/values processed in this call (must be > 1).
 * \param oSizeROI \ref roi_specification ROI width and height of ALL images in the batch, MUST match the ROI used when the label markers UF image was generated.
 * \param pPSNR Device memory pointer to output array of the computed PSNR for nBatchSize * sizeof(Npp32f) * 3 image pairs. 
 * \param pDeviceBufferList Device memory pointer to the list of NppiBufferDescriptor buffer descriptors specifying per image device
 * memory buffer pointers and size as returned by at least one nppiPSNRBatchGetBufferHostSize call.
 * \param nppStreamCtx \ref application_managed_stream_context.
 * \return \ref image_data_error_codes, \ref roi_error_codes
 */
NppStatus nppiPSNRBatch_8u_C3R_Ctx(const NppiImageDescriptor * pSrc1BatchList, 
                                   const NppiImageDescriptor * pSrc2BatchList, 
                                   int nBatchSize, NppiSize oSizeROI, Npp32f * pPSNR, 
                                   NppiBufferDescriptor * pDeviceBufferList, NppStreamContext nppStreamCtx);

/** @} PSNRBatch */

/** @name SSIMBatch
 * @{
 */

/**
 * 1 channel 8-bit unsigned SSIM for a batch of image pairs for a single ROI.
 * Provided oSizeROI will be used for all images passed in pSrc1BatchList and pSrc2BatchList
 * arguments. API user must ensure that provided ROI (oSizeROI) does not go beyond the borders of any of provided images.
 *
 * \param pSrc1BatchList \ref source_batch_images_pointer device memory pointer to the list of device memory image descriptors, per image oSize must be initialized.
 * \param pSrc2BatchList \ref source_batch_images_pointer device memory pointer to the list of device memory image descriptors, per image oSize must be initialized.
 * \param nBatchSize Number of \ref NppiImageDescriptor, NppiBufferDescriptor, and new max number structures/values processed in this call (must be > 1).
 * \param oSizeROI \ref roi_specification ROI width and height of ALL images in the batch, MUST match the ROI used when the label markers UF image was generated.
 * \param pSSIM Device memory pointer to output array of the computed SSIM for nBatchSize * sizeof(Npp32f) * 1 image pairs. 
 * \param pDeviceBufferList Device memory pointer to the list of NppiBufferDescriptor buffer descriptors specifying per image device
 * memory buffer pointers and size as returned by at least one nppiSSIMBatchGetBufferHostSize call.
 * \param nppStreamCtx \ref application_managed_stream_context.
 * \return \ref image_data_error_codes, \ref roi_error_codes
 */
NppStatus nppiSSIMBatch_8u_C1R_Ctx(const NppiImageDescriptor * pSrc1BatchList, 
                                   const NppiImageDescriptor * pSrc2BatchList, 
                                   int nBatchSize, NppiSize oSizeROI, Npp32f * pSSIM, 
                                   NppiBufferDescriptor * pDeviceBufferList, NppStreamContext nppStreamCtx);
/**
 * 3 channel 8-bit unsigned SSIM for a batch of image pairs for a single ROI.
 * Provided oSizeROI will be used for all images passed in pSrc1BatchList and pSrc2BatchList
 * arguments. API user must ensure that provided ROI (oSizeROI) does not go beyond the borders of any of provided images.
 *
 * \param pSrc1BatchList \ref source_batch_images_pointer device memory pointer to the list of device memory image descriptors, per image oSize must be initialized.
 * \param pSrc2BatchList \ref source_batch_images_pointer device memory pointer to the list of device memory image descriptors, per image oSize must be initialized.
 * \param nBatchSize Number of \ref NppiImageDescriptor, NppiBufferDescriptor, and new max number structures/values processed in this call (must be > 1).
 * \param oSizeROI \ref roi_specification ROI width and height of ALL images in the batch, MUST match the ROI used when the label markers UF image was generated.
 * \param pSSIM Device memory pointer to output array of the computed SSIM for nBatchSize * sizeof(Npp32f) * 3 image pairs. 
 * \param pDeviceBufferList Device memory pointer to the list of NppiBufferDescriptor buffer descriptors specifying per image device
 * memory buffer pointers and size as returned by at least one nppiSSIMBatchGetBufferHostSize call.
 * \param nppStreamCtx \ref application_managed_stream_context.
 * \return \ref image_data_error_codes, \ref roi_error_codes
 */
NppStatus nppiSSIMBatch_8u_C3R_Ctx(const NppiImageDescriptor * pSrc1BatchList, 
                                   const NppiImageDescriptor * pSrc2BatchList, 
                                   int nBatchSize, NppiSize oSizeROI, Npp32f * pSSIM, 
                                   NppiBufferDescriptor * pDeviceBufferList, NppStreamContext nppStreamCtx);

/** @} SSIMBatch */

/** @name WMSSSIMBatch
 * @{
 */

/**
 * 1 channel 8-bit unsigned WMSSSIM for a batch of image pairs for a single ROI.
 * Provided oSizeROI will be used for all images passed in pSrc1BatchList and pSrc2BatchList
 * arguments. API user must ensure that provided ROI (oSizeROI) does not go beyond the borders of any of provided images.
 * NOTE: this API call can only process oSizeROI dimensions 16 pixels by 16 pixels and above. Any oSizeROI dimensions less than 16 pixels by 16 pixels
 * will result in undefined behaviour.
 *
 * \param pSrc1BatchList \ref source_batch_images_pointer device memory pointer to the list of device memory image descriptors, per image oSize must be initialized.
 * \param pSrc2BatchList \ref source_batch_images_pointer device memory pointer to the list of device memory image descriptors, per image oSize must be initialized.
 * \param nBatchSize Number of \ref NppiImageDescriptor, NppiBufferDescriptor, and new max number structures/values processed in this call (must be > 1).
 * \param oSizeROI \ref roi_specification ROI width and height of ALL images in the batch, MUST match the ROI used when the label markers UF image was generated.
 * \param pWMSSSIM Device memory pointer to output array of the computed WMSSSIM for nBatchSize * sizeof(Npp32f) * 1 image pairs. 
 * \param pDeviceBufferList Device memory pointer to the list of NppiBufferDescriptor buffer descriptors specifying per image device
 * memory buffer pointers and size as returned by at least one nppiWMSSSIMBatchGetBufferHostSize call.
 * \param nppStreamCtx \ref application_managed_stream_context.
 * \return \ref image_data_error_codes, \ref roi_error_codes
 */
NppStatus nppiWMSSSIMBatch_8u_C1R_Ctx(const NppiImageDescriptor * pSrc1BatchList, 
                                      const NppiImageDescriptor * pSrc2BatchList, 
                                      int nBatchSize, NppiSize oSizeROI, Npp32f * pWMSSSIM, 
                                      NppiBufferDescriptor * pDeviceBufferList, NppStreamContext nppStreamCtx);
/**
 * 3 channel 8-bit unsigned WMSSSIM for a batch of image pairs for a single ROI.
 * Provided oSizeROI will be used for all images passed in pSrc1BatchList and pSrc2BatchList
 * arguments. API user must ensure that provided ROI (oSizeROI) does not go beyond the borders of any of provided images.
 * NOTE: this API call can only process ROI dimensions 16 pixels by 16 pixels and above. Any ROI dimensions less than 16 pixels by 16 pixels
 * will result in undefined behaviour.
 *
 * \param pSrc1BatchList \ref source_batch_images_pointer device memory pointer to the list of device memory image descriptors, per image oSize must be initialized.
 * \param pSrc2BatchList \ref source_batch_images_pointer device memory pointer to the list of device memory image descriptors, per image oSize must be initialized.
 * \param nBatchSize Number of \ref NppiImageDescriptor, NppiBufferDescriptor, and new max number structures/values processed in this call (must be > 1).
 * \param oSizeROI \ref roi_specification ROI width and height of ALL images in the batch, MUST match the ROI used when the label markers UF image was generated.
 * \param pWMSSSIM Device memory pointer to output array of the computed WMSSSIM for nBatchSize * sizeof(Npp32f) * 3 image pairs. 
 * \param pDeviceBufferList Device memory pointer to the list of NppiBufferDescriptor buffer descriptors specifying per image device
 * memory buffer pointers and size as returned by at least one nppiWMSSSIMBatchGetBufferHostSize call.
 * \param nppStreamCtx \ref application_managed_stream_context.
 * \return \ref image_data_error_codes, \ref roi_error_codes
 */
NppStatus nppiWMSSSIMBatch_8u_C3R_Ctx(const NppiImageDescriptor * pSrc1BatchList, 
                                      const NppiImageDescriptor * pSrc2BatchList, 
                                      int nBatchSize, NppiSize oSizeROI, Npp32f * pWMSSSIM, 
                                      NppiBufferDescriptor * pDeviceBufferList, NppStreamContext nppStreamCtx);

/**
 * 4 channel 8-bit unsigned WMSSSIM for a batch of image pairs for a single ROI.
 * Provided oSizeROI will be used for all images passed in pSrc1BatchList and pSrc2BatchList
 * arguments. API user must ensure that provided ROI (oSizeROI) does not go beyond the borders of any of provided images.
 * NOTE: this API call can only process ROI dimensions 16 pixels by 16 pixels and above. Any ROI dimensions less than 16 pixels by 16 pixels
 * will result in undefined behaviour.
 *
 * \param pSrc1BatchList \ref source_batch_images_pointer device memory pointer to the list of device memory image descriptors, per image oSize must be initialized.
 * \param pSrc2BatchList \ref source_batch_images_pointer device memory pointer to the list of device memory image descriptors, per image oSize must be initialized.
 * \param nBatchSize Number of \ref NppiImageDescriptor, NppiBufferDescriptor, and new max number structures/values processed in this call (must be > 1).
 * \param oSizeROI \ref roi_specification ROI width and height of ALL images in the batch, MUST match the ROI used when the label markers UF image was generated.
 * \param pWMSSSIM Device memory pointer to output array of the computed WMSSSIM for nBatchSize * sizeof(Npp32f) * 4 image pairs. 
 * \param pDeviceBufferList Device memory pointer to the list of NppiBufferDescriptor buffer descriptors specifying per image device
 * memory buffer pointers and size as returned by at least one nppiWMSSSIMBatchGetBufferHostSize call.
 * \param nppStreamCtx \ref application_managed_stream_context.
 * \return \ref image_data_error_codes, \ref roi_error_codes
 */
NppStatus nppiWMSSSIMBatch_8u_C4R_Ctx(const NppiImageDescriptor * pSrc1BatchList, 
                                      const NppiImageDescriptor * pSrc2BatchList, 
                                      int nBatchSize, NppiSize oSizeROI, Npp32f * pWMSSSIM, 
                                      NppiBufferDescriptor * pDeviceBufferList, NppStreamContext nppStreamCtx);

/** @} WMSSSIMBatch */

/** @name MSEBatchGetBufferHostSize
 * @{
 */

/** 
 * Buffer size for a single image pair in the batch of image pairs for \ref nppiMSEBatch_8u_C1R_Ctx
 * For common parameter descriptions, see \ref CommonGetBufferHostSizeParameters.
 */
NppStatus 
nppiMSEBatchGetBufferHostSize_8u_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for a single image pair in the batch of image pairs for \ref nppiMSEBatch_8u_C3R_Ctx
 * For common parameter descriptions, see \ref CommonGetBufferHostSizeParameters.
 */
NppStatus 
nppiMSEBatchGetBufferHostSize_8u_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);

/** 
 * Buffer size for a single image pair in the batch of image pairs for \ref nppiPSNRBatch_8u_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonGetBufferHostSizeParameters.
 */
NppStatus 
nppiPSNRBatchGetBufferHostSize_8u_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for a single image pair in the batch of image pairs for \ref nppiPSNRBatch_8u_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonGetBufferHostSizeParameters.
 */
NppStatus 
nppiPSNRBatchGetBufferHostSize_8u_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);

/** 
 * Buffer size for a single image pair in the batch of image pairs for \ref nppiSSIMBatch_8u_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonGetBufferHostSizeParameters.
 */
NppStatus 
nppiSSIMBatchGetBufferHostSize_8u_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for a single image pair in the batch of image pairs for \ref nppiSSIMBatch_8u_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonGetBufferHostSizeParameters.
 */
NppStatus 
nppiSSIMBatchGetBufferHostSize_8u_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);

/** 
 * Buffer size for a single image pair in the batch of image pairs for \ref nppiWMSSSIMBatch_8u_C1R_Ctx.
 * For common parameter descriptions, see \ref CommonGetBufferHostSizeParameters.
 */
NppStatus 
nppiWMSSSIMBatchGetBufferHostSize_8u_C1R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for a single image pair in the batch of image pairs for \ref nppiWMSSSIMBatch_8u_C3R_Ctx.
 * For common parameter descriptions, see \ref CommonGetBufferHostSizeParameters.
 */
NppStatus 
nppiWMSSSIMBatchGetBufferHostSize_8u_C3R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);
/** 
 * Buffer size for a single image pair in the batch of image pairs for \ref nppiWMSSSIMBatch_8u_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonGetBufferHostSizeParameters.
 */
NppStatus 
nppiWMSSSIMBatchGetBufferHostSize_8u_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);

/** 
 * Buffer size for a single image pair in the batch of image pairs for \ref nppiWMSSSIMBatch_8u_C4R_Ctx.
 * For common parameter descriptions, see \ref CommonGetBufferHostSizeParameters.
 */
NppStatus 
nppiWMSSSIMBatchGetBufferHostSize_8u_C4R_Ctx(NppiSize oSizeROI, size_t * hpBufferSize/* host pointer */, NppStreamContext nppStreamCtx);

/** @} MSEBatchGetBufferHostSize */

/** @} batch_image_quality_assessment */


/**
 * \section batch_image_quality_assessment_advanced IQABatchAdvanced
 *  @defgroup batch_image_quality_assessment_advanced IQABatchAdvanced
 * Primitives for computing the image quality for a batch of image pairs, such as MSE, PSNR, SSIM, and MS-SSIM with per-image \ref
 * roi_specification
 * @{
 */
/** @name MSEBatchAdvanced
 * @{
 */

/**
 * 1 channel 8-bit unsigned MSE for a batch of image pairs with per-image ROI
 *
 * \param pSrc1BatchList \ref source_batch_images_pointer device memory pointer to the list of device memory image descriptors.
 * \param pSrc2BatchList \ref source_batch_images_pointer device memory pointer to the list of device memory image descriptors.
 * \param nBatchSize Number of \ref NppiImageDescriptor, NppiBufferDescriptor, and new max number structures/values processed in this call (must be > 1).
 * \param oMaxSizeROI \ref roi_specification maximum ROI width and height of ALL images in the batch.
 * \param pMSE Device memory pointer to output array of the computed MSE for nBatchSize * sizeof(Npp32f * 1 image pairs. 
 * \param pDeviceBufferList Device memory pointer to the list of NppiBufferDescriptor buffer descriptors specifying per image device
 * memory buffer pointers and size as returned by at least one nppiMSEBatchGetBufferHostSize call.
 * \param nppStreamCtx \ref application_managed_stream_context.
 * \return \ref image_data_error_codes, \ref roi_error_codes
 */
NppStatus nppiMSEBatch_8u_C1R_Advanced_Ctx(const NppiImageDescriptor * pSrc1BatchList, 
                                           const NppiImageDescriptor * pSrc2BatchList, 
                                           int nBatchSize, NppiSize oMaxSizeROI, Npp32f * pMSE, 
                                           NppiBufferDescriptor * pDeviceBufferList, NppStreamContext nppStreamCtx);

/**
 * 3 channel 8-bit unsigned MSE for a batch of image pairs with per-image ROI
 *
 * \param pSrc1BatchList \ref source_batch_images_pointer device memory pointer to the list of device memory image descriptors.
 * \param pSrc2BatchList \ref source_batch_images_pointer device memory pointer to the list of device memory image descriptors.
 * \param nBatchSize Number of \ref NppiImageDescriptor, NppiBufferDescriptor, and new max number structures/values processed in this call (must be > 1).
 * \param oMaxSizeROI \ref roi_specification maximum ROI width and height of ALL images in the batch.
 * \param pMSE Device memory pointer to output array of the computed MSE for nBatchSize * sizeof(Npp32f) * 3 image pairs. 
 * \param pDeviceBufferList Device memory pointer to the list of NppiBufferDescriptor buffer descriptors specifying per image device
 * memory buffer pointers and size as returned by at least one nppiMSEBatchGetBufferHostSize call.
 * \param nppStreamCtx \ref application_managed_stream_context.
 * \return \ref image_data_error_codes, \ref roi_error_codes
 */
NppStatus nppiMSEBatch_8u_C3R_Advanced_Ctx(const NppiImageDescriptor * pSrc1BatchList, 
                                           const NppiImageDescriptor * pSrc2BatchList, 
                                           int nBatchSize, NppiSize oMaxSizeROI, Npp32f * pMSE, 
                                           NppiBufferDescriptor * pDeviceBufferList, NppStreamContext nppStreamCtx);

/** @} MSEBatchAdvanced */

/** @name PSNRBatchAdvanced
 * @{
 */

/**
 * 1 channel 8-bit unsigned PSNR for a batch of image pairs with per-image ROI
 *
 * \param pSrc1BatchList \ref source_batch_images_pointer device memory pointer to the list of device memory image descriptors.
 * \param pSrc2BatchList \ref source_batch_images_pointer device memory pointer to the list of device memory image descriptors.
 * \param nBatchSize Number of \ref NppiImageDescriptor, NppiBufferDescriptor, and new max number structures/values processed in this call (must be > 1).
 * \param oMaxSizeROI \ref roi_specification maximum ROI width and height of ALL images in the batch.
 * \param pPSNR Device memory pointer to output array of the computed PSNR for nBatchSize * sizeof(Npp32f) * 1 image pairs. 
 * \param pDeviceBufferList Device memory pointer to the list of NppiBufferDescriptor buffer descriptors specifying per image device
 * memory buffer pointers and size as returned by at least one nppiPSNRBatchGetBufferHostSize call.
 * \param nppStreamCtx \ref application_managed_stream_context.
 * \return \ref image_data_error_codes, \ref roi_error_codes
 */
NppStatus nppiPSNRBatch_8u_C1R_Advanced_Ctx(const NppiImageDescriptor * pSrc1BatchList, 
                                            const NppiImageDescriptor * pSrc2BatchList, 
                                            int nBatchSize, NppiSize oMaxSizeROI, Npp32f * pPSNR, 
                                            NppiBufferDescriptor * pDeviceBufferList, NppStreamContext nppStreamCtx);

/**
 * 3 channel 8-bit unsigned PSNR for a batch of image pairs with per-image ROI
 *
 * \param pSrc1BatchList \ref source_batch_images_pointer device memory pointer to the list of device memory image descriptors.
 * \param pSrc2BatchList \ref source_batch_images_pointer device memory pointer to the list of device memory image descriptors.
 * \param nBatchSize Number of \ref NppiImageDescriptor, NppiBufferDescriptor, and new max number structures/values processed in this call (must be > 1).
 * \param oMaxSizeROI \ref roi_specification maximum ROI width and height of ALL images in the batch.
 * \param pPSNR Device memory pointer to output array of the computed PSNR for nBatchSize * sizeof(Npp32f) * 3 image pairs. 
 * \param pDeviceBufferList Device memory pointer to the list of NppiBufferDescriptor buffer descriptors specifying per image device
 * memory buffer pointers and size as returned by at least one nppiPSNRBatchGetBufferHostSize call.
 * \param nppStreamCtx \ref application_managed_stream_context.
 * \return \ref image_data_error_codes, \ref roi_error_codes
 */
NppStatus nppiPSNRBatch_8u_C3R_Advanced_Ctx(const NppiImageDescriptor * pSrc1BatchList, 
                                            const NppiImageDescriptor * pSrc2BatchList, 
                                            int nBatchSize, NppiSize oMaxSizeROI, Npp32f * pPSNR, 
                                            NppiBufferDescriptor * pDeviceBufferList, NppStreamContext nppStreamCtx);

/** @} PSNRBatchAdvanced */

/** @name WMSSSIMBatchAdvanced
 * @{
 */

/**
 * 1 channel 8-bit unsigned WMSSSIM for a batch of image pairs with per-image ROI
 * NOTE: It is the user's responsibility to make sure the dimensions of per-image ROIs are 16 pixels by 16 pixels and above. Any per-image ROI dimensions less than 16 pixels by 16 pixels will result in undefined behaviour.
 *
 * \param pSrc1BatchList \ref source_batch_images_pointer device memory pointer to the list of device memory image descriptors.
 * \param pSrc2BatchList \ref source_batch_images_pointer device memory pointer to the list of device memory image descriptors.
 * \param nBatchSize Number of \ref NppiImageDescriptor, NppiBufferDescriptor, and new max number structures/values processed in this call (must be > 1).
 * \param oMaxSizeROI \ref roi_specification maximum ROI width and height of ALL images in the batch.
 * \param pWMSSSIM Device memory pointer to output array of the computed WMSSSIM for nBatchSize * sizeof(Npp32f) * 1 image pairs. 
 * \param pDeviceBufferList Device memory pointer to the list of NppiBufferDescriptor buffer descriptors specifying per image device
 * memory buffer pointers and size as returned by at least one nppiWMSSSIMBatchGetBufferHostSize call.
 * \param nppStreamCtx \ref application_managed_stream_context.
 * \return \ref image_data_error_codes, \ref roi_error_codes
 */
NppStatus nppiWMSSSIMBatch_8u_C1R_Advanced_Ctx(const NppiImageDescriptor * pSrc1BatchList, 
                                               const NppiImageDescriptor * pSrc2BatchList, 
                                               int nBatchSize, NppiSize oMaxSizeROI, Npp32f * pWMSSSIM, 
                                               NppiBufferDescriptor * pDeviceBufferList, NppStreamContext nppStreamCtx);

/**
 * 3 channel 8-bit unsigned WMSSSIM for a batch of image pairs with per-image ROI
 * NOTE: It is the user's responsibility to make sure the dimensions of per-image ROIs are 16 pixels by 16 pixels and above. Any per-image ROI dimensions less than 16 pixels by 16 pixels will result in undefined behaviour.
 *
 * \param pSrc1BatchList \ref source_batch_images_pointer device memory pointer to the list of device memory image descriptors.
 * \param pSrc2BatchList \ref source_batch_images_pointer device memory pointer to the list of device memory image descriptors.
 * \param nBatchSize Number of \ref NppiImageDescriptor, NppiBufferDescriptor, and new max number structures/values processed in this call (must be > 1).
 * \param oMaxSizeROI \ref roi_specification maximum ROI width and height of ALL images in the batch.
 * \param pWMSSSIM Device memory pointer to output array of the computed WMSSSIM for nBatchSize * sizeof(Npp32f) * 3 image pairs. 
 * \param pDeviceBufferList Device memory pointer to the list of NppiBufferDescriptor buffer descriptors specifying per image device
 * memory buffer pointers and size as returned by at least one nppiWMSSSIMBatchGetBufferHostSize call.
 * \param nppStreamCtx \ref application_managed_stream_context.
 * \return \ref image_data_error_codes, \ref roi_error_codes
 */
NppStatus nppiWMSSSIMBatch_8u_C3R_Advanced_Ctx(const NppiImageDescriptor * pSrc1BatchList, 
                                               const NppiImageDescriptor * pSrc2BatchList, 
                                               int nBatchSize, NppiSize oMaxSizeROI, Npp32f * pWMSSSIM, 
                                               NppiBufferDescriptor * pDeviceBufferList, NppStreamContext nppStreamCtx);

/**
 * 4 channel 8-bit unsigned WMSSSIM for a batch of image pairs with per-image ROI
 * NOTE: It is the user's responsibility to make sure the dimensions of per-image ROIs are 16 pixels by 16 pixels and above. Any per-image ROI dimensions less than 16 pixels by 16 pixels will result in undefined behaviour.
 *
 * \param pSrc1BatchList \ref source_batch_images_pointer device memory pointer to the list of device memory image descriptors.
 * \param pSrc2BatchList \ref source_batch_images_pointer device memory pointer to the list of device memory image descriptors.
 * \param nBatchSize Number of \ref NppiImageDescriptor, NppiBufferDescriptor, and new max number structures/values processed in this call (must be > 1).
 * \param oMaxSizeROI \ref roi_specification maximum ROI width and height of ALL images in the batch.
 * \param pWMSSSIM Device memory pointer to output array of the computed WMSSSIM for nBatchSize * sizeof(Npp32f) * 4 image pairs. 
 * \param pDeviceBufferList Device memory pointer to the list of NppiBufferDescriptor buffer descriptors specifying per image device
 * memory buffer pointers and size as returned by at least one nppiWMSSSIMBatchGetBufferHostSize call.
 * \param nppStreamCtx \ref application_managed_stream_context.
 * \return \ref image_data_error_codes, \ref roi_error_codes
 */
NppStatus nppiWMSSSIMBatch_8u_C4R_Advanced_Ctx(const NppiImageDescriptor * pSrc1BatchList, 
                                               const NppiImageDescriptor * pSrc2BatchList, 
                                               int nBatchSize, NppiSize oMaxSizeROI, Npp32f * pWMSSSIM, 
                                               NppiBufferDescriptor * pDeviceBufferList, NppStreamContext nppStreamCtx);

/** @} WMSSSIMBatchAdvanced */

/** @} batch_image_quality_assessment_advanced */

/** @name Profiles
 * @{
 */

/**
 * One-channel 32-bit Circular radial profile based upon a specified center point.
 *
 * \param pSrc \ref source_image_pointer.
 * \param nSrcStep \ref source_image_line_step.
 * \param oSizeROI \ref roi_specification.
 * \param oCenter center of circle that radial profile is computed from.  Center does not have to be on the image.
 * \param pProfileData pointer to output NppiProfileData array containing count, mean, and std_dev as a function of radius.
 * \param nProfileSamples number of radii to be sampled: from 0 through nProfileSamples-1.
 * \param nppStreamCtx \ref application_managed_stream_context.
 * \return \ref image_data_error_codes, \ref roi_error_codes.
*/
NppStatus nppiCircularRadialProfile_32f_C1R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSizeROI, NppiPoint32f oCenter,
    NppiProfileData *pProfileData, Npp16u nProfileSamples, NppStreamContext nppStreamCtx);

/**
 * One-channel 32-bit Elliptical radial profile based upon a specified center point, angle, and astigmatism.
 *
 * \param pSrc \ref source_image_pointer.
 * \param nSrcStep \ref source_image_line_step.
 * \param oSizeROI \ref roi_specification.
 * \param oCenter center of circle that radial profile is computed from.  Center does not have to be on the image.
 * \param nAstigmatismRatio ratio of orientation axis to orthogonal axis.  (i.e. 1.5 means stretched 1.5x)
 * \param nOrientationRadians orientation of the astigmatism, clockwise in radians with 0.0 being vertical.
 * \param pProfileData pointer to output NppiProfileData array containing count, mean, and std_dev as a function of radius.
 * \param nProfileSamples number of radii to be sampled: from 0 through nProfileSamples-1.
 * \param nppStreamCtx \ref application_managed_stream_context.
 * \return \ref image_data_error_codes, \ref roi_error_codes.
*/
NppStatus nppiEllipticalRadialProfile_32f_C1R_Ctx(const Npp32f * pSrc, int nSrcStep, NppiSize oSizeROI, NppiPoint32f oCenter,
    Npp32f nAstigmatismRatio, Npp32f nOrientationRadians, NppiProfileData *pProfileData, Npp16u nProfileSamples, NppStreamContext nppStreamCtx);


/** @} Profiles */


/** @} image_statistics_functions */


#ifdef __cplusplus
} /* extern "C" */
#endif

#endif /* NV_NPPI_STATISTICS_FUNCTIONS_H */

