﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/clouddirectory/CloudDirectoryRequest.h>
#include <aws/clouddirectory/CloudDirectory_EXPORTS.h>
#include <aws/clouddirectory/model/ConsistencyLevel.h>
#include <aws/clouddirectory/model/ObjectAttributeRange.h>
#include <aws/clouddirectory/model/ObjectReference.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>

#include <utility>

namespace Aws {
namespace CloudDirectory {
namespace Model {

/**
 */
class ListIndexRequest : public CloudDirectoryRequest {
 public:
  AWS_CLOUDDIRECTORY_API ListIndexRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListIndex"; }

  AWS_CLOUDDIRECTORY_API Aws::String SerializePayload() const override;

  AWS_CLOUDDIRECTORY_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The ARN of the directory that the index exists in.</p>
   */
  inline const Aws::String& GetDirectoryArn() const { return m_directoryArn; }
  inline bool DirectoryArnHasBeenSet() const { return m_directoryArnHasBeenSet; }
  template <typename DirectoryArnT = Aws::String>
  void SetDirectoryArn(DirectoryArnT&& value) {
    m_directoryArnHasBeenSet = true;
    m_directoryArn = std::forward<DirectoryArnT>(value);
  }
  template <typename DirectoryArnT = Aws::String>
  ListIndexRequest& WithDirectoryArn(DirectoryArnT&& value) {
    SetDirectoryArn(std::forward<DirectoryArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies the ranges of indexed values that you want to query.</p>
   */
  inline const Aws::Vector<ObjectAttributeRange>& GetRangesOnIndexedValues() const { return m_rangesOnIndexedValues; }
  inline bool RangesOnIndexedValuesHasBeenSet() const { return m_rangesOnIndexedValuesHasBeenSet; }
  template <typename RangesOnIndexedValuesT = Aws::Vector<ObjectAttributeRange>>
  void SetRangesOnIndexedValues(RangesOnIndexedValuesT&& value) {
    m_rangesOnIndexedValuesHasBeenSet = true;
    m_rangesOnIndexedValues = std::forward<RangesOnIndexedValuesT>(value);
  }
  template <typename RangesOnIndexedValuesT = Aws::Vector<ObjectAttributeRange>>
  ListIndexRequest& WithRangesOnIndexedValues(RangesOnIndexedValuesT&& value) {
    SetRangesOnIndexedValues(std::forward<RangesOnIndexedValuesT>(value));
    return *this;
  }
  template <typename RangesOnIndexedValuesT = ObjectAttributeRange>
  ListIndexRequest& AddRangesOnIndexedValues(RangesOnIndexedValuesT&& value) {
    m_rangesOnIndexedValuesHasBeenSet = true;
    m_rangesOnIndexedValues.emplace_back(std::forward<RangesOnIndexedValuesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The reference to the index to list.</p>
   */
  inline const ObjectReference& GetIndexReference() const { return m_indexReference; }
  inline bool IndexReferenceHasBeenSet() const { return m_indexReferenceHasBeenSet; }
  template <typename IndexReferenceT = ObjectReference>
  void SetIndexReference(IndexReferenceT&& value) {
    m_indexReferenceHasBeenSet = true;
    m_indexReference = std::forward<IndexReferenceT>(value);
  }
  template <typename IndexReferenceT = ObjectReference>
  ListIndexRequest& WithIndexReference(IndexReferenceT&& value) {
    SetIndexReference(std::forward<IndexReferenceT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of objects in a single page to retrieve from the index
   * during a request. For more information, see <a
   * href="http://docs.aws.amazon.com/clouddirectory/latest/developerguide/limits.html">Amazon
   * Cloud Directory Limits</a>.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListIndexRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The pagination token.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListIndexRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The consistency level to execute the request at.</p>
   */
  inline ConsistencyLevel GetConsistencyLevel() const { return m_consistencyLevel; }
  inline bool ConsistencyLevelHasBeenSet() const { return m_consistencyLevelHasBeenSet; }
  inline void SetConsistencyLevel(ConsistencyLevel value) {
    m_consistencyLevelHasBeenSet = true;
    m_consistencyLevel = value;
  }
  inline ListIndexRequest& WithConsistencyLevel(ConsistencyLevel value) {
    SetConsistencyLevel(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_directoryArn;

  Aws::Vector<ObjectAttributeRange> m_rangesOnIndexedValues;

  ObjectReference m_indexReference;

  int m_maxResults{0};

  Aws::String m_nextToken;

  ConsistencyLevel m_consistencyLevel{ConsistencyLevel::NOT_SET};
  bool m_directoryArnHasBeenSet = false;
  bool m_rangesOnIndexedValuesHasBeenSet = false;
  bool m_indexReferenceHasBeenSet = false;
  bool m_maxResultsHasBeenSet = false;
  bool m_nextTokenHasBeenSet = false;
  bool m_consistencyLevelHasBeenSet = false;
};

}  // namespace Model
}  // namespace CloudDirectory
}  // namespace Aws
