#!/bin/sh

# Copyright (C) 2024-2025 Internet Systems Consortium, Inc. ("ISC")
#
# This Source Code Form is subject to the terms of the Mozilla Public
# License, v. 2.0. If a copy of the MPL was not distributed with this
# file, You can obtain one at http://mozilla.org/MPL/2.0/.

# Exit with error if commands exit with non-zero and if undefined variables are
# used.
set -eu

# shellcheck disable=SC2034
# SC2034: ... appears unused. Verify use (or export if used externally).
prefix="/usr"

# Include utilities based on location of this script. Check for sources first,
# so that the unexpected situations with weird paths fall on the default
# case of installed.
script_path=$(cd "$(dirname "${0}")" && pwd)
if test "${script_path}" = "/build/kea/src/build/src/share/database/scripts/mysql"; then
    # shellcheck source=./src/bin/admin/admin-utils.sh.in
    . "/build/kea/src/build/src/bin/admin/admin-utils.sh"
else
    # shellcheck source=./src/bin/admin/admin-utils.sh.in
    . "${prefix}/share/kea/scripts/admin-utils.sh"
fi

# Check only major version to allow for intermediary backported schema changes.
version=$(mysql_version "${@}" | cut -d '.' -f 1)
if test "${version}" != '23'; then
    printf 'This script upgrades 23.* to 24.0. '
    printf 'Reported version is %s. Skipping upgrade.\n' "${version}"
    exit 0
fi

# Get the schema name from database argument. We need this to
# query information_schema for the right database.
for arg in "${@}"
do
    if ! printf '%s' "${arg}" | grep -Eq -- '^--'
    then
        schema="$arg"
        break
    fi
done

# Make sure we have the schema.
if [ -z "$schema" ]
then
    printf "Could not find database schema name in cmd line args: %s\n" "${*}"
    exit 255
fi

mysql "$@" <<EOF

-- This line starts the schema upgrade to version 24.0.

SET @disable_audit = 1;

DELETE FROM dhcp4_global_parameter WHERE name='control-socket.socket-name';
DELETE FROM dhcp4_global_parameter WHERE name='control-socket.socket-type';

DELETE FROM dhcp6_global_parameter WHERE name='control-socket.socket-name';
DELETE FROM dhcp6_global_parameter WHERE name='control-socket.socket-type';

SET @disable_audit = 0;

-- Create a function to conditionally migrate option_def data type
-- values.  If they are updating from 2.6.1 this has been done already
-- and we don't want to do it twice.
DROP PROCEDURE IF EXISTS updateOptionDataDef;
DELIMITER $$
CREATE PROCEDURE updateOptionDataDef()
BEGIN
    DECLARE skipper TINYINT;

    SELECT COUNT(table_name) FROM information_schema.tables
        WHERE table_schema LIKE database() AND table_name = 'option_def_data_type' INTO skipper;

    IF skipper = 0 THEN
        -- First we migrate existing OPT_RECORD_TYPE values
        -- If they're coming from pre-2.6.0 record type is 17, from
        -- 2.6.0 or 2.7.0 it is 18. No viable to way to know where we
        -- started.
        SET @disable_audit = 1;
        UPDATE dhcp4_option_def SET type = 254 WHERE record_types IS NOT NULL;
        UPDATE dhcp6_option_def SET type = 254 WHERE record_types IS NOT NULL;

        -- Create the table that enumerates option definition data types.
        CREATE TABLE option_def_data_type (
            id TINYINT UNSIGNED NOT NULL PRIMARY KEY,
            name VARCHAR(32) NOT NULL
        ) ENGINE = InnoDB;

        -- Now insert supported types.
        -- We skip (9, 'any-address') as it is not externally supported.
        INSERT INTO option_def_data_type VALUES
            (0, 'empty'),
            (1, 'binary'),
            (2, 'boolean'),
            (3, 'int8"'),
            (4, 'int16'),
            (5, 'int32'),
            (6, 'uint8'),
            (7, 'uint16'),
            (8, 'uint32'),
            (10, 'ipv4-address'),
            (11, 'ipv6-address'),
            (12, 'ipv6-prefix'),
            (13, 'psid'),
            (14, 'string'),
            (15, 'tuple'),
            (16, 'fqdn'),
            (17, 'internal'),
            (254, 'record');

        --  Add foreign key constraints to enforce only valid types.
        ALTER TABLE dhcp4_option_def
          ADD CONSTRAINT fk_option_def_data_type4 FOREIGN KEY (type) REFERENCES option_def_data_type(id);

        ALTER TABLE dhcp6_option_def
          ADD CONSTRAINT fk_option_def_data_type6 FOREIGN KEY (type) REFERENCES option_def_data_type(id);
        SET @disable_audit = 0;
    END IF;
END $$
DELIMITER ;

CALL updateOptionDataDef();

-- Get rid of the now obsolete function.
DROP PROCEDURE IF EXISTS updateOptionDataDef;

-- Update the schema version number.
UPDATE schema_version
    SET version = '24', minor = '0';

-- This line concludes the schema upgrade to version 24.0.

EOF
