﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/iam/IAMRequest.h>
#include <aws/iam/IAM_EXPORTS.h>

#include <utility>

namespace Aws {
namespace IAM {
namespace Model {

/**
 */
class UntagInstanceProfileRequest : public IAMRequest {
 public:
  AWS_IAM_API UntagInstanceProfileRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UntagInstanceProfile"; }

  AWS_IAM_API Aws::String SerializePayload() const override;

 protected:
  AWS_IAM_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The name of the IAM instance profile from which you want to remove tags.</p>
   * <p>This parameter allows (through its <a
   * href="http://wikipedia.org/wiki/regex">regex pattern</a>) a string of characters
   * consisting of upper and lowercase alphanumeric characters with no spaces. You
   * can also include any of the following characters: _+=,.@-</p>
   */
  inline const Aws::String& GetInstanceProfileName() const { return m_instanceProfileName; }
  inline bool InstanceProfileNameHasBeenSet() const { return m_instanceProfileNameHasBeenSet; }
  template <typename InstanceProfileNameT = Aws::String>
  void SetInstanceProfileName(InstanceProfileNameT&& value) {
    m_instanceProfileNameHasBeenSet = true;
    m_instanceProfileName = std::forward<InstanceProfileNameT>(value);
  }
  template <typename InstanceProfileNameT = Aws::String>
  UntagInstanceProfileRequest& WithInstanceProfileName(InstanceProfileNameT&& value) {
    SetInstanceProfileName(std::forward<InstanceProfileNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of key names as a simple array of strings. The tags with matching keys
   * are removed from the specified instance profile.</p>
   */
  inline const Aws::Vector<Aws::String>& GetTagKeys() const { return m_tagKeys; }
  inline bool TagKeysHasBeenSet() const { return m_tagKeysHasBeenSet; }
  template <typename TagKeysT = Aws::Vector<Aws::String>>
  void SetTagKeys(TagKeysT&& value) {
    m_tagKeysHasBeenSet = true;
    m_tagKeys = std::forward<TagKeysT>(value);
  }
  template <typename TagKeysT = Aws::Vector<Aws::String>>
  UntagInstanceProfileRequest& WithTagKeys(TagKeysT&& value) {
    SetTagKeys(std::forward<TagKeysT>(value));
    return *this;
  }
  template <typename TagKeysT = Aws::String>
  UntagInstanceProfileRequest& AddTagKeys(TagKeysT&& value) {
    m_tagKeysHasBeenSet = true;
    m_tagKeys.emplace_back(std::forward<TagKeysT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_instanceProfileName;

  Aws::Vector<Aws::String> m_tagKeys;
  bool m_instanceProfileNameHasBeenSet = false;
  bool m_tagKeysHasBeenSet = false;
};

}  // namespace Model
}  // namespace IAM
}  // namespace Aws
