﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/neptune-graph/NeptuneGraphRequest.h>
#include <aws/neptune-graph/NeptuneGraph_EXPORTS.h>
#include <aws/neptune-graph/model/ExportFilter.h>
#include <aws/neptune-graph/model/ExportFormat.h>
#include <aws/neptune-graph/model/ParquetType.h>

#include <utility>

namespace Aws {
namespace NeptuneGraph {
namespace Model {

/**
 */
class StartExportTaskRequest : public NeptuneGraphRequest {
 public:
  AWS_NEPTUNEGRAPH_API StartExportTaskRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "StartExportTask"; }

  AWS_NEPTUNEGRAPH_API Aws::String SerializePayload() const override;

  /**
   * Helper function to collect parameters (configurable and static hardcoded) required for endpoint computation.
   */
  AWS_NEPTUNEGRAPH_API EndpointParameters GetEndpointContextParams() const override;

  ///@{
  /**
   * <p>The source graph identifier of the export task.</p>
   */
  inline const Aws::String& GetGraphIdentifier() const { return m_graphIdentifier; }
  inline bool GraphIdentifierHasBeenSet() const { return m_graphIdentifierHasBeenSet; }
  template <typename GraphIdentifierT = Aws::String>
  void SetGraphIdentifier(GraphIdentifierT&& value) {
    m_graphIdentifierHasBeenSet = true;
    m_graphIdentifier = std::forward<GraphIdentifierT>(value);
  }
  template <typename GraphIdentifierT = Aws::String>
  StartExportTaskRequest& WithGraphIdentifier(GraphIdentifierT&& value) {
    SetGraphIdentifier(std::forward<GraphIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ARN of the IAM role that will allow data to be exported to the
   * destination.</p>
   */
  inline const Aws::String& GetRoleArn() const { return m_roleArn; }
  inline bool RoleArnHasBeenSet() const { return m_roleArnHasBeenSet; }
  template <typename RoleArnT = Aws::String>
  void SetRoleArn(RoleArnT&& value) {
    m_roleArnHasBeenSet = true;
    m_roleArn = std::forward<RoleArnT>(value);
  }
  template <typename RoleArnT = Aws::String>
  StartExportTaskRequest& WithRoleArn(RoleArnT&& value) {
    SetRoleArn(std::forward<RoleArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The format of the export task.</p>
   */
  inline ExportFormat GetFormat() const { return m_format; }
  inline bool FormatHasBeenSet() const { return m_formatHasBeenSet; }
  inline void SetFormat(ExportFormat value) {
    m_formatHasBeenSet = true;
    m_format = value;
  }
  inline StartExportTaskRequest& WithFormat(ExportFormat value) {
    SetFormat(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon S3 URI where data will be exported to.</p>
   */
  inline const Aws::String& GetDestination() const { return m_destination; }
  inline bool DestinationHasBeenSet() const { return m_destinationHasBeenSet; }
  template <typename DestinationT = Aws::String>
  void SetDestination(DestinationT&& value) {
    m_destinationHasBeenSet = true;
    m_destination = std::forward<DestinationT>(value);
  }
  template <typename DestinationT = Aws::String>
  StartExportTaskRequest& WithDestination(DestinationT&& value) {
    SetDestination(std::forward<DestinationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The KMS key identifier of the export task.</p>
   */
  inline const Aws::String& GetKmsKeyIdentifier() const { return m_kmsKeyIdentifier; }
  inline bool KmsKeyIdentifierHasBeenSet() const { return m_kmsKeyIdentifierHasBeenSet; }
  template <typename KmsKeyIdentifierT = Aws::String>
  void SetKmsKeyIdentifier(KmsKeyIdentifierT&& value) {
    m_kmsKeyIdentifierHasBeenSet = true;
    m_kmsKeyIdentifier = std::forward<KmsKeyIdentifierT>(value);
  }
  template <typename KmsKeyIdentifierT = Aws::String>
  StartExportTaskRequest& WithKmsKeyIdentifier(KmsKeyIdentifierT&& value) {
    SetKmsKeyIdentifier(std::forward<KmsKeyIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The parquet type of the export task.</p>
   */
  inline ParquetType GetParquetType() const { return m_parquetType; }
  inline bool ParquetTypeHasBeenSet() const { return m_parquetTypeHasBeenSet; }
  inline void SetParquetType(ParquetType value) {
    m_parquetTypeHasBeenSet = true;
    m_parquetType = value;
  }
  inline StartExportTaskRequest& WithParquetType(ParquetType value) {
    SetParquetType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The export filter of the export task.</p>
   */
  inline const ExportFilter& GetExportFilter() const { return m_exportFilter; }
  inline bool ExportFilterHasBeenSet() const { return m_exportFilterHasBeenSet; }
  template <typename ExportFilterT = ExportFilter>
  void SetExportFilter(ExportFilterT&& value) {
    m_exportFilterHasBeenSet = true;
    m_exportFilter = std::forward<ExportFilterT>(value);
  }
  template <typename ExportFilterT = ExportFilter>
  StartExportTaskRequest& WithExportFilter(ExportFilterT&& value) {
    SetExportFilter(std::forward<ExportFilterT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Tags to be applied to the export task.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  StartExportTaskRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  StartExportTaskRequest& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_graphIdentifier;

  Aws::String m_roleArn;

  ExportFormat m_format{ExportFormat::NOT_SET};

  Aws::String m_destination;

  Aws::String m_kmsKeyIdentifier;

  ParquetType m_parquetType{ParquetType::NOT_SET};

  ExportFilter m_exportFilter;

  Aws::Map<Aws::String, Aws::String> m_tags;
  bool m_graphIdentifierHasBeenSet = false;
  bool m_roleArnHasBeenSet = false;
  bool m_formatHasBeenSet = false;
  bool m_destinationHasBeenSet = false;
  bool m_kmsKeyIdentifierHasBeenSet = false;
  bool m_parquetTypeHasBeenSet = false;
  bool m_exportFilterHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace NeptuneGraph
}  // namespace Aws
