﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/appsync/AppSyncRequest.h>
#include <aws/appsync/AppSync_EXPORTS.h>
#include <aws/appsync/model/TypeDefinitionFormat.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace AppSync {
namespace Model {

/**
 */
class CreateTypeRequest : public AppSyncRequest {
 public:
  AWS_APPSYNC_API CreateTypeRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateType"; }

  AWS_APPSYNC_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The API ID.</p>
   */
  inline const Aws::String& GetApiId() const { return m_apiId; }
  inline bool ApiIdHasBeenSet() const { return m_apiIdHasBeenSet; }
  template <typename ApiIdT = Aws::String>
  void SetApiId(ApiIdT&& value) {
    m_apiIdHasBeenSet = true;
    m_apiId = std::forward<ApiIdT>(value);
  }
  template <typename ApiIdT = Aws::String>
  CreateTypeRequest& WithApiId(ApiIdT&& value) {
    SetApiId(std::forward<ApiIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The type definition, in GraphQL Schema Definition Language (SDL) format.</p>
   * <p>For more information, see the <a
   * href="http://graphql.org/learn/schema/">GraphQL SDL documentation</a>.</p>
   */
  inline const Aws::String& GetDefinition() const { return m_definition; }
  inline bool DefinitionHasBeenSet() const { return m_definitionHasBeenSet; }
  template <typename DefinitionT = Aws::String>
  void SetDefinition(DefinitionT&& value) {
    m_definitionHasBeenSet = true;
    m_definition = std::forward<DefinitionT>(value);
  }
  template <typename DefinitionT = Aws::String>
  CreateTypeRequest& WithDefinition(DefinitionT&& value) {
    SetDefinition(std::forward<DefinitionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The type format: SDL or JSON.</p>
   */
  inline TypeDefinitionFormat GetFormat() const { return m_format; }
  inline bool FormatHasBeenSet() const { return m_formatHasBeenSet; }
  inline void SetFormat(TypeDefinitionFormat value) {
    m_formatHasBeenSet = true;
    m_format = value;
  }
  inline CreateTypeRequest& WithFormat(TypeDefinitionFormat value) {
    SetFormat(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_apiId;

  Aws::String m_definition;

  TypeDefinitionFormat m_format{TypeDefinitionFormat::NOT_SET};
  bool m_apiIdHasBeenSet = false;
  bool m_definitionHasBeenSet = false;
  bool m_formatHasBeenSet = false;
};

}  // namespace Model
}  // namespace AppSync
}  // namespace Aws
