﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/appmesh/AppMeshRequest.h>
#include <aws/appmesh/AppMesh_EXPORTS.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace AppMesh {
namespace Model {

/**
 * <zonbook></zonbook><xhtml></xhtml><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/appmesh-2019-01-25/ListVirtualNodesInput">AWS
 * API Reference</a></p>
 */
class ListVirtualNodesRequest : public AppMeshRequest {
 public:
  AWS_APPMESH_API ListVirtualNodesRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListVirtualNodes"; }

  AWS_APPMESH_API Aws::String SerializePayload() const override;

  AWS_APPMESH_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  ///@{
  /**
   * <p>The maximum number of results returned by <code>ListVirtualNodes</code> in
   * paginated output. When you use this parameter, <code>ListVirtualNodes</code>
   * returns only <code>limit</code> results in a single page along with a
   * <code>nextToken</code> response element. You can see the remaining results of
   * the initial request by sending another <code>ListVirtualNodes</code> request
   * with the returned <code>nextToken</code> value. This value can be between 1 and
   * 100. If you don't use this parameter, <code>ListVirtualNodes</code> returns up
   * to 100 results and a <code>nextToken</code> value if applicable.</p>
   */
  inline int GetLimit() const { return m_limit; }
  inline bool LimitHasBeenSet() const { return m_limitHasBeenSet; }
  inline void SetLimit(int value) {
    m_limitHasBeenSet = true;
    m_limit = value;
  }
  inline ListVirtualNodesRequest& WithLimit(int value) {
    SetLimit(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the service mesh to list virtual nodes in.</p>
   */
  inline const Aws::String& GetMeshName() const { return m_meshName; }
  inline bool MeshNameHasBeenSet() const { return m_meshNameHasBeenSet; }
  template <typename MeshNameT = Aws::String>
  void SetMeshName(MeshNameT&& value) {
    m_meshNameHasBeenSet = true;
    m_meshName = std::forward<MeshNameT>(value);
  }
  template <typename MeshNameT = Aws::String>
  ListVirtualNodesRequest& WithMeshName(MeshNameT&& value) {
    SetMeshName(std::forward<MeshNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Web Services IAM account ID of the service mesh owner. If the
   * account ID is not your own, then it's the ID of the account that shared the mesh
   * with your account. For more information about mesh sharing, see <a
   * href="https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html">Working
   * with shared meshes</a>.</p>
   */
  inline const Aws::String& GetMeshOwner() const { return m_meshOwner; }
  inline bool MeshOwnerHasBeenSet() const { return m_meshOwnerHasBeenSet; }
  template <typename MeshOwnerT = Aws::String>
  void SetMeshOwner(MeshOwnerT&& value) {
    m_meshOwnerHasBeenSet = true;
    m_meshOwner = std::forward<MeshOwnerT>(value);
  }
  template <typename MeshOwnerT = Aws::String>
  ListVirtualNodesRequest& WithMeshOwner(MeshOwnerT&& value) {
    SetMeshOwner(std::forward<MeshOwnerT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The <code>nextToken</code> value returned from a previous paginated
   * <code>ListVirtualNodes</code> request where <code>limit</code> was used and the
   * results exceeded the value of that parameter. Pagination continues from the end
   * of the previous results that returned the <code>nextToken</code> value.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListVirtualNodesRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}
 private:
  int m_limit{0};

  Aws::String m_meshName;

  Aws::String m_meshOwner;

  Aws::String m_nextToken;
  bool m_limitHasBeenSet = false;
  bool m_meshNameHasBeenSet = false;
  bool m_meshOwnerHasBeenSet = false;
  bool m_nextTokenHasBeenSet = false;
};

}  // namespace Model
}  // namespace AppMesh
}  // namespace Aws
