﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/appflow/AppflowRequest.h>
#include <aws/appflow/Appflow_EXPORTS.h>
#include <aws/appflow/model/ConnectionMode.h>
#include <aws/appflow/model/ConnectorProfileConfig.h>
#include <aws/appflow/model/ConnectorType.h>
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace Appflow {
namespace Model {

/**
 */
class CreateConnectorProfileRequest : public AppflowRequest {
 public:
  AWS_APPFLOW_API CreateConnectorProfileRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateConnectorProfile"; }

  AWS_APPFLOW_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p> The name of the connector profile. The name is unique for each
   * <code>ConnectorProfile</code> in your Amazon Web Services account. </p>
   */
  inline const Aws::String& GetConnectorProfileName() const { return m_connectorProfileName; }
  inline bool ConnectorProfileNameHasBeenSet() const { return m_connectorProfileNameHasBeenSet; }
  template <typename ConnectorProfileNameT = Aws::String>
  void SetConnectorProfileName(ConnectorProfileNameT&& value) {
    m_connectorProfileNameHasBeenSet = true;
    m_connectorProfileName = std::forward<ConnectorProfileNameT>(value);
  }
  template <typename ConnectorProfileNameT = Aws::String>
  CreateConnectorProfileRequest& WithConnectorProfileName(ConnectorProfileNameT&& value) {
    SetConnectorProfileName(std::forward<ConnectorProfileNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The ARN (Amazon Resource Name) of the Key Management Service (KMS) key you
   * provide for encryption. This is required if you do not want to use the Amazon
   * AppFlow-managed KMS key. If you don't provide anything here, Amazon AppFlow uses
   * the Amazon AppFlow-managed KMS key. </p>
   */
  inline const Aws::String& GetKmsArn() const { return m_kmsArn; }
  inline bool KmsArnHasBeenSet() const { return m_kmsArnHasBeenSet; }
  template <typename KmsArnT = Aws::String>
  void SetKmsArn(KmsArnT&& value) {
    m_kmsArnHasBeenSet = true;
    m_kmsArn = std::forward<KmsArnT>(value);
  }
  template <typename KmsArnT = Aws::String>
  CreateConnectorProfileRequest& WithKmsArn(KmsArnT&& value) {
    SetKmsArn(std::forward<KmsArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The type of connector, such as Salesforce, Amplitude, and so on. </p>
   */
  inline ConnectorType GetConnectorType() const { return m_connectorType; }
  inline bool ConnectorTypeHasBeenSet() const { return m_connectorTypeHasBeenSet; }
  inline void SetConnectorType(ConnectorType value) {
    m_connectorTypeHasBeenSet = true;
    m_connectorType = value;
  }
  inline CreateConnectorProfileRequest& WithConnectorType(ConnectorType value) {
    SetConnectorType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The label of the connector. The label is unique for each
   * <code>ConnectorRegistration</code> in your Amazon Web Services account. Only
   * needed if calling for CUSTOMCONNECTOR connector type/.</p>
   */
  inline const Aws::String& GetConnectorLabel() const { return m_connectorLabel; }
  inline bool ConnectorLabelHasBeenSet() const { return m_connectorLabelHasBeenSet; }
  template <typename ConnectorLabelT = Aws::String>
  void SetConnectorLabel(ConnectorLabelT&& value) {
    m_connectorLabelHasBeenSet = true;
    m_connectorLabel = std::forward<ConnectorLabelT>(value);
  }
  template <typename ConnectorLabelT = Aws::String>
  CreateConnectorProfileRequest& WithConnectorLabel(ConnectorLabelT&& value) {
    SetConnectorLabel(std::forward<ConnectorLabelT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> Indicates the connection mode and specifies whether it is public or private.
   * Private flows use Amazon Web Services PrivateLink to route data over Amazon Web
   * Services infrastructure without exposing it to the public internet. </p>
   */
  inline ConnectionMode GetConnectionMode() const { return m_connectionMode; }
  inline bool ConnectionModeHasBeenSet() const { return m_connectionModeHasBeenSet; }
  inline void SetConnectionMode(ConnectionMode value) {
    m_connectionModeHasBeenSet = true;
    m_connectionMode = value;
  }
  inline CreateConnectorProfileRequest& WithConnectionMode(ConnectionMode value) {
    SetConnectionMode(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> Defines the connector-specific configuration and credentials. </p>
   */
  inline const ConnectorProfileConfig& GetConnectorProfileConfig() const { return m_connectorProfileConfig; }
  inline bool ConnectorProfileConfigHasBeenSet() const { return m_connectorProfileConfigHasBeenSet; }
  template <typename ConnectorProfileConfigT = ConnectorProfileConfig>
  void SetConnectorProfileConfig(ConnectorProfileConfigT&& value) {
    m_connectorProfileConfigHasBeenSet = true;
    m_connectorProfileConfig = std::forward<ConnectorProfileConfigT>(value);
  }
  template <typename ConnectorProfileConfigT = ConnectorProfileConfig>
  CreateConnectorProfileRequest& WithConnectorProfileConfig(ConnectorProfileConfigT&& value) {
    SetConnectorProfileConfig(std::forward<ConnectorProfileConfigT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The <code>clientToken</code> parameter is an idempotency token. It ensures
   * that your <code>CreateConnectorProfile</code> request completes only once. You
   * choose the value to pass. For example, if you don't receive a response from your
   * request, you can safely retry the request with the same <code>clientToken</code>
   * parameter value.</p> <p>If you omit a <code>clientToken</code> value, the Amazon
   * Web Services SDK that you are using inserts a value for you. This way, the SDK
   * can safely retry requests multiple times after a network error. You must provide
   * your own value for other use cases.</p> <p>If you specify input parameters that
   * differ from your first request, an error occurs. If you use a different value
   * for <code>clientToken</code>, Amazon AppFlow considers it a new call to
   * <code>CreateConnectorProfile</code>. The token is active for 8 hours.</p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  CreateConnectorProfileRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_connectorProfileName;

  Aws::String m_kmsArn;

  ConnectorType m_connectorType{ConnectorType::NOT_SET};

  Aws::String m_connectorLabel;

  ConnectionMode m_connectionMode{ConnectionMode::NOT_SET};

  ConnectorProfileConfig m_connectorProfileConfig;

  Aws::String m_clientToken{Aws::Utils::UUID::PseudoRandomUUID()};
  bool m_connectorProfileNameHasBeenSet = false;
  bool m_kmsArnHasBeenSet = false;
  bool m_connectorTypeHasBeenSet = false;
  bool m_connectorLabelHasBeenSet = false;
  bool m_connectionModeHasBeenSet = false;
  bool m_connectorProfileConfigHasBeenSet = false;
  bool m_clientTokenHasBeenSet = true;
};

}  // namespace Model
}  // namespace Appflow
}  // namespace Aws
