# Copyright (C) 2023 Intel Corporation
# SPDX-License-Identifier: MIT

from dataclasses import dataclass
from pathlib import Path
from typing import Optional
import warnings

import pandas as pd
import polars as pl

from cli.writers.base import OutputWriter
from mpp import ViewAttributes

warnings.filterwarnings("ignore", message=".*Polars found a filename. Ensure you pass a path to the file instead"
                                          " of a python file object when possible for best performance..*")

class CSVWriter(OutputWriter):
    """
    Write Pandas DataFrame content to CSV
    """

    def __init__(self, csv_directory_path: Path):
        """
        Initialize an output CSV writer

        :param csv_directory_path: location in which to create CSV files
        """
        self.__csv_files = {}
        self.__csv_directory = csv_directory_path

    def write(self,
              title: str,
              content: pd.DataFrame,
              attributes: Optional[ViewAttributes] = None) -> None:
        """
        Write/append the content of a Pandas DataFrame to a CSV file.
        The file is created if it does not exist.

        :param title: CSV file name (basename, without path)
        :param content: DataFrame to write
        :param attributes: view attributes (optional)
        """
        csv_info, mode, print_header = self._setup_to_csv(title)
        self._to_csv(content, csv_info, mode, print_header)
        csv_info.start_row += len(content)

    def close(self):
        """
        Finalize and close the CSV file. Subsequent operations on this CSV Writer object are not permitted.
        """
        # Function defined to override the generic documentation of the base class.
        # Pandas will automatically close and finalize the generated CSV files during the `write` operation.
        pass

    def _setup_to_csv(self, title):
        csv_info = self._get_csv_file(title)
        mode = 'w' if csv_info.start_row == 0 else 'a'
        print_header = True if csv_info.start_row == 0 else False
        return csv_info, mode, print_header

    def _to_csv(self, content, csv_info, mode, print_header):
        # Pandas "to_csv" function works significantly faster with larger chunk size (default is None).
        # chunksize=200 seems to yield a good speed-up with a reasonable memory consumption.
        content.to_csv(csv_info.path, header=print_header, index=False, mode=mode, date_format='%m/%d/%Y %H:%M:%S.%f',
                       chunksize=200)

    def _get_csv_file(self, name: str) -> '_CSVInfo':
        """
        Allocate a new CSV file name for the view, or return an existing file if already allocated

        :param name: must be in the format "{view name}_{view type}",
                     e.g. "socket_summary", "system_details", etc.
        """
        if name not in self.__csv_files:
            csv_path = self.__csv_directory / Path(f'{name}.csv')
            self.__csv_files[name] = self._CSVInfo(name, csv_path)
        return self.__csv_files[name]

    @dataclass
    class _CSVInfo:
        name: str
        path: Path
        start_row: int = 0


class PolarsCSVWriter(CSVWriter):

    def _to_csv(self, content, csv_info, mode, print_header):
        content = pl.from_pandas(content)
        with open(csv_info.path, mode, encoding='utf-8') as f:
            content.write_csv(f, include_header=print_header, date_format='%m/%d/%Y %H:%M:%S.%f')
        return content


class CSVWriterFactory:

    @classmethod
    def create(cls, is_polars: bool, csv_directory_path: Path) -> CSVWriter:
        """
        Create a new CSVWriter object

        @param csv_directory_path: location in which to create CSV files
        @param is_parallel: whether the input file is large enough for parallel processing and should use Polars
        instead of Pandas for CSV writing
        """
        # TODO: will clean up with API refactor, combine with Parallel/Serial DataProcessorFactory?
        if is_polars:
            return PolarsCSVWriter(csv_directory_path)
        return CSVWriter(csv_directory_path)
